package ai.passio.passiosdk.passiofood.recognition

import ai.passio.passiosdk.core.config.SDKFileType
import ai.passio.passiosdk.core.tflite.TFLiteRecognizer
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.ClassificationCandidate
import ai.passio.passiosdk.passiofood.ObjectDetectionCandidate
import ai.passio.passiosdk.passiofood.config.passio_dataset
import ai.passio.passiosdk.passiofood.file.PassioFoodFileManager
import ai.passio.passiosdk.passiofood.metadata.MetadataManager
import ai.passio.passiosdk.passiofood.mlkit.BarcodeDetectorFactory
import ai.passio.passiosdk.passiofood.mlkit.PassioBarcodeScanner
import ai.passio.passiosdk.passiofood.mlkit.TextRecognitionFactory
import android.content.Context
import android.util.Log
import com.google.mlkit.vision.text.TextRecognizer
import java.io.FileNotFoundException

internal abstract class PassioModelHolder(
    protected val labelManager: MetadataManager,
    protected val fileManager: PassioFoodFileManager,
) {
    var objectDetector: TFLiteRecognizer<List<ObjectDetectionCandidate>>? = null
        protected set
    var hnnDetector: TFLiteRecognizer<List<ClassificationCandidate>>? = null
        protected set
    var barcodeDetector: PassioBarcodeScanner? = null
        protected set
    var ocrDetector: TextRecognizer? = null
        protected set

    open fun initializeModels(
        context: Context,
        version: Int
    ): List<String> {
        val erroredModels = mutableListOf<String>()
        fileManager.getAvailableFileTypes().forEach { fileType ->
            if (fileType == passio_dataset) {
                return@forEach
            }

            val filename = "${fileType.name}.$version.${getExtension()}"
            val success = initializeModel(context, fileType, version)
            if (success) {
                PassioLog.i(this::class.java.simpleName, "$filename configured")
                return@forEach
            }

            if (fileType.alternative() == null) {
                erroredModels.add(filename)
                return@forEach
            }

            val alternativeSuccess = initializeModel(
                context,
                fileType.alternative()!!,
                version
            )
            if (alternativeSuccess) {
                PassioLog.i(
                    this::class.java.simpleName,
                    "$filename configured using alternative ${fileType.alternative()?.name}"
                )
                return@forEach
            }

            erroredModels.add(filename)
        }

        barcodeDetector = BarcodeDetectorFactory.create()
        ocrDetector = TextRecognitionFactory.getTextRecognizer()

        return erroredModels
    }

    protected fun <T> assetCatch(name: String, initBlock: () -> T): T? {
        return try {
            initBlock()
        } catch (e: FileNotFoundException) {
            Log.e(this::class.java.simpleName, "Asset $name not found!")
            null
        }
    }

    fun getObjectDetectionInputSize(): Int {
        if (objectDetector == null) {
            throw IllegalStateException("Object detector was not initialized!")
        }
        return objectDetector!!.inputSize.width
    }

    protected abstract fun initializeModel(
        context: Context,
        fileType: SDKFileType,
        version: Int
    ): Boolean

    abstract fun getExtension(): String

}