package ai.passio.passiosdk.core.aes

import ai.passio.passiosdk.core.utils.FileUtil
import android.content.res.AssetManager
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException
import java.io.InputStream
import java.nio.ByteBuffer
import java.util.Locale
import javax.crypto.Cipher
import javax.crypto.CipherInputStream
import javax.crypto.spec.IvParameterSpec
import javax.crypto.spec.SecretKeySpec

private const val KEY_SIZE = 32
private const val CYPHER_INSTANCE = "AES/CBC/PKCS7Padding"
private const val SECRET_KEY_INSTANCE = "PBKDF2WithHmacSHA1"
private const val AES_SALT = "exampleSalt"

internal fun ByteArray.toHexUpper(): String {
    return this.joinToString(separator = "") { String.format("%02X", (it.toInt() and 0xFF)) }
}

internal fun String.hexToByteArray(): ByteArray {
    return this.chunked(2).map { it.toUpperCase(Locale.ROOT).toInt(16).toByte() }.toByteArray()
}

internal class CryptoHandler private constructor() {

    companion object {
        internal val instance: CryptoHandler by lazy { CryptoHandler() }

        fun readFile(filename: String, assetManager: AssetManager): ByteArray? {
            var inputStream: InputStream? = null
            try {
                inputStream = assetManager.open(filename)
                val bytes = ByteArray(inputStream.available())
                inputStream.read(bytes)
                return bytes
            } catch (e: IOException) {
                e.printStackTrace()
            } finally {
                try {
                    inputStream?.close()
                } catch (e: IOException) {
                    e.printStackTrace()
                }
            }
            return null
        }
    }

    private val cipher = Cipher.getInstance(CYPHER_INSTANCE)

    fun getOutputSize(inputSize: Int, key: ByteArray, iv: ByteArray): Int {
        val keySpec = SecretKeySpec(key, "AES")
        val ivSpec = IvParameterSpec(iv)
        cipher.init(Cipher.DECRYPT_MODE, keySpec, ivSpec)
        return cipher.getOutputSize(inputSize)
    }

    fun decryptInMemory(
        inputBuffer: ByteBuffer,
        outputBuffer: ByteBuffer,
        key: ByteArray,
        iv: ByteArray
    ) {
        val keySpec = SecretKeySpec(key, "AES")
        val ivSpec = IvParameterSpec(iv)
        cipher.init(Cipher.DECRYPT_MODE, keySpec, ivSpec)

        cipher.doFinal(inputBuffer, outputBuffer)
    }

    fun encrypt(data: ByteArray, key: ByteArray, iv: ByteArray): ByteArray {
        val keySpec = SecretKeySpec(key, "AES")
        val ivSpec = IvParameterSpec(iv)
        cipher.init(Cipher.ENCRYPT_MODE, keySpec, ivSpec)
        return cipher.doFinal(data)
    }

    fun decrypt(data: ByteArray, key: ByteArray, iv: ByteArray): ByteArray {
        val keySpec = SecretKeySpec(key, "AES")
        val ivSpec = IvParameterSpec(iv)
        cipher.init(Cipher.DECRYPT_MODE, keySpec, ivSpec)
        return cipher.doFinal(data)
    }

    fun decrypt(
        inputStream: InputStream,
        outPath: String,
        key: ByteArray,
        iv: ByteArray
    ) {
        val keySpec = SecretKeySpec(key, "AES")
        val ivSpec = IvParameterSpec(iv)
        cipher.init(Cipher.DECRYPT_MODE, keySpec, ivSpec)

        var outputStream: FileOutputStream? = null
        var cipherInputStream: CipherInputStream? = null
        try {
            outputStream = FileOutputStream(outPath)
            cipherInputStream = CipherInputStream(inputStream, cipher)
            var count: Int
            val buffer = ByteArray(4096)
            while (cipherInputStream.read(buffer).also { count = it } != -1) {
                outputStream.write(buffer, 0, count)
            }
        } catch (e: IOException) {
            e.printStackTrace()
        } finally {
            outputStream?.flush()
            outputStream?.close()
            cipherInputStream?.close()
        }
    }

    fun decryptToFile(
        key: ByteArray,
        iv: ByteArray,
        file: File,
        dst: String
    ) {
        val keySpec = SecretKeySpec(key, "AES")
        val ivSpec = IvParameterSpec(iv)
        cipher.init(Cipher.DECRYPT_MODE, keySpec, ivSpec)

        FileUtil.openTempFile(dst) { tmpFile ->
            val fis = FileInputStream(file)
            var fos: FileOutputStream? = null
            var cis: CipherInputStream? = null

            try {
                fos = FileOutputStream(tmpFile)
                cis = CipherInputStream(fis, cipher)
                var b: Int
                val d = ByteArray(4096)
                while (cis.read(d).also { b = it } != -1) {
                    fos.write(d, 0, b)
                }
                true
            } catch (e: Exception) {
                e.printStackTrace()
                false
            } finally {
                fos?.flush()
                fos?.close()
                cis?.close()
            }
        }

    }

}