package ai.passio.passiosdk.passiofood.user

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.report.PostJsonNetworkTask
import ai.passio.passiosdk.passiofood.data.measurement.UnitEnergy
import ai.passio.passiosdk.passiofood.data.measurement.UnitMass
import ai.passio.passiosdk.passiofood.data.model.PassioFoodAmount
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.passiofood.data.model.PassioNutrients
import ai.passio.passiosdk.passiofood.data.model.PassioResult
import ai.passio.passiosdk.passiofood.user.UserService.toJsonPayload
import org.json.JSONArray
import org.json.JSONObject

private const val URL = "https://api.passiolife.com/v2/products/food/submit/product"
private const val REPORT_URL = "https://api.passiolife.com/v2/products/food/report/product"

internal object UserService {

    fun reportFoodItem(
        refCode: String,
        productCode: String,
        notes: List<String>?,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        if (refCode.isEmpty() && productCode.isEmpty()) {
            callback(PassioResult.Error("Either productCode or refCode must be supplied"))
            return
        }

        TokenService.getInstance().getToken({ token ->
            reportFoodItemInternal(token, refCode, productCode, notes, callback)
        }, { errorMessage ->
            callback(PassioResult.Error(errorMessage))
        })
    }

    private fun reportFoodItemInternal(
        token: String,
        refCode: String,
        productCode: String,
        notes: List<String>?,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        val jsonObject = JSONObject().apply {
            if (refCode.isNotEmpty()) put("refCode", refCode)
            if (productCode.isNotEmpty()) put("productCode", productCode)
            if (notes != null) put("notes", JSONArray(notes))
        }
        val json = jsonObject.toString()
        val headers = mapOf(
            "Content-Type" to "application/json",
            "Authorization" to token
        )
        val task = PostJsonNetworkTask(REPORT_URL, headers, json)
        NetworkService.instance.doRequest(task, object : NetworkCallback<Boolean> {
            override fun onFailure(code: Int, message: String) {
                callback(PassioResult.Error(message))
            }

            override fun onTokenExpired() {
                reportFoodItem(refCode, productCode, notes, callback)
            }

            override fun onSuccess(result: Boolean) {
                callback(PassioResult.Success(result))
            }
        })
    }

    fun submitUserFood(
        foodItem: PassioFoodItem,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        TokenService.getInstance().getToken(
            { token ->
                submitUserFoodInternal(token, foodItem, callback)
            }, { errorMessage ->
                callback(PassioResult.Error(errorMessage))
            })
    }

    private fun submitUserFoodInternal(
        token: String,
        foodItem: PassioFoodItem,
        callback: (result: PassioResult<Boolean>) -> Unit
    ) {
        val json = foodItem.toJsonPayload()
        val headers = mapOf(
            "Content-Type" to "application/json",
            "Authorization" to token
        )
        val task = PostJsonNetworkTask(URL, headers, json)
        NetworkService.instance.doRequest(task, object : NetworkCallback<Boolean> {
            override fun onFailure(code: Int, message: String) {
                callback(PassioResult.Error(message))
            }

            override fun onTokenExpired() {
                submitUserFood(foodItem, callback)
            }

            override fun onSuccess(result: Boolean) {
                callback(PassioResult.Success(result))
            }
        })
    }

    private fun PassioFoodItem.toJsonPayload(): String {
        val ingredient = this.ingredients.first()
        val jsonPayload = JSONObject()

        val jBranded = JSONObject().apply {
            put("ingredients", ingredient.metadata.ingredientsDescription)
            put("owner", this@toJsonPayload.details)
            put("productCode", ingredient.metadata.barcode)
        }
        jsonPayload.put("branded", jBranded)
        jsonPayload.put("id", this.id)
        jsonPayload.put("name", this.name)

        val nutrients = this.nutrientsReference()
        val jArray = nutrients.toJsonPayload()
        jsonPayload.put("nutrients", jArray)

        val jPortions = this.amount.toJsonPayload()
        jsonPayload.put("portions", jPortions)

        return jsonPayload.toString()
    }

    internal fun PassioNutrients.toJsonPayload(): JSONArray {
        val jsonArray = JSONArray()

        val nutrientAccessors: List<Triple<() -> Any?, Long, String>> = listOf(
            Triple({ alcohol() }, 1603211196555, "alcohol"),
            Triple({ calcium() }, 1603211196577, "calcium"),
            Triple({ calories() }, 1603211196548, "calories"),
            Triple({ carbs() }, 1603211196546, "carbs"),
            Triple({ cholesterol() }, 1603211196677, "cholesterol"),
            Triple({ chromium() }, 1603211196586, "chromium"),
            Triple({ fibers() }, 1603211196571, "fibers"),
            Triple({ folicAcid() }, 1603211196640, "folicAcid"),
            Triple({ fat() }, 1603211196545, "fat"),
            Triple({ iron() }, 1603211196579, "iron"),
            Triple({ iodine() }, 1603211196590, "iodine"),
            Triple({ magnesium() }, 1603211196580, "magnesium"),
            Triple({ monounsaturatedFat() }, 1603211196709, "monounsaturatedFat"),
            Triple({ phosphorus() }, 1603211196581, "phosphorus"),
            Triple({ polyunsaturatedFat() }, 1603211196710, "polyunsaturatedFat"),
            Triple({ potassium() }, 1603211196582, "potassium"),
            Triple({ protein() }, 1603211196544, "protein"),
            Triple({ satFat() }, 1603211196679, "satFat"),
            Triple({ selenium() }, 1603211196593, "selenium"),
            Triple({ sodium() }, 1603211196583, "sodium"),
            Triple({ sugars() }, 1603211196751, "sugarTotal"),
            Triple({ sugarsAdded() }, 1603211196674, "sugarAdded"),
            Triple({ sugarAlcohol() }, 1603211196576, "sugarAlcohol"),
            Triple({ transFat() }, 1603211196678, "transFat"),
            Triple({ vitaminD() }, 1603211196604, "vitaminD"),
            Triple({ vitaminB6() }, 1603211196631, "vitaminB6"),
            Triple({ vitaminB12() }, 1603211196634, "vitaminB12"),
            Triple({ vitaminB12Added() }, 1603211196676, "vitaminB12Added"),
            Triple({ vitaminC() }, 1603211196626, "vitaminC"),
            Triple({ vitaminE() }, 1603211196599, "vitaminE"),
            Triple({ vitaminEAdded() }, 1603211196675, "vitaminEAdded"),
            Triple({ vitaminKPhylloquinone() }, 1603211196639, "vitaminKPhylloquinone"),
            Triple({ vitaminKMenaquinone4() }, 1603211196637, "vitaminKMenaquinone4"),
            Triple({ vitaminKDihydrophylloquinone() }, 1603211196638, "vitaminKDihydrophylloquinone"),
            Triple({ vitaminARAE() }, 1603211196596, "vitaminARAE"),
            Triple({ zinc() }, 1603211196585, "zinc")
        )

        for ((getter, id, shortName) in nutrientAccessors) {
            when (val result = getter()) {
                is UnitMass -> {
                    val jNutrient = JSONObject()
                    jNutrient.put("amount", result.value)
                    jNutrient.put("id", id)

                    val jDefaultNutrient = JSONObject().apply {
                        put("shortName", shortName)
                        put("unit", result.unit.symbol)
                    }

                    jNutrient.put("nutrient", jDefaultNutrient)
                    jsonArray.put(jNutrient)
                }
                is UnitEnergy -> {
                    val jNutrient = JSONObject()
                    jNutrient.put("amount", result.value)
                    jNutrient.put("id", id)

                    val jDefaultNutrient = JSONObject().apply {
                        put("shortName", shortName)
                        put("unit", result.unit.symbol)
                    }

                    jNutrient.put("nutrient", jDefaultNutrient)
                    jsonArray.put(jNutrient)
                }
                null -> {}
            }
        }

        vitaminA()?.let { vitaminAAmount ->
            val jNutrient = JSONObject()
            jNutrient.put("amount", vitaminAAmount)
            jNutrient.put("id", 1603211196594)

            val jDefaultNutrient = JSONObject().apply {
                put("shortName", "vitaminA")
                put("unit", "IU")
            }

            jNutrient.put("nutrient", jDefaultNutrient)
            jsonArray.put(jNutrient)
        }

        return jsonArray
    }

    private fun PassioFoodAmount.toJsonPayload(): JSONArray {
        val jsonArray = JSONArray()
        servingSizes.forEach { servingSize ->
            val servingUnit =
                servingUnits.firstOrNull { it.unitName == servingSize.unitName } ?: return@forEach
            val jWeight = JSONObject().apply {
                put("unit", servingUnit.weight.unit.symbol)
                put("value", servingUnit.weight.value)
            }

            val jServing = JSONObject().apply {
                put("name", servingSize.unitName)
                put("quantity", servingSize.quantity)
                put("weight", jWeight)
            }

            jsonArray.put(jServing)
        }
        return jsonArray
    }
}