package ai.passio.passiosdk.core.authentication

import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.network.SimpleNetworkCallback
import ai.passio.passiosdk.core.sharedpreferences.TokenPreferencesManager
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.config.SDKProperties
import android.content.Context
import org.json.JSONObject

internal abstract class TokenService protected constructor(
    context: Context,
    protected val licenseKey: String
) {
    companion object {
        private var INSTANCE: TokenService? = null

        fun create(context: Context, licenseKey: String, useProxy: Boolean) {
            if (useProxy) {
                INSTANCE = ProxyTokenService(context)
                return
            }

            INSTANCE = if (licenseKey.length == AuthenticationService.KEY_LENGTH_LEGACY) {
                LegacyTokenService(context, licenseKey)
            } else {
                LicenseTokenService(context, licenseKey)
            }
        }

        fun getInstance(): TokenService {
            if (INSTANCE == null) {
                throw IllegalAccessException("Token Service instance wasn't created")
            }
            return INSTANCE!!
        }
    }

    private val sharedPrefs = TokenPreferencesManager(context)

    init {
        val currentLicense = sharedPrefs.getLicense()
        if (currentLicense == null || currentLicense != licenseKey) {
            sharedPrefs.deleteToken()
            sharedPrefs.setLicense(licenseKey)
        }
    }

    fun invalidateToken() {
        sharedPrefs.deleteToken()
    }

    open fun getTokenSync(): String? {
        val tokenExpirationTime = sharedPrefs.getTokenExpirationTime()

        if (System.currentTimeMillis() < tokenExpirationTime) {
            val token = sharedPrefs.getToken()
            if (token != null) {
                return token
            }
        }

        val baseUrl = getUrl(SDKProperties.dev)
        val task = TokenFetchTask(baseUrl)
        val result = NetworkService.instance.syncRequest(task) ?: return null

        val responseJson = JSONObject(result)
        val token = "Bearer ${responseJson.getString("access_token")}"
        val expiresIn = responseJson.getInt("expires_in") * 1000L
        PassioLog.i(
            TokenService::class.java.simpleName,
            "Token fetch success, expiration time: $expiresIn ms"
        )
        sharedPrefs.setTokenExpirationTime(
            token,
            System.currentTimeMillis() + expiresIn
        )
        return token
    }

    open fun getToken(
        callback: (token: String) -> Unit,
        errorCallback: (error: String) -> Unit
    ) {
        val tokenExpirationTime = sharedPrefs.getTokenExpirationTime()

        if (System.currentTimeMillis() < tokenExpirationTime) {
            val token = sharedPrefs.getToken()
            if (token != null) {
                callback(token)
                return
            }
        }

        val baseUrl = getUrl(SDKProperties.dev)

        NetworkService.instance.doRequest(
            TokenFetchTask(baseUrl),
            object : SimpleNetworkCallback<String>() {
                override fun onFailure(code: Int, message: String) {
                    PassioLog.i(
                        TokenService::class.java.simpleName,
                        "Token fetch failed - $message"
                    )
                    errorCallback(message)
                }

                override fun onSuccess(result: String) {
                    val responseJson = JSONObject(result)
                    val token = "Bearer ${responseJson.getString("access_token")}"
                    val expiresIn = responseJson.getInt("expires_in") * 1000L
                    PassioLog.i(
                        TokenService::class.java.simpleName,
                        "Token fetch success, expiration time: $expiresIn ms"
                    )
                    sharedPrefs.setTokenExpirationTime(
                        token,
                        System.currentTimeMillis() + expiresIn
                    )
                    callback(token)
                }
            }
        )

        PassioLog.i(TokenService::class.java.simpleName, "Token fetch request")
    }

    abstract fun getUrl(dev: Boolean): String
}