package ai.passio.passiosdk.passiofood.search

import ai.passio.passiosdk.core.authentication.TokenService
import ai.passio.passiosdk.core.network.NetworkCallback
import ai.passio.passiosdk.core.network.NetworkService
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.passiofood.data.model.PassioFoodItem
import ai.passio.passiosdk.core.network.NetworkStringTask
import ai.passio.passiosdk.core.network.PostNetworkTask
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.InflammatoryEffectData
import ai.passio.passiosdk.passiofood.PassioFoodDataInfo
import ai.passio.passiosdk.passiofood.config.SDKProperties
import ai.passio.passiosdk.passiofood.data.DataUtils
import ai.passio.passiosdk.passiofood.toDataModel
import ai.passio.passiosdk.passiofood.token.TokenUsageTracker
import ai.passio.passiosdk.passiofood.upc.ResponseIngredient
import ai.passio.passiosdk.passiofood.utils.DIIHelper
import android.net.Uri
import android.util.Log
import org.json.JSONArray
import org.json.JSONObject

internal object SearchService {

    fun fetchSearchResult(
        term: String,
        callback: (response: SearchResponse?) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            searchWithToken(token, term, callback)
        }
    }

    fun fetchSemanticSearchResult(
        term: String,
        callback: (response: SearchResponse?) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            semanticSearchWithToken(token, term, callback)
        }
    }

    private fun fetchToken(callback: (token: String?) -> Unit) {
        TokenService.getInstance().getToken(
            { token ->
                callback(token)
            }, { _ ->
                callback(null)
            }
        )
    }

    private fun searchWithToken(
        token: String,
        term: String,
        callback: (response: SearchResponse?) -> Unit
    ) {
        val encoded = Uri.encode(term)
        val url = NativeUtils.instance.getSearchURL(SDKProperties.dev)
        val headers = mutableMapOf("Authorization" to token)
        SDKProperties.languageCode?.let {
            headers["Localization-ISO"] = it
        }
        val task = NetworkStringTask("$url?term=$encoded", headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.e(
                    SearchService::class.java.simpleName,
                    "Could not fetch search result: $message"
                )
                callback(null)
            }

            override fun onTokenExpired() {
                fetchSearchResult(term, callback)
            }

            override fun onSuccess(result: String) {
                try {
                    val response = SearchResponse(result)
                    callback(response)
                } catch (e: Exception) {
                    PassioLog.e(
                        this@SearchService::class.java.simpleName,
                        e.message ?: ""
                    )
                    callback(null)
                }
            }
        }, "searchForFood")
    }

    private fun semanticSearchWithToken(
        token: String,
        term: String,
        callback: (response: SearchResponse?) -> Unit
    ) {
        val encoded = Uri.encode(term)
        val url = NativeUtils.instance.getSearchURL(SDKProperties.dev) +
                "?term=$encoded" +
                if (SDKProperties.textModel != null) "&model=${SDKProperties.textModel}" else ""
        val headers = mutableMapOf("Authorization" to token)
        SDKProperties.languageCode?.let {
            headers["Localization-ISO"] = it
        }
        val task = NetworkStringTask(url, headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.e(
                    SearchService::class.java.simpleName,
                    "Could not fetch search result: $message"
                )
                callback(null)
            }

            override fun onTokenExpired() {
                fetchSemanticSearchResult(term, callback)
            }

            override fun onSuccess(result: String) {
                try {
                    val response = SearchResponse(result)
                    callback(response)
                } catch (e: Exception) {
                    PassioLog.e(
                        this@SearchService::class.java.simpleName,
                        e.message ?: ""
                    )
                    callback(null)
                }
            }
        }, "searchForFoodSemantic")
    }

    fun fetchFoodItem(
        labelId: String,
        type: String,
        resultId: String,
        useShortName: Boolean,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            foodItemWithToken(
                token,
                labelId,
                type,
                resultId,
                useShortName,
                callback
            )
        }
    }

    private fun foodItemWithToken(
        token: String,
        labelId: String,
        type: String,
        resultId: String,
        useShortName: Boolean,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        val encoded = DataUtils.encodeMetadata(useShortName)
        val url = NativeUtils.instance.getFoodFetchURL(SDKProperties.dev)
        val headers = mutableMapOf("Authorization" to token)
        SDKProperties.languageCode?.let {
            headers["Localization-ISO"] = it
        }
        val task = NetworkStringTask("$url$labelId/$type/$resultId?metadata=$encoded", headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                callback(null)
            }

            override fun onTokenExpired() {
                fetchFoodItem(
                    labelId,
                    type,
                    resultId,
                    useShortName,
                    callback
                )
            }

            override fun onSuccess(result: String) {
                try {
                    val response = ResponseFood(result)
                    val responseItem = response.results.first()
                    val foodItem = PassioFoodItem.fromSearchResponse(responseItem, useShortName)
                    callback(foodItem)
                } catch (e: Exception) {
                    PassioLog.e(
                        this@SearchService::class.java.simpleName,
                        e.message ?: ""
                    )
                    callback(null)
                }
            }
        }, "fetchFoodItemForDataInfo")
    }

    fun fetchFoodItemForRefCode(
        refCode: String,
        callback: (foodItem: PassioFoodItem?) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            fetchFoodItemForRefCodeInternal(
                token,
                refCode,
                "fetchFoodItemForRefCode",
                { fetchFoodItemForRefCode(refCode, callback) },
                { response ->
                    if (response == null) {
                        callback(null)
                        return@fetchFoodItemForRefCodeInternal
                    }

                    val responseItem = response.results.first()
                    val shortName = DataUtils.metadataHasShortName(refCode)
                    val foodItem = PassioFoodItem.fromSearchResponse(responseItem, shortName)
                    callback(foodItem)
                })
        }
    }

    fun fetchNutrientsForRefCode(
        refCode: String,
        callback: (data: List<InflammatoryEffectData>?) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            fetchFoodItemForRefCodeInternal(
                token,
                refCode,
                "fetchInflammatoryEffectData",
                { fetchNutrientsForRefCode(refCode, callback) },
                { response ->
                    if (response == null) {
                        callback(null)
                        return@fetchFoodItemForRefCodeInternal
                    }

                    val responseItem = response.results.first()
                    val nutrients = mutableListOf<InflammatoryEffectData>()

                    responseItem.ingredients.forEach { ingredient ->
                        val nutrientsIngredient = ingredient.nutrients?.mapNotNull { upcNutrient ->
                            val shortName = upcNutrient.nutrient?.shortName ?: ""
                            val longName = upcNutrient.nutrient?.name ?: ""
                            val name = DIIHelper.shortNames.firstOrNull { it == shortName }
                                ?: DIIHelper.longNames.firstOrNull { it == longName }
                            if (name == null) {
                                null
                            } else {
                                InflammatoryEffectData(
                                    ingredient.name ?: "",
                                    name,
                                    upcNutrient.amount ?: 0.0,
                                    upcNutrient.nutrient?.unit ?: "",
                                    DIIHelper.inflammationScores[name] ?: 0.0,
                                )
                            }
                        }
                        if (nutrientsIngredient != null) {
                            nutrients.addAll(nutrientsIngredient)
                        }
                    }

                    callback(nutrients)
                })
        }
    }

    fun fetchTagsForRefCode(
        refCode: String,
        callback: (data: List<String>?) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(null)
                return@fetchToken
            }

            fetchFoodItemForRefCodeInternal(
                token,
                refCode,
                "fetchTagsFor",
                { fetchTagsForRefCode(refCode, callback) },
                { response ->
                    if (response == null) {
                        callback(null)
                        return@fetchFoodItemForRefCodeInternal
                    }

                    val responseItem = response.results.first()
                    val tags = mutableSetOf<String>()

                    responseItem.ingredients.forEach { ingredient ->
                        ingredient.tags?.let {
                            tags.addAll(it)
                        }
                    }

                    callback(tags.toList())
                })
        }
    }

    private fun fetchFoodItemForRefCodeInternal(
        token: String,
        refCode: String,
        apiName: String,
        tokenExpired: () -> Unit,
        callback: (response: ResponseFood?) -> Unit
    ) {
        val url = NativeUtils.instance.getRefCodeURL(SDKProperties.dev)
        val headers = mutableMapOf("Authorization" to token)
        SDKProperties.languageCode?.let {
            headers["Localization-ISO"] = it
        }
        val task = NetworkStringTask("$url$refCode", headers)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                callback(null)
            }

            override fun onTokenExpired() {
                tokenExpired()
            }

            override fun onSuccess(result: String) {
                try {
                    callback(ResponseFood(result))
                } catch (e: Exception) {
                    PassioLog.e(
                        this@SearchService::class.java.simpleName,
                        e.message ?: ""
                    )
                    callback(null)
                }
            }
        }, apiName)
    }

    fun fetchPredictNextIngredients(
        currentIngredients: List<String>,
        callback: (response: List<PassioFoodDataInfo>) -> Unit
    ) {
        fetchToken { token ->
            if (token == null) {
                callback(emptyList())
                return@fetchToken
            }

            predictNextIngredientsWithToken(token, currentIngredients, callback)
        }
    }

    private fun predictNextIngredientsWithToken(
        token: String,
        currentIngredients: List<String>,
        callback: (response: List<PassioFoodDataInfo>) -> Unit
    ) {
        val url = NativeUtils.instance.getNextIngredientPredictionURL(SDKProperties.dev)
        val headers = mutableMapOf(
            "Content-Type" to "application/json",
            "Authorization" to token,
        )
        SDKProperties.languageCode?.let {
            headers["Localization-ISO"] = it
        }
        val content = currentIngredients.reduce { acc, s -> "$acc, $s" }
        val payload = JSONObject().apply {
            put("content", content)
        }.toString()
        val task = PostNetworkTask(url, headers, payload, 15000, 15000)
        NetworkService.instance.doRequestTrackTokens(task, object : NetworkCallback<String> {
            override fun onFailure(code: Int, message: String) {
                PassioLog.e(
                    SearchService::class.java.simpleName,
                    "Could not fetch predict next ingredient: $message"
                )
                callback(emptyList())
            }

            override fun onTokenExpired() {
                fetchPredictNextIngredients(currentIngredients, callback)
            }

            override fun onSuccess(result: String) {
                val jArray = JSONArray(result)
                val list = mutableListOf<PassioFoodDataInfo>()
                for (i in 0 until jArray.length()) {
                    val response = SearchResult(jArray.getJSONObject(i).toString())
                    val dataInfo = PassioFoodDataInfo(
                        response.refCode,
                        response.displayName,
                        response.brandName,
                        response.iconId,
                        response.score,
                        response.scoredName,
                        response.labelId,
                        response.type,
                        response.resultId,
                        true,
                        response.nutritionPreview.toDataModel(),
                        response.tags
                    )
                    list.add(dataInfo)
                }
                callback(list)
            }
        }, "predictNextIngredients")
    }
}