package ai.passio.passiosdk.passiofood.data.model

import ai.passio.passiosdk.passiofood.PassioID
import ai.passio.passiosdk.passiofood.upc.ResponseIngredient
import org.json.JSONException

internal data class PassioIDAttributes constructor(
    /**
     * ID of the food entry.
     */
    val passioID: PassioID,
    /**
     * Name of the food entry.
     */
    val name: String,
    /**
     * Defines the type of the food entry.
     */
    val entityType: PassioIDEntityType,
    /**
     * Contains a list of all of the parent nodes of the food entry.
     */
    val parents: List<PassioAlternative>?,
    /**
     * Contains a list of all of the children nodes of the food entry.
     */
    val children: List<PassioAlternative>?,
    /**
     * Contains a list of all of the sibling nodes of the food entry.
     */
    val siblings: List<PassioAlternative>?,
    /**
     * If the food entry represent a single food item (as opposed to a recipe), this field will
     * contain the information for the food item.
     */
    val passioFoodItemData: PassioFoodItemData?,
    /**
     * If the food entry represent a recipe (as opposed to a single food item), this field will
     * contain the information for that recipe.
     */
    val passioFoodRecipe: PassioFoodRecipe?,
    val condiments: List<PassioFoodItemData>? = null,
    val confusionAlternatives: List<PassioID>? = null,
    val invisibleIngredients: List<PassioID>? = null
) {

    val parentsPassioIDs = parents?.map { it.passioID }
    val childrenPassioIDs = children?.map { it.passioID }

    internal constructor(
        copy: PassioIDAttributes,
        passioID: PassioID,
        name: String,
        children: List<PassioAlternative>?,
        parents: List<PassioAlternative>?,
        siblings: List<PassioAlternative>?,
        confusionAlternatives: List<PassioID>? = null,
        invisibleIngredients: List<PassioID>? = null
    ) : this(
        passioID,
        name,
        copy.entityType,
        parents,
        children,
        siblings,
        copy.passioFoodItemData,
        copy.passioFoodRecipe,
        copy.condiments,
        confusionAlternatives,
        invisibleIngredients
    )

    internal constructor(
        passioID: PassioID,
        name: String,
        entityType: PassioIDEntityType,
        parents: List<PassioAlternative>? = null,
        children: List<PassioAlternative>? = null,
        siblings: List<PassioAlternative>? = null,
        recipePassio: PassioFoodRecipe?,
        confusionAlternatives: List<PassioID>? = null,
        invisibleIngredients: List<PassioID>? = null
    ) : this(
        passioID,
        name,
        entityType,
        parents,
        children,
        siblings,
        null,
        recipePassio,
        null,
        confusionAlternatives,
        invisibleIngredients
    )

    internal constructor(
        passioID: PassioID,
        children: List<PassioID>?
    ) : this(
        passioID,
        "",
        PassioIDEntityType.item,
        null,
        children?.map { PassioAlternative(it, "") },
        null,
        null,
        null,
        null,
        null
    )

    internal constructor(
        passioID: PassioID,
        name: String,
        entityType: PassioIDEntityType,
        parents: List<PassioAlternative>? = null,
        children: List<PassioAlternative>? = null,
        siblings: List<PassioAlternative>? = null,
        passioFoodItemDataForDefault: PassioFoodItemData?,
        confusionAlternatives: List<PassioID>? = null,
        invisibleIngredients: List<PassioID>? = null
    ) : this(
        passioID,
        name,
        entityType,
        parents,
        children,
        siblings,
        passioFoodItemDataForDefault,
        null,
        null,
        null
    )

    /**
     * Helper method to transform the [PassioIDAttributes] object to a [PassioFoodItemData] object
     * if the PassioIDAttributes object corresponds to a single food item. This is used to enter
     */
    fun passioFoodItemDataForGroup(): PassioFoodItemData? {
        if (passioFoodItemData == null) {
            return null
        }

        val foodItemDataGroup = passioFoodItemData
        foodItemDataGroup.passioID = passioID
        foodItemDataGroup.name = name
        return foodItemDataGroup
    }

    /**
     * Helper method to transform the [PassioIDAttributes] object to a [PassioFoodRecipe] object
     * if the PassioIDAttributes object corresponds to a recipe.
     */
    fun foodRecipeForGroup(): PassioFoodRecipe? {
        if (passioFoodRecipe == null) return null

        val foodItemDataForRecipe = passioFoodRecipe
        foodItemDataForRecipe.passioID = passioID
        foodItemDataForRecipe.name = name
        return foodItemDataForRecipe
    }

    fun isOpenFood(): Boolean {
        return passioFoodItemData?.isOpenFood()
            ?: passioFoodRecipe?.isOpenFood() ?: false
    }

    fun getOpenFoodLicense(): String? {
        return passioFoodItemData?.getOpenFoodLicense()
            ?: passioFoodRecipe?.foodItems?.firstOrNull {
                it.getOpenFoodLicense() != null
            }?.getOpenFoodLicense()
    }

    @Throws(JSONException::class)
    internal constructor(
        responseIngredient: ResponseIngredient,
        type: PassioIDEntityType,
        localAttrs: PassioIDAttributes? = null,
    ) : this(
        type.value + responseIngredient.branded?.productCode,
        responseIngredient.name ?: "",
        type,
        localAttrs?.parents,
        localAttrs?.children,
        localAttrs?.siblings,
        PassioFoodItemData(responseIngredient),
        null,
        null,
        null,
        null
    )

    internal fun deepCopy(): PassioIDAttributes {
        return PassioIDAttributes(
            passioID,
            name,
            entityType,
            parents,
            children,
            siblings,
            passioFoodItemData?.deepCopy(),
            passioFoodRecipe?.deepCopy(),
            condiments,
            confusionAlternatives,
            invisibleIngredients
        )
    }
}