package ai.passio.passiosdk.core.camera

import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.core.utils.YuvToRgbConverter
import android.annotation.SuppressLint
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.ImageFormat
import android.graphics.Matrix
import android.media.Image
import android.util.Log
import android.util.Size
import android.view.MotionEvent
import androidx.camera.core.*
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.camera.view.PreviewView
import androidx.core.content.ContextCompat
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min


private const val RATIO_4_3_VALUE = 4.0 / 3.0
private const val RATIO_16_9_VALUE = 16.0 / 9.0

internal class PassioCameraXProxy : PassioCameraProxy() {

    private var camera: Camera? = null
    private val cameraExecutor: ExecutorService = Executors.newSingleThreadExecutor()
    private var cameraProvider: ProcessCameraProvider? = null
    private var frameSize: Size? = null
    private var previewSize: Size? = null
    private var captureAspectRatio: Int = AspectRatio.RATIO_4_3
    private var previewUseCase: Preview? = null
    private var imageCapture: ImageCapture? = null
    private var converter: YuvToRgbConverter? = null

    private fun createImageCapture(viewProvider: PassioCameraViewProvider) {
        imageCapture = ImageCapture.Builder()
            .setTargetRotation(viewProvider.requestPreviewView().display.rotation)
            .setTargetResolution(Size(1080, 1920))
            .build()
        converter = YuvToRgbConverter(viewProvider.requestPreviewView().context)
    }

    @SuppressLint("RestrictedApi")
    override fun startCamera(
        displayRotation: Int,
        @CameraSelector.LensFacing cameraFacing: Int,
        onCameraReady: (data: PassioCameraData) -> Unit
    ) {
        val viewProvider = viewProviderRef?.get()
        if (viewProvider == null) {
            PassioLog.e(
                this::class.java.simpleName,
                "Unable to start camera, view provider is null!"
            )
            return
        }

        val cameraSelector = CameraSelector.Builder().requireLensFacing(cameraFacing).build()
        val cameraProviderFuture =
            ProcessCameraProvider.getInstance(viewProvider.requestPreviewView().context)
        cameraProviderFuture.addListener({

            cameraProvider = cameraProviderFuture.get() as ProcessCameraProvider
            cameraProvider!!.unbindAll()

            previewUseCase = Preview.Builder().apply {
                setTargetRotation(displayRotation)
                setTargetAspectRatio(AspectRatio.RATIO_16_9)
            }.build().also {
                viewProvider.requestPreviewView().implementationMode =
                    PreviewView.ImplementationMode.PERFORMANCE
                it.setSurfaceProvider(viewProvider.requestPreviewView().surfaceProvider)
            }

            val imageAnalyzer = ImageAnalysis.Builder().apply {
                setTargetRotation(displayRotation)
                setTargetAspectRatio(AspectRatio.RATIO_16_9)
            }.build().also {
                it.setAnalyzer(cameraExecutor, CameraAnalyzer())
            }

            createImageCapture(viewProvider)

            try {
                camera =
                    cameraProvider!!.bindToLifecycle(
                        viewProvider.requestCameraLifecycleOwner(),
                        cameraSelector,
                        previewUseCase,
                        imageAnalyzer,
                        imageCapture
                    )
                frameSize = imageAnalyzer.attachedSurfaceResolution
                previewSize = previewUseCase!!.attachedSurfaceResolution

                passioCameraListener?.onFrameSize(
                    frameSize!!.width,
                    frameSize!!.height,
                    previewSize!!.width,
                    previewSize!!.height,
                    getSensorRotation() ?: 0
                )
                val minZoom = camera!!.cameraInfo.zoomState.value?.minZoomRatio
                val maxZoom = camera!!.cameraInfo.zoomState.value?.maxZoomRatio
                val previewResolution = previewUseCase!!.resolutionInfo?.resolution
                val data = PassioCameraData(
                    minZoom ?: 0f,
                    maxZoom ?: 0f,
                    previewResolution?.width ?: 0,
                    previewResolution?.height ?: 0
                )

                PassioLog.i(
                    this::class.java.simpleName,
                    "Camera configured: $data"
                )
                onCameraReady(data)
            } catch (e: Exception) {
                Log.e(PassioCameraXProxy::class.java.simpleName, "Use case binding failed", e)
            }

        }, ContextCompat.getMainExecutor(viewProvider.requestPreviewView().context))
    }

    @SuppressLint("RestrictedApi")
    override fun startCamera(
        configurator: PassioCameraConfigurator,
        onCameraReady: (data: PassioCameraData) -> Unit
    ) {
        val viewProvider = viewProviderRef?.get()
        if (viewProvider == null) {
            PassioLog.e(
                this::class.java.simpleName,
                "Unable to start camera, view provider is null!"
            )
            return
        }

        val cameraSelector =
            CameraSelector.Builder().requireLensFacing(configurator.cameraFacing()).build()
        val cameraProviderFuture =
            ProcessCameraProvider.getInstance(viewProvider.requestPreviewView().context)
        cameraProviderFuture.addListener({
            if (viewProviderRef?.get() == null) {
                PassioLog.e(
                    this::class.java.simpleName,
                    "Unable to start camera, view provider is null!"
                )
                return@addListener
            }

            cameraProvider = cameraProviderFuture.get() as ProcessCameraProvider
            cameraProvider!!.unbindAll()

            val imageAnalyzer = configurator.analyzer().also {
                it.setAnalyzer(cameraExecutor, CameraAnalyzer())
            }
            previewUseCase = configurator.preview()

            createImageCapture(viewProvider)

            try {
                camera =
                    cameraProvider!!.bindToLifecycle(
                        viewProvider.requestCameraLifecycleOwner(),
                        cameraSelector,
                        previewUseCase,
                        imageAnalyzer,
                        imageCapture
                    )

                val minZoom = camera!!.cameraInfo.zoomState.value?.minZoomRatio
                val maxZoom = camera!!.cameraInfo.zoomState.value?.maxZoomRatio
                if (minZoom != null && maxZoom != null) {
                    val zoomLevel = configurator.setZoomRatio(minZoom, maxZoom)
                    camera!!.cameraControl.setZoomRatio(zoomLevel)
                }

                frameSize = imageAnalyzer.attachedSurfaceResolution
                previewSize = previewUseCase!!.attachedSurfaceResolution

                passioCameraListener?.onFrameSize(
                    frameSize!!.width,
                    frameSize!!.height,
                    previewSize!!.width,
                    previewSize!!.height,
                    getSensorRotation() ?: 0
                )
                val previewResolution = previewUseCase!!.resolutionInfo?.resolution
                val data = PassioCameraData(
                    minZoom ?: 0f,
                    maxZoom ?: 0f,
                    previewResolution?.width ?: 0,
                    previewResolution?.height ?: 0
                )

                PassioLog.i(
                    this::class.java.simpleName,
                    "Frame size: $frameSize, preview size: $previewSize"
                )
                onCameraReady(data)
            } catch (e: Exception) {
                Log.e(PassioCameraXProxy::class.java.simpleName, "Use case binding failed", e)
            }

        }, ContextCompat.getMainExecutor(viewProvider.requestPreviewView().context))
    }

    override fun stopCamera() {
        super.stopCamera()
        cameraProvider?.unbindAll()
        cameraProvider = null
        camera = null
    }


    private fun aspectRatio(width: Int, height: Int): Int {
        val previewRatio = max(width, height).toDouble() / min(width, height)
        if (abs(previewRatio - RATIO_4_3_VALUE) <= abs(previewRatio - RATIO_16_9_VALUE)) {
            return AspectRatio.RATIO_4_3
        }
        return AspectRatio.RATIO_16_9
    }

    override fun changeAspectRatio(
        displayRotation: Int,
        aspectRatio: Int,
        @CameraSelector.LensFacing cameraFacing: Int,
        onCameraReady: (data: PassioCameraData) -> Unit
    ) {
        this.captureAspectRatio = aspectRatio
        startCamera(displayRotation, cameraFacing, onCameraReady)
    }

    override fun runOnCameraThread(task: Runnable) {
        cameraExecutor.submit(task)
    }

    override fun getSensorRotation(): Int? = camera?.cameraInfo?.sensorRotationDegrees

    override fun getFrameSize(): Size = frameSize ?: Size(0, 0)

    override fun getPreviewSize(): Size = previewSize ?: Size(0, 0)

    @SuppressLint("ClickableViewAccessibility")
    override fun enableTapToFocus() {
        val previewView = viewProviderRef?.get()?.requestPreviewView() ?: return

        previewView.setOnTouchListener { v, event ->
            return@setOnTouchListener when (event.action) {
                MotionEvent.ACTION_DOWN -> {
                    true
                }

                MotionEvent.ACTION_UP -> {
                    val factory: MeteringPointFactory = SurfaceOrientedMeteringPointFactory(
                        previewView.width.toFloat(), previewView.height.toFloat()
                    )
                    val autoFocusPoint = factory.createPoint(event.x, event.y)
                    try {
                        camera?.cameraControl?.startFocusAndMetering(
                            FocusMeteringAction.Builder(
                                autoFocusPoint,
                                FocusMeteringAction.FLAG_AF
                            ).disableAutoCancel().build()
                        )
                    } catch (e: CameraInfoUnavailableException) {
                        Log.e(
                            PassioCameraXProxy::class.java.simpleName,
                            "Cannot access camera, manual focus failed",
                            e
                        )
                    }
                    true
                }

                else -> false
            }
        }
    }

    override fun enableFlashlight(enabled: Boolean): Boolean {
        if (cameraProvider == null || previewUseCase == null) {
            return false
        }

        if (!cameraProvider!!.isBound(previewUseCase!!)) {
            return false
        }

        camera?.cameraControl?.enableTorch(enabled) ?: return false
        return true
    }

    override fun setZoomLevel(zoom: Float): Boolean {
        if (cameraProvider == null || previewUseCase == null) {
            return false
        }

        if (!cameraProvider!!.isBound(previewUseCase!!)) {
            return false
        }

        val zoomLevels = getMinMaxCameraZoomLevel() ?: return false
        val minZoom = zoomLevels.first ?: 1f
        val maxZoom = zoomLevels.second ?: 1f

        val correctedZoom = zoom.coerceIn(minZoom, maxZoom)
        camera?.cameraControl?.setZoomRatio(correctedZoom) ?: return false

        return true
    }

    override fun getMinMaxCameraZoomLevel(): Pair<Float?, Float?>? {
        if (cameraProvider == null || previewUseCase == null) {
            return null
        }

        if (!cameraProvider!!.isBound(previewUseCase!!)) {
            return null
        }

        val minZoom = camera?.cameraInfo?.zoomState?.value?.minZoomRatio
        val maxZoom = camera?.cameraInfo?.zoomState?.value?.maxZoomRatio

        return minZoom to maxZoom
    }

    override fun takePicture(onResult: (bitmap: Bitmap?) -> Unit) {
        if (imageCapture == null) {
            onResult(null)
            return
        }

        imageCapture!!.takePicture(cameraExecutor, object : ImageCapture.OnImageCapturedCallback() {
            override fun onCaptureSuccess(image: ImageProxy) {
                if (converter == null) {
                    onResult(null)
                    return
                }

                val width = image.width
                val height = image.height
                var bitmap: Bitmap? = null

                if (image.format == ImageFormat.JPEG) {
                    val buffer = image.planes[0].buffer
                    val bytes = ByteArray(buffer.capacity())
                    buffer.get(bytes)
                    bitmap = BitmapFactory.decodeByteArray(bytes, 0, bytes.size, null)

                    val rotationDegrees = image.imageInfo.rotationDegrees

                    // Step 4: Rotate the bitmap if necessary
                    if (rotationDegrees != 0) {
                        val matrix = Matrix()
                        matrix.postRotate(rotationDegrees.toFloat())

                        bitmap = Bitmap.createBitmap(bitmap, 0, 0, bitmap.width, bitmap.height, matrix, true)
                    }
                } else {
                    bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888)
                    converter!!.yuvToRgb(image, bitmap)
                }

                onResult(bitmap)
            }

            override fun onError(exception: ImageCaptureException) {
                PassioLog.e(
                    this@PassioCameraXProxy::class.java.simpleName,
                    "Take picture error: ${exception.message}"
                )
                onResult(null)
            }
        })
    }

    protected fun finalize() {
        cameraExecutor.shutdown()
    }

    private inner class CameraAnalyzer : ImageAnalysis.Analyzer {

        private var lastAnalyzedTimestamp = 0L

        @SuppressLint("UnsafeOptInUsageError")
        override fun analyze(image: ImageProxy) {
            //            image.close()
            if (image.image == null) {
                image.close()
                return
            }

            val currentTimestamp = System.currentTimeMillis()
            if (currentTimestamp - lastAnalyzedTimestamp >= frameTime) {
                lastAnalyzedTimestamp = currentTimestamp
                passioCameraListener?.analyzeImage(image)
            } else {
                image.close()
            }
        }
    }
}