package ai.passio.passiosdk.core.file

import ai.passio.passiosdk.core.aes.CryptoHandler
import ai.passio.passiosdk.core.config.SDKFileConfig
import ai.passio.passiosdk.core.config.SDKFileType
import ai.passio.passiosdk.core.os.NativeUtils
import ai.passio.passiosdk.core.utils.FileUtil
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.config.passio_dataset
import ai.passio.passiosdk.passiofood.config.passio_metadata
import android.content.Context
import android.net.Uri
import android.util.Log
import java.io.File
import java.io.FileOutputStream
import java.lang.Exception

private fun String.fileTypeName() = this.split(".")[0]
private fun String.fileVersion() = this.split(".")[1].toInt()
private fun String.fileExtension() = this.split(".")[2]

internal abstract class PassioFileManager(private val sdkFileConfig: SDKFileConfig) {

    companion object {
        const val CACHE_ROOT = "passio"
        const val CACHE_MODELS = "model"
        const val CACHE_TEMP = "temp"
        const val CACHE_DOWNLOAD = "download"
        const val CACHE_METADATA = "metadata"
        const val TEMP_EXTENSION = "ptmp"
        const val FILES_NOT_COMPLETE = 1
        const val CACHE_NUTRITION_FOLDER = "Nutrition"

        private fun Context.getModelsDir(): String {
            return this.cacheDir.absolutePath + File.separator +
                    CACHE_ROOT + File.separator +
                    CACHE_NUTRITION_FOLDER + File.separator +
                    CACHE_MODELS
        }

        fun Context.getTempDir(): String {
            return this.cacheDir.absolutePath + File.separator +
                    CACHE_ROOT + File.separator +
                    CACHE_NUTRITION_FOLDER + File.separator +
                    CACHE_TEMP
        }

        private fun Context.getDownloadDir(): String {
            return this.cacheDir.absolutePath + File.separator +
                    CACHE_ROOT + File.separator +
                    CACHE_NUTRITION_FOLDER + File.separator +
                    CACHE_DOWNLOAD
        }

        fun Context.getMetadataDir(): String {
            return this.cacheDir.absolutePath + File.separator +
                    CACHE_ROOT + File.separator +
                    CACHE_NUTRITION_FOLDER + File.separator +
                    CACHE_METADATA
        }

        fun Context.getMetadataDownloadDir(): String {
            return this.cacheDir.absolutePath + File.separator +
                    CACHE_ROOT + File.separator +
                    CACHE_NUTRITION_FOLDER + File.separator +
                    CACHE_METADATA + File.separator +
                    CACHE_DOWNLOAD
        }
    }

    abstract fun getAvailableFileTypes(): List<SDKFileType>

    abstract fun printVersions(context: Context)

    protected fun getVersionFromFile(file: File): Int = file.name.fileVersion()

    fun migrateFromLegacyFolderStructure(context: Context) {
        val legacyModelPath = context.cacheDir.absolutePath + File.separator + CACHE_ROOT
        val legacyTempPath = context.cacheDir.absolutePath + File.separator + CACHE_TEMP
        val legacyDownloadPath = context.cacheDir.absolutePath + File.separator + CACHE_DOWNLOAD

        migrateFromLegacyFolder(legacyModelPath, context.getModelsDir())
        migrateFromLegacyFolder(legacyTempPath, context.getTempDir())
        migrateFromLegacyFolder(legacyDownloadPath, context.getDownloadDir())

        val legacyTempFolder = File(legacyTempPath)
        if (legacyTempFolder.exists() && legacyTempFolder.isDirectory) {
            legacyTempFolder.delete()
        }

        val legacyDownloadFolder = File(legacyDownloadPath)
        if (legacyDownloadFolder.exists() && legacyDownloadFolder.isDirectory) {
            legacyDownloadFolder.delete()
        }
    }

    private fun migrateFromLegacyFolder(legacyPath: String, newPath: String) {
        val fileNames = getFileNamesFromFolder(legacyPath)
        fileNames.forEach { name ->
            val file = File(legacyPath + File.separator + name)
            if (file.exists() && !file.isDirectory) {
                if (FileUtil.copyFileToCache(file, newPath) != null) {
                    file.delete()
                }
            }
        }
    }

    fun getInstalledVersion(context: Context): Int {
        val availableTypes = getAvailableFileTypes()
        val installedFiles = getInstalledFiles(context, availableTypes)
        return when {
            installedFiles.isEmpty() -> 0 // No files present
            installedFiles.size == availableTypes.size -> getVersionFromFile(installedFiles.first()) // All files present
            else -> FILES_NOT_COMPLETE // Some files present, partial install
        }
    }

    fun deleteInstalledFiles(context: Context) {
        val root = File(context.getModelsDir())
        if (!root.exists() || !root.isDirectory) {
            return
        }

        root.listFiles()?.forEach { file ->
            file?.delete()
        }
    }

    fun getDownloadedVersion(context: Context): Int {
        val availableTypes = getAvailableFileTypes()
        val root = File(context.getDownloadDir())
        if (!root.exists() && !root.isDirectory) {
            return 0
        }

        val tempFiles = mutableListOf<File>()
        availableTypes.forEach { requiredFileType ->
            val file = getStoredFile(context, requiredFileType, context.getDownloadDir())
            if (file != null) tempFiles.add(file)
        }

        return when {
            tempFiles.isEmpty() -> 0 // No files present
            tempFiles.size == availableTypes.size -> getVersionFromFile(tempFiles.first()) // All files present
            else -> FILES_NOT_COMPLETE // Some files present, partial install
        }
    }

    fun deleteDownloadedFiles(context: Context) {
        val root = File(context.getDownloadDir())
        if (!root.exists() || !root.isDirectory) {
            return
        }

        root.listFiles()?.forEach { file ->
            file?.delete()
        }
    }

    fun getTempVersion(context: Context): Int {
        val availableTypes = getAvailableFileTypes()
        val root = File(context.getTempDir())
        if (!root.exists() && !root.isDirectory) {
            return 0
        }

        val tempFiles = mutableListOf<File>()
        availableTypes.forEach { requiredFileType ->
            val file = getStoredFile(context, requiredFileType, context.getTempDir())
            if (file != null) tempFiles.add(file)
        }

        return when {
            tempFiles.isEmpty() -> 0 // No files present
            tempFiles.size == availableTypes.size -> getVersionFromFile(tempFiles.first()) // All files present
            else -> FILES_NOT_COMPLETE // Some files present, partial install
        }
    }

    fun deleteTempFiles(context: Context) {
        val root = File(context.getTempDir())
        if (!root.exists() || !root.isDirectory) {
            return
        }

        root.listFiles()?.forEach { file ->
            file?.delete()
        }
    }

    private fun getInstalledFiles(
        context: Context,
        requiredFiles: List<SDKFileType>
    ): List<File> {
        val root = File(context.getModelsDir())
        if (!root.exists() && !root.isDirectory) {
            return listOf()
        }

        val installedFiles = mutableListOf<File>()
        requiredFiles.forEach { requiredFileType ->
            val file = getStoredFile(context, requiredFileType)
            if (file != null) installedFiles.add(file)
        }

        return installedFiles
    }

    fun getShippedVersion(context: Context): Int {
        val availableTypes = getAvailableFileTypes()
        val shippedFiles = getFilesFromAssets(context, availableTypes)
        if (shippedFiles.isEmpty()) {
            return 0
        }
        if (shippedFiles.size != availableTypes.size) {
            return FILES_NOT_COMPLETE
        }

        return shippedFiles.first().fileVersion()
    }

    fun getFilesFromAssets(
        context: Context,
        requiredFiles: List<SDKFileType>
    ): List<String> {
        val files = mutableListOf<String>()

        FileUtil.listAssetFiles(context.assets, "").forEach { fileName ->
            requiredFiles.forEach { requiredFileType ->
                val alternativeName = requiredFileType.alternative()?.name
                if (fileName.startsWith(requiredFileType.name)) {
                    files.add(fileName)
                } else if (alternativeName != null && fileName.startsWith(alternativeName)) {
                    files.add(fileName)
                }
            }
        }

        return files
    }

    fun getStoredFile(
        context: Context,
        fileType: SDKFileType,
        rootDir: String = context.getModelsDir(),
        checkAlternatives: Boolean = true
    ): File? {
        val root = File(rootDir)
        if (!root.exists() && !root.isDirectory) {
            return null
        }

        root.listFiles()?.forEach { file ->
            if (file.isFile) {
                val components = file.name.split(".")
                if (components.size != 3) {
                    return@forEach
                }

                if (components[0] == fileType.name ||
                    (checkAlternatives && components[0] == fileType.alternative()?.name)
                ) {
                    return file
                }
            }
        }

        return null
    }

    private fun copyFromDownloadToTempWithLowMemory(context: Context) {
        val srcRoot = File(context.getDownloadDir())
        if (!srcRoot.exists() && !srcRoot.isDirectory) {
            return
        }

        val dstRoot = File(context.getTempDir())
        if (!dstRoot.exists()) {
            dstRoot.mkdirs()
        }

        srcRoot.listFiles()?.forEach { file ->
            val cmp = file.name.split(".")
            val dstFile = File(dstRoot, "${cmp[0]}.${cmp[1]}.passio")
            CryptoHandler.instance.decryptToFile(
                NativeUtils.instance.getKey1().toByteArray(),
                NativeUtils.instance.getKey2().toByteArray(),
                file,
                dstFile.absolutePath
            )
        }
    }

    private fun copyFromDownloadToTemp(context: Context, fileNames: List<String>) {
        val srcRoot = File(context.getDownloadDir())
        if (!srcRoot.exists() && !srcRoot.isDirectory) {
            return
        }

        srcRoot.listFiles()?.forEach { file ->
            if (file.name in fileNames) {
                FileUtil.copyFileToCache(file, context.getTempDir())
            }
        }
    }

    private fun deleteDownloadFolder(context: Context) {
        val root = File(context.getDownloadDir())
        if (!root.exists() && !root.isDirectory) {
            return
        }

        FileUtil.deleteRecursive(root)
    }

    fun copyToTempWithVersionUpgrade(context: Context, newFile: File): Uri? {
        val root = File(context.getTempDir())
        if (!root.exists() && !root.isDirectory) {
            root.mkdirs()
        }

        val newFileComponents = newFile.name.split(".")
        val newFileName = newFileComponents[0]
        val newFileVersion = newFileComponents[1].toInt()

        root.listFiles()?.forEach { file ->
            if (file.isFile) {
                val components = file.name.split(".")
                if (components.size != 3) {
                    return@forEach
                }

                if (newFileName == components[0] && newFileVersion > components[1].toInt()) {
                    file.delete()
                }
            }
        }

        return FileUtil.copyFileToCache(newFile, context.getTempDir())
    }

    fun copyFromTempToPassioWithPrune(context: Context) {
        val root = File(context.getTempDir())
        if (!root.exists() && !root.isDirectory) {
            return
        }

        root.listFiles()?.forEach { file ->
            if (file.isFile) {
                val components = file.name.split(".")
                if (components.size != 3) {
                    return@forEach
                }

                copyFileAndDeleteOldVersions(context, components[0], file)
            }
        }

        return
    }

    private fun copyFileAndDeleteOldVersions(context: Context, fileName: String, file: File) {
        val fileType = sdkFileConfig.nameToSDKFileType(fileName)
        val newVersion = file.name.split(".")[1]
        FileUtil.copyFileToCache(
            file,
            context.getModelsDir()
        )
        FileUtil.deleteFile(Uri.fromFile(file))
        deleteOldVersions(
            context,
            fileType,
            newVersion.toInt()
        )
    }

    private fun deleteOldVersions(
        context: Context,
        sdkFileType: SDKFileType,
        newestVersion: Int
    ) {
        val root = File(context.getModelsDir())
        if (!root.exists() && !root.isDirectory) {
            return
        }

        root.listFiles()?.forEach { file ->
            val fileName = file.name.split(".")[0]
            val sdkFileAlternativeName = sdkFileType.alternative()?.name
            if (file.isFile &&
                (fileName == sdkFileType.name || fileName == sdkFileAlternativeName)
            ) {
                val fileVersion = file.name.split(".")[1]
                if (fileVersion.toInt() < newestVersion) {
                    FileUtil.deleteFile(Uri.fromFile(file))
                }
            }
        }
    }

    fun getPathToDownloadFolder(context: Context): String {
        return context.getDownloadDir()
    }

    private fun getFileNamesFromFolder(path: String): List<String> {
        val root = File(path)
        if (!root.exists() && !root.isDirectory) {
            return listOf()
        }

        return root.listFiles()?.map { it.name } ?: listOf()
    }

    fun getDeliveredFilesVersion(files: List<Uri>): Int {
        var deliveredVersion = 0
        for (uri in files) {
            if (uri.path == null) {
                return 0
            }

            val file = File(uri.path!!)
            if (!file.exists()) {
                return 0
            }

            val components = file.name.split(".")
            if (components.size != 3) {
                return 0
            }

            val fileVersion = file.name.fileVersion()
            if (deliveredVersion == 0) {
                deliveredVersion = fileVersion
            }
            // Check if all files have the same version
            if (fileVersion != deliveredVersion) {
                return 0
            }
        }

        return deliveredVersion
    }

    // TODO refactor unzip when download is done
    fun transferFromDownloadToTemp(
        context: Context,
        withoutDecryption: List<SDKFileType> = emptyList()
    ) {
        val files = getFileNamesFromFolder(context.getDownloadDir())

        // Get files with the highest version
        val maxVersion = files.maxOfOrNull { it.fileVersion() } ?: return
        val maxVersionFiles = files.filter { it.fileVersion() == maxVersion }

        val passioFiles = maxVersionFiles.filter { it.fileExtension() == "passiosecure" }
        val passio2Files = maxVersionFiles.filter { it.fileExtension() == "passiosecure2" }

        if (passio2Files.size >= passioFiles.size) {
            val rootDir = context.getDownloadDir()
            val outPath = context.getTempDir()
            val outDir = File(outPath)
            if (!outDir.exists() && !outDir.isDirectory) {
                outDir.mkdirs()
            }
            passio2Files.forEach { fileName ->
                val fileType = sdkFileConfig.nameToSDKFileType(fileName.fileTypeName())
                if (fileType in withoutDecryption) {
                    val file = getStoredFile(context, fileType, context.getDownloadDir())
                        ?: return@forEach
                    val newName = "${fileName.fileTypeName()}.${fileName.fileVersion()}.passio2"
                    FileUtil.copyFileToCache(file, context.getTempDir(), newName)
                    return@forEach
                }

                val passio2File = File(rootDir + File.separator + fileName)
                val filePath =
                    outPath + File.separator + fileName.replace("passiosecure2", "passio2")

                CryptoHandler.instance.decryptToFile(
                    NativeUtils.instance.getKey1().toByteArray(),
                    NativeUtils.instance.getKey2().toByteArray(),
                    passio2File,
                    filePath
                )
            }
        } else {
            copyFromDownloadToTemp(context, passioFiles)
        }

        deleteDownloadFolder(context)
    }

    fun isSDKUsingCompressedModels(context: Context): Boolean {
        val root = File(context.getModelsDir())
        if (!root.exists() && !root.isDirectory) {
            return false
        }

        val files = root.listFiles() ?: return false

        files.forEach { file ->
            if (file.extension != "passio2") {
                return false
            }
        }

        return true
    }

    fun isSDKUsingSecuredModels(context: Context): Boolean {
        val root = File(context.getModelsDir())
        if (!root.exists() && !root.isDirectory) {
            return false
        }

        val files = root.listFiles() ?: return false

        files.forEach { file ->
            if (file.extension != "passiosecure") {
                return false
            }
        }

        return true
    }

    fun logDirectoryTree(context: Context) {
        val rootFile = File(context.getModelsDir())
        var print = FileUtil.printDirectoryTree(rootFile) + "\n"
        val tempFile = File(context.getTempDir())
        print += FileUtil.printDirectoryTree(tempFile) + "\n"
        val downFile = File(context.getDownloadDir())
        print += FileUtil.printDirectoryTree(downFile)
        PassioLog.i(this::class.java.simpleName, "Folder structure: \n$print")
    }

    fun deleteDeprecatedFiles(context: Context) {
        val allowedFiles = getAvailableFileTypes()
        val alternativeFiles = allowedFiles.mapNotNull { it.alternative() }
        val rootFolder = File(context.cacheDir.absolutePath + File.separator + CACHE_ROOT)
        deleteFileTypesNotAllowed(rootFolder, allowedFiles + alternativeFiles)
        val tempFolder = File(context.cacheDir.absolutePath + File.separator + CACHE_TEMP)
        deleteFileTypesNotAllowed(tempFolder, allowedFiles + alternativeFiles)
        val downFolder = File(context.cacheDir.absolutePath + File.separator + CACHE_DOWNLOAD)
        deleteFileTypesNotAllowed(downFolder, allowedFiles + alternativeFiles)
    }

    private fun deleteFileTypesNotAllowed(root: File, allowed: List<SDKFileType>) {
        root.listFiles()?.forEach { file ->
            if (file.isFile &&
                file.name.split(".")[0] !in allowed.map { it.name }
            ) {
                file.delete()
            }
        }
    }

    fun clearTempFiles(context: Context) {
        val rootFolder = File(context.getModelsDir())
        deleteAllTempFilesIn(rootFolder)
        val tempFolder = File(context.getTempDir())
        deleteAllTempFilesIn(tempFolder)
        val downFolder = File(context.getDownloadDir())
        deleteAllTempFilesIn(downFolder)
        val metaDownFolder = File(context.getMetadataDownloadDir())
        deleteAllTempFilesIn(metaDownFolder)
    }

    fun clearTempFilesInDownload(context: Context) {
        val downFolder = File(context.cacheDir.absolutePath + File.separator + CACHE_DOWNLOAD)
        deleteAllTempFilesIn(downFolder)
    }

    private fun deleteAllTempFilesIn(dir: File) {
        if (!dir.isDirectory) {
            return
        }

        dir.listFiles()?.forEach { file ->
            if (file.extension == TEMP_EXTENSION) {
                PassioLog.e(this::class.java.simpleName, "Removed temp file: ${file.name}")
                file.delete()
            }
        }
    }

    fun saveMetadata(context: Context, name: String, contents: String, toDownload: Boolean): File? {
        val path = if (toDownload) {
            val root = File(context.getMetadataDownloadDir())
            if (!root.exists()) {
                root.mkdirs()
            }
            context.getMetadataDownloadDir() + File.separator + name
        } else {
            val root = File(context.getMetadataDir())
            if (!root.exists()) {
                root.mkdirs()
            }
            context.getMetadataDir() + File.separator + name
        }
        return FileUtil.openTempFile(path) { tempFile ->
            val stream = FileOutputStream(tempFile)
            try {
                stream.write(contents.toByteArray())
            } catch (e: Exception) {
                return@openTempFile false
            } finally {
                stream.close()
            }
            true
        }
    }

    fun getMetadataWithUpdate(context: Context): File? {
        val downloadedFile =
            getStoredFile(context, passio_metadata, context.getMetadataDownloadDir())
        val localFile = getStoredFile(context, passio_metadata, context.getMetadataDir())
        if (downloadedFile == null) {
            return localFile
        }

        val dstFile = FileUtil.copyFile(
            downloadedFile,
            context.getMetadataDir() + File.separator + downloadedFile.name
        ) ?: return null
        downloadedFile.delete()
        if (dstFile != localFile) {
            localFile?.delete()
        }
        return dstFile
    }

    fun getDatasetID(context: Context, useAssets: Boolean): String? {
        if (useAssets) {
            FileUtil.listAssetFiles(context.assets, "").forEach { filename ->
                if (filename.startsWith(passio_dataset.name)) {
                    return FileUtil.loadTextFileFromAssets(context.assets, filename)
                }
            }
            return null
        } else {
            val datasetFile = getStoredFile(context, passio_dataset) ?: return null
            return FileUtil.stringFromFile(datasetFile)
        }
    }
}