package ai.passio.passiosdk.core.network

import ai.passio.passiosdk.core.utils.FileUtil
import android.net.Uri
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.net.URL
import javax.net.ssl.HttpsURLConnection

internal class NetworkFileTask(
    urlString: String,
    headers: Map<String, String>,
    private val outPath: String,
) : NetworkTask<File?>(urlString, headers) {

    override fun executeTask(): File? {
        val url = URL(url)
        var connection: HttpsURLConnection? = null
        return try {
            connection = (url.openConnection() as? HttpsURLConnection)
            connection?.run {
                // Timeout for reading InputStream arbitrarily set to 3000ms.
                readTimeout = 20000
                // Timeout for connection.connect() arbitrarily set to 3000ms.
                connectTimeout = 3000
                // For this use case, set HTTP method to GET.
                requestMethod = "GET"
                headers.forEach { (k, v) ->
                    setRequestProperty(k, v)
                }
                // Already true by default but setting just in case; needs to be true since this request
                // is carrying an input (response) body.
                doInput = true
                // Open communications link (network traffic occurs here).
                if (enableLog) {
                    logRequest(this)
                }
                connect()
                if (enableLog) {
                    logResponse(this)
                }
                //publishProgress(CONNECT_SUCCESS)
                checkResponse(this)
                // Retrieve the response body as an InputStream.
                //publishProgress(GET_INPUT_STREAM_SUCCESS, 0)

                if (responseCode == 204) {
                    return null
                }

                inputStream?.let { stream ->
                    val rootPath = outPath.substring(0, outPath.indexOfLast { it == '/' })
                    val rootFolder = File(rootPath)
                    if (!rootFolder.exists()) {
                        rootFolder.mkdirs()
                    }

                    // Exception is caught on the NetworkService side
                    val endFile = FileUtil.openTempFile(outPath) { tmpFile ->
                        val fos = FileOutputStream(tmpFile)
                        val block = ByteArray(1024)
                        var nRead = 0
                        while ({ nRead = stream.read(block, 0, block.size); nRead }() != -1) {
                            fos.write(block, 0, nRead)
                        }
                        fos.flush()
                        fos.close()
                        true
                    }
                    endFile
                }
            } ?: throw IOException("Could not open http connection")
        } finally {
            // Close Stream and disconnect HTTPS connection.
            connection?.inputStream?.close()
            connection?.disconnect()
        }
    }
}