package ai.passio.passiosdk.core.network

import ai.passio.passiosdk.core.authentication.TokenService
import java.io.IOException
import java.text.ParseException
import java.util.concurrent.Callable
import java.util.concurrent.Executors

internal class NetworkService() {

    companion object {
        const val NO_INTERNET_CONNECTION = "noInternetConnection"
        private const val TOKEN_EXPIRED_MESSAGE = "token is expired"

        private var internalInstance: NetworkService? = null

        val instance: NetworkService
            get() {
                if (internalInstance == null) {
                    internalInstance = NetworkService()
                }
                return internalInstance!!
            }
    }

    private val executor = Executors.newCachedThreadPool()
    private val enableLog: Boolean = true
    private var proxyHeaders: Map<String, String>? = null

    fun setProxyHeaders(headers: Map<String, String>?) {
        proxyHeaders = headers
    }

    private fun <T> requestInternal(
        networkTask: NetworkTask<T>,
        callback: NetworkCallback<T>,
        request: () -> T
    ) {
        executor.execute {
            networkTask.enableLog = enableLog
            // Add proxy headers to the API call
            if (proxyHeaders != null) {
                val appendMap = networkTask.headers.toMutableMap()
                proxyHeaders?.forEach { (k, v) ->
                    appendMap[k] = v
                }
                networkTask.headers = appendMap
            }
            try {
                val response = request()
                callback.onSuccess(response)
            } catch (e: NetworkException) {
                if (e.code == 401 && e.message?.lowercase()
                        ?.contains(TOKEN_EXPIRED_MESSAGE) == true
                ) {
                    TokenService.getInstance().invalidateToken()
                    callback.onTokenExpired()
                } else {
                    callback.onFailure(e.code, "Code: ${e.code}, message: ${e.message}")
                }
            } catch (e: IOException) {
                callback.onFailure(-1, "Not found")
            } catch (e: ParseException) {
                callback.onFailure(-1, e.message ?: "")
            }
        }
    }

    fun <T> doRequest(
        networkTask: NetworkTask<T>,
        callback: NetworkCallback<T>
    ) {
        requestInternal(networkTask, callback) {
            networkTask.executeTask()
        }
    }

    fun <T> doRequestTrackTokens(
        networkTask: NetworkTask<T>,
        callback: NetworkCallback<T>,
        apiName: String
    ) {
        requestInternal(networkTask, callback) {
            networkTask.executeTaskTrackTokens(apiName)
        }
    }

    fun <T> syncRequest(task: NetworkTask<T>): T? {
        task.enableLog = enableLog
        val future = executor.submit(Callable {
            try {
                task.executeTask()
            } catch (e: Exception) {
                null
            }
        })
        return future.get()
    }

    fun <T> batchRequest(
        tasks: List<NetworkTask<T>>,
    ): List<T?> {
        val resultFutures = mutableListOf<Callable<T?>>()

        tasks.forEach { task ->
            resultFutures.add(Callable {
                try {
                    task.executeTask()
                } catch (e: Exception) {
                    null
                }
            })
        }

        val executedTasks = executor.invokeAll(resultFutures)
        return executedTasks.map { it.get() }
    }
}