package ai.passio.passiosdk.passiofood.data.measurement

import ai.passio.passiosdk.core.utils.PassioLog

/**
 * Class that represents a measurement of mass.
 *
 * @property unit defines the unit of mass.
 * @property value defines how much of [unit]s are represented by this instance.
 */
data class UnitMass(val unit: Unit = Grams, val value: Double = 0.0) {

    fun gramsValue(): Double = value * unit.converter.coefficient * 1000

    operator fun plus(add: UnitMass?): UnitMass {
        if (add == null) return this

        return if (unit == add.unit) {
            UnitMass(unit, value + add.value)
        } else {
            UnitMass(Grams, gramsValue() + add.gramsValue())
        }
    }

    operator fun times(mult: Double): UnitMass {
        return UnitMass(unit, (value * mult))
    }

    operator fun div(dividend: UnitMass): Double {
        return gramsValue() / dividend.gramsValue()
    }

    override fun toString(): String {
        return value.toString() + unit.symbol
    }
}

/**
 * Base class used to create units for mass, energy, etc.
 *
 * @property converter utility class that transforms the given unit
 *           to the base unit.
 * @property symbol the string that denominates the given unit
 */
open class Unit(val converter: Converter = Converter(), val symbol: String = "") {
    companion object {
        fun unitFromString(symbol: String): Unit? {
            return when (symbol.lowercase()) {
                Grams.symbol -> Grams
                Kilograms.symbol -> Kilograms
                Dekagrams.symbol -> Dekagrams
                Decigrams.symbol -> Decigrams
                Centigrams.symbol -> Centigrams
                Milligrams.symbol -> Milligrams
                Micrograms.symbol, "mcg" -> Micrograms
                Milliliters.symbol -> Milliliters
                KiloCalories.symbol -> KiloCalories
                KiloJoule.symbol -> KiloJoule
                "iu" -> Unit()
                else -> {
                    PassioLog.e("PassioUnit", "No known unit: $symbol")
                    null
                }
            }
        }
    }
}

data class Converter(val coefficient: Double = 1.0)

/**
 * Unit of mass that represents kilograms. The base unit of mass.
 */
object Kilograms : Unit(Converter(1.0), "kg")

/**
 * Unit of mass that represents dekagrams
 */
object Dekagrams : Unit(Converter(0.01), "dag")

/**
 * Unit of mass that represents grams.
 */
object Grams : Unit(Converter(0.001), "g") {
    const val unitName = "gram"
}

/**
 * Unit of mass that represents decigrams.
 */
object Decigrams : Unit(Converter(0.0001), "dg")

/**
 * Unit of mass that represents centigrams.
 */
object Centigrams : Unit(Converter(0.00001), "cg")

/**
 * Unit of mass that represents milligrams.
 */
object Milligrams : Unit(Converter(0.000001), "mg")

/**
 * Unit of mass that represents micrograms.
 */
object Micrograms : Unit(Converter(0.000000001), "ug")

/**
 * Unit of mass that represents milliliters. Equivalent to grams.
 */
object Milliliters : Unit(Converter(0.001), "ml")

object Liters : Unit(Converter(1.0), "l")

/**
 * Unit of energy that represents kiloCalories. Base unit of energy.
 */
object KiloCalories : Unit(Converter(1.0), "kcal")

object KiloJoule : Unit(Converter(0.239006), "kj")

object Ounce : Unit(Converter(0.035274), "oz")

/**
 * Class that represents a measurement of energy.
 *
 * @property unit defines the unit of energy.
 * @property value defines how much of [unit]s are represented by this instance.
 */
class UnitEnergy(val unit: Unit = KiloCalories, val value: Double = 0.0) {

    fun kcalValue(): Double = value * unit.converter.coefficient

    operator fun plus(add: UnitEnergy?): UnitEnergy {
        if (add == null) return this

        return if (unit == add.unit) {
            UnitEnergy(unit, value + add.value)
        } else {
            UnitEnergy(KiloCalories, kcalValue() + add.kcalValue())
        }
    }

    operator fun times(mult: Double): UnitEnergy {
        return UnitEnergy(unit, kcalValue() * mult)
    }

    override fun toString(): String {
        return value.toString() + unit.symbol
    }
}