package ai.passio.passiosdk.passiofood.recognition

import ai.passio.passiosdk.core.annotation.CameraThread
import ai.passio.passiosdk.core.annotation.PassioThread
import ai.passio.passiosdk.core.camera.PassioCameraProxy
import ai.passio.passiosdk.core.config.PassioMode
import ai.passio.passiosdk.core.config.PassioStatus
import ai.passio.passiosdk.core.config.SDKFileType
import ai.passio.passiosdk.core.utils.BitmapUtil
import ai.passio.passiosdk.core.utils.DimenUtil
import ai.passio.passiosdk.core.utils.ImageUtils
import ai.passio.passiosdk.core.utils.PassioLog
import ai.passio.passiosdk.passiofood.*
import ai.passio.passiosdk.passiofood.file.PassioFoodFileManager
import ai.passio.passiosdk.passiofood.metadata.MetadataManager
import ai.passio.passiosdk.passiofood.mlkit.BarcodeDetectorFactory
import ai.passio.passiosdk.passiofood.mlkit.BarcodeProxy
import ai.passio.passiosdk.passiofood.mlkit.TextRecognitionFactory
import ai.passio.passiosdk.passiofood.nutritionfacts.NutritionFactsReader
import ai.passio.passiosdk.passiofood.nutritionfacts.PassioNutritionFacts
import ai.passio.passiosdk.passiofood.time.VotingFrameAveraging
import ai.passio.passiosdk.passiofood.utils.AggregateHelper
import ai.passio.passiosdk.passiofood.voting.*
import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Matrix
import android.graphics.RectF
import android.util.Log
import androidx.camera.core.ImageProxy
import com.google.mlkit.vision.common.InputImage
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.atomic.AtomicReference

@Suppress("PrivatePropertyName")
internal open class PassioRecognizer(
    private val cameraProxy: PassioCameraProxy,
    private val mainExecutor: Executor,
    private val fileManager: PassioFoodFileManager,
    private val labelManager: MetadataManager,
) : PassioCameraProxy.PassioCameraListener {

    companion object {
        internal const val TF_OD_API_INPUT_SIZE = 360
        internal const val TF_HNN_API_INPUT_SIZE = 260
        internal const val TF_YOLO_INPUT_SIZE = 320
    }

    private var isReady = AtomicBoolean(false)
    private val recognitionQueue: Executor by lazy { Executors.newSingleThreadExecutor() }

    private var frameWidth = 0
    private var frameHeight = 0
    private var orientation = 0

    private var barcodeScanningListener = AtomicReference<BarcodeScanningListener?>()

    private val resourcesAnalyzingImage = AtomicInteger(0)

    private lateinit var modelHolder: PassioModelHolder

    @PassioThread
    fun initializeFromCompressedExternalModels(
        context: Context,
        version: Int
    ): List<String> {
        try {
            modelHolder = PassioExternalCompressedModelHolder(labelManager, fileManager)
            return modelHolder.initializeModels(context, version)
        } catch (e: Exception) {
            return emptyList()
        }
    }

    @PassioThread
    fun initializeFromSecuredExternalModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioExternalSecuredModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    @PassioThread
    fun initializeFromExternalModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioExternalModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    @PassioThread
    fun initializeFromAssetModels(
        context: Context,
        version: Int,
        assetFiles: List<String>
    ): List<String> {
        return if (assetFiles[0].endsWith("passiosecure2")) {
            initializeFromAssetCompressedModels(
                context,
                version
            )
        } else if (assetFiles[0].endsWith("passiosecure")) {
            initializeFromAssetSecuredModels(
                context,
                version
            )
        } else {
            initializeFromAssetModels(
                context,
                version
            )
        }
    }

    @PassioThread
    fun initializeRemoteOnly(context: Context) {
        modelHolder = object : PassioModelHolder(labelManager, fileManager) {

            override fun initializeModels(context: Context, version: Int): List<String> {
                barcodeDetector = BarcodeDetectorFactory.create()
                ocrDetector = TextRecognitionFactory.getTextRecognizer()
                return emptyList()
            }

            override fun initializeModel(
                context: Context,
                fileType: SDKFileType,
                version: Int
            ): Boolean {
                throw IllegalAccessException("Should not be invoked")
            }

            override fun getExtension(): String = ""
        }
        modelHolder.initializeModels(context, 0)
    }

    private fun initializeFromAssetCompressedModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioAssetCompressedSecuredModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    private fun initializeFromAssetSecuredModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioAssetSecuredModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    private fun initializeFromAssetModels(
        context: Context,
        version: Int
    ): List<String> {
        modelHolder = PassioAssetModelHolder(labelManager, fileManager)
        return modelHolder.initializeModels(context, version)
    }

    fun onPassioStatus(status: PassioStatus) {
        isReady.set(status.mode == PassioMode.IS_READY_FOR_DETECTION)
    }

    fun runBarcodeDetection(
        bitmap: Bitmap,
        onDetectionCompleted: (candidates: List<BarcodeCandidate>) -> Unit
    ) {
        if (!isReady.get()) {
            onDetectionCompleted(listOf())
            return
        }

        recognitionQueue.execute {
            runBarcodeDetectionInternal(bitmap) { barcodes ->
                // Prune barcodes
                val filtered = barcodes.filter { it.boundingBox != null }

                val transformedResult = filtered.map {
                    val relativeBoundingBox = RectF(
                        it.boundingBox!!.left.toFloat() / bitmap.width,
                        it.boundingBox!!.top.toFloat() / bitmap.height,
                        it.boundingBox!!.right.toFloat() / bitmap.width,
                        it.boundingBox!!.bottom.toFloat() / bitmap.height
                    )
                    BarcodeCandidate(it.value, relativeBoundingBox)
                }
                onDetectionCompleted(transformedResult)
            }
        }
    }

    private fun runBarcodeDetectionInternal(
        inputImage: InputImage,
        onResult: (result: List<BarcodeProxy>) -> Unit
    ) {
        modelHolder.barcodeDetector!!.process(inputImage,
            { barcodes -> onResult(barcodes) },
            { onResult(listOf()) })
    }

    @SuppressLint("UnsafeOptInUsageError")
    protected fun runBarcodeDetectionInternal(
        imageProxy: ImageProxy,
        onResult: (result: List<BarcodeProxy>) -> Unit
    ) {
        try {
            val image = InputImage.fromMediaImage(
                imageProxy.image!!,
                imageProxy.imageInfo.rotationDegrees
            )
            runBarcodeDetectionInternal(image, onResult)
        } catch (e: Exception) {
            Log.e(PassioSDKImpl::class.java.simpleName, "${e.message}")
            onResult(listOf())
        }
    }

    private fun runBarcodeDetectionInternal(
        bitmap: Bitmap,
        onResult: (result: List<BarcodeProxy>) -> Unit
    ) {
        try {
            val image = InputImage.fromBitmap(bitmap, 0)
            runBarcodeDetectionInternal(image, onResult)
        } catch (e: java.lang.IllegalStateException) {
            Log.e(PassioSDKImpl::class.java.simpleName, "${e.message}")
            onResult(listOf())
        }
    }

    override fun onFrameSize(
        frameWidth: Int,
        frameHeight: Int,
        previewWidth: Int,
        previewHeight: Int,
        orientation: Int
    ) {
        this.frameWidth = frameWidth
        this.frameHeight = frameHeight
        this.orientation = orientation
    }

    @CameraThread
    override fun analyzeImage(imageProxy: ImageProxy) {
        if (!isReady.get()) {
            imageProxy.close()
            return
        }

        if (resourcesAnalyzingImage.get() != 0) {
            imageProxy.close()
            return
        }

        if (imageProxy.width != frameWidth && imageProxy.height != frameHeight) {
            imageProxy.close()
            return
        }

        val currentListener = barcodeScanningListener.get()
        if (currentListener == null) {
            imageProxy.close()
            return
        }

        runBarcodeDetectionInternal(
            imageProxy = imageProxy,
        ) { results ->
            // Prune barcodes
            val filtered = results.filter { it.boundingBox != null }

            val transformedResult = filtered.map {
                val relativeBoundingBox = RectF(
                    it.boundingBox!!.left.toFloat() / imageProxy.width,
                    it.boundingBox!!.top.toFloat() / imageProxy.height,
                    it.boundingBox!!.right.toFloat() / imageProxy.width,
                    it.boundingBox!!.bottom.toFloat() / imageProxy.height
                )
                BarcodeCandidate(it.value, relativeBoundingBox)
            }

            mainExecutor.execute {
                if (currentListener == barcodeScanningListener.get()) {
                    currentListener.onBarcodeResult(
                        candidates = transformedResult
                    )
                }
            }

            imageProxy.close()
        }

    }

    fun startBarcodeScanning(listener: BarcodeScanningListener) {
        this.barcodeScanningListener.set(listener)
    }

    fun stopBarcodeScanning() {
        this.barcodeScanningListener.set(null)
    }

    protected fun Boolean.toInt() = if (this) 1 else 0

    fun runJustOCR(bitmap: Bitmap, callback: (result: String) -> Unit) {
        val inputImage = InputImage.fromBitmap(bitmap, 0)
        modelHolder.ocrDetector!!.process(inputImage).addOnSuccessListener {
            callback(it.text)
        }.addOnFailureListener {
            callback("")
        }
    }
}