package ai.vyro.imagine.sdk.impl

import ai.vyro.imagine.sdk.Imagine
import ai.vyro.imagine.sdk.features.AspectRatio
import ai.vyro.imagine.sdk.features.generations.GenerationsHandler
import ai.vyro.imagine.sdk.features.generations.GenerationsStyle
import ai.vyro.imagine.sdk.features.inpaint.InPaintHandler
import ai.vyro.imagine.sdk.features.inpaint.InPaintStyle
import ai.vyro.imagine.sdk.features.remix.ImageRemixControl
import ai.vyro.imagine.sdk.features.remix.ImageRemixStyle
import ai.vyro.imagine.sdk.features.remix.RemixHandler
import ai.vyro.imagine.sdk.features.upscale.UpscaleHandler
import ai.vyro.imagine.sdk.features.upscale.UpscaleStyle
import ai.vyro.imagine.sdk.features.variations.VariationsHandler
import ai.vyro.imagine.sdk.models.Response
import ai.vyro.imagine.sdk.models.response.Image

internal class ImagineImpl(
    private val generationsHandler: GenerationsHandler,
    private val remixHandler: RemixHandler,
    private val upscaleHandler: UpscaleHandler,
    private val variationsHandler: VariationsHandler,
    private val inpaintHandler: InPaintHandler,
) : Imagine {
    override suspend fun generations(
        prompt: String,
        style: GenerationsStyle,
        aspectRatio: AspectRatio?,
        negPrompt: String?,
        cfg: Float?,
        seed: Int?,
        steps: Int?,
        highResolution: Boolean
    ): Response<Image> {
        return generationsHandler(
            prompt = prompt,
            style = style.id,
            aspectRatio = aspectRatio?.ratio,
            negPrompt = negPrompt,
            cfg = cfg,
            seed = seed,
            steps = steps,
            highResResults = if (highResolution) 1 else 0,
        )
    }

    override suspend fun remix(
        image: ByteArray,
        prompt: String,
        style: ImageRemixStyle,
        control: ImageRemixControl,
        negPrompt: String?,
        cfg: Float?,
        seed: Int?,
        steps: Int?,
        strength: Int?
    ): Response<Image> {
        return remixHandler(
            imageBytes = image,
            prompt = prompt,
            style = style.id,
            control = control.id,
            negPrompt = negPrompt,
            cfg = cfg,
            seed = seed,
            steps = steps,
            strength = strength,
        )
    }

    override suspend fun upscale(
        image: ByteArray,
        style: UpscaleStyle
    ): Response<Image> {
        return upscaleHandler(
            imageBytes = image,
            modelVersion = style.id,
        )
    }

    override suspend fun variations(
        image: ByteArray,
        prompt: String,
        style: GenerationsStyle,
        negPrompt: String?,
        cfg: Float?,
        seed: Int?,
        steps: Int?,
        strength: Int?
    ): Response<Image> {
        return variationsHandler(
            imageBytes = image,
            prompt = prompt,
            style = style.id,
            negPrompt = negPrompt,
            cfg = cfg,
            seed = seed,
            steps = steps,
            strength = strength,
        )
    }

    override suspend fun inpaint(
        image: ByteArray,
        mask: ByteArray,
        prompt: String,
        style: InPaintStyle
    ): Response<Image> {
        return inpaintHandler(
            imageBytes = image,
            maskBytes = mask,
            prompt = prompt,
            modelVersion = style.id,
        )
    }
}
