package com.alan.alansdk;

import android.app.Activity;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import androidx.annotation.Keep;
import androidx.annotation.Nullable;

import com.alan.alansdk.alanbase.AudioConfig;
import com.alan.alansdk.alanbase.ConnectionState;
import com.alan.alansdk.alanbase.DialogState;
import com.alan.alansdk.alanbase.SdkParams;
import com.alan.alansdk.alanbase.recorder.AlanRecorder;
import com.alan.alansdk.alanbase.speaker.AlanARMSpeaker;
import com.alan.alansdk.alanbase.speaker.AlanOboeSpeaker;
import com.alan.alansdk.alanbase.speaker.AlanSpeaker;
import com.alan.alansdk.events.EventCommand;
import com.alan.alansdk.events.EventDialogId;
import com.alan.alansdk.events.EventOptions;
import com.alan.alansdk.events.EventParsed;
import com.alan.alansdk.events.EventRecognised;
import com.alan.alansdk.events.EventText;
import com.alan.alansdk.logging.AlanLogger;
import com.alan.alansdk.prefs.AlanPrefs;
import com.alan.alansdk.qr.QRScanActivity;
import com.alan.alansdk.screenshot.ScreenShooter;
import com.alan.alansdk.screenshot.ScreenshotAddon;
import com.alan.alansdk.screenshot.UploadThread;
import com.alan.alansdk.wakeword.DownloadListener;
import com.alan.alansdk.wakeword.FileUtils;
import com.alan.alansdk.wakeword.ModelDownloader;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

public class Alan implements AlanSDK {

    static {
        System.loadLibrary("native-lib");
    }

    public static String PLATFORM_SUFFIX = "";
    public static String PLATFORM_VERSION_SUFFIX = "";

    public static boolean QR_EVENT_BUS_ENABLED = false;

    private static final String STAGE_ENDPOINT = "wss://studio.alan-stage.app";
    private static final String PROD_ENDPOINT = "wss://studio.alan.app";

    private static int REQUEST_CODE_QR = 65056;

    private volatile static Alan INSTANCE;
    private AlanSpeaker speaker;
    private AlanRecorder recorder = new AlanRecorder();
    private boolean isInited = false;
    private final String server = BuildConfig.API_ENDPOINT;

    public AlanState alanState;
    private DialogState state = DialogState.IDLE;
    private ConnectionState connectionState = ConnectionState.IDLE;

    private static String modelPath;
    private boolean isWakeWordEnabled;

    ScreenShooter screenShooter = new ScreenShooter();

    private static Set<WeakReference<AlanCallback>> callbacks = new LinkedHashSet<>();

    private Handler mainHandler = new Handler(Looper.getMainLooper());
    private AlanPrefs prefs;

    private static String lastUsedServer;
    private static String lastUsedProject;
    private static String lastUsedDialog;
    private static String lastUsedAuth;

    private Context context;

    /// Flag indicates that "buttonReady" is sent
    private boolean hasButtonReadyEvent;
    /// Flag indicates that "playAllowed" is sent
    private boolean hasPlayAllowedEvent;

    private Alan(Context context) {
        this.context = context;
        AlanLogger.init(context);
        prefs = new AlanPrefs(context);
        AudioConfig.determineSupportedSampleRate(context);

        speaker = new AlanARMSpeaker();

        alanState = AlanState.CONNECTING;
    }

    private boolean isArm32(Context context) {
        String arch = System.getProperty("os.arch");
        String abilist = System.getProperty("ro.product.cpu.abilist");
        boolean is64Bit = (abilist != null && abilist.contains("64"));
        AlanLogger.d("Arch is: " + arch);
        AlanLogger.d("Abilist is: " + abilist);
        return arch == null || (arch.contains("armeabi") && !is64Bit);
    }

    public static synchronized Alan getInstance(Context context) {
        if (INSTANCE == null) {
            synchronized (Alan.class) {
                if (INSTANCE == null) {
                    INSTANCE = new Alan(context);
                }
            }
        }
        return INSTANCE;
    }

    public void stop() {
        AlanLogger.i("Stopping Alan");
        isInited = false;
        deactivate();
        speaker.release();
        stopRecording();
        stopNative();
    }

    private boolean isRecordingBlocked = false;

    @Deprecated
    public void record() {
        if (!isRecordingBlocked) {
            recorder.startRecording();
        }
    }

    public void stopRecording() {
        recorder.stopRecording();
    }

    public void blockRecording(boolean shouldBlock) {
        isRecordingBlocked = shouldBlock;
        if (shouldBlock) {
            stopRecording();
        }
    }

    @Deprecated
    public void speak() {
        speaker.startPlaying();
    }

    private void stopPlaying() {
        speaker.stopPlaying();
    }

    private void handleOptions(EventOptions options) {
        boolean needRestart = false;

        prefs.setKeepMicrophoneInBackground(options.getAndroidSettings().keepMicrophoneInBackground);
        prefs.setWakewordInBackground(options.getAndroidSettings().wakeWordInBackground);

        if (prefs.wakewordInBackground()) {
            AlanLogger.i("Starting service");
            ComponentName componentName = null;
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
                componentName = context.getApplicationContext().
                        startForegroundService(new Intent(context.getApplicationContext(), AlanService.class));
            } else {
                componentName = context.getApplicationContext().
                        startService(new Intent(context.getApplicationContext(), AlanService.class));
            }
            if (componentName != null) {
                AlanLogger.d("Service started with name: " + componentName.flattenToShortString());
            } else {
                AlanLogger.w("Service did not started");
            }
        } else {
            AlanLogger.i("Stopping service");
            context.stopService(new Intent(context, AlanService.class));
        }

        if (prefs.isWakeWordEnabled() != options.getAndroidSettings().isWakeWordEnabled()) {
            prefs.setWakeWordState(options.getAndroidSettings().isWakeWordEnabled());
            this.isWakeWordEnabled = options.getAndroidSettings().isWakeWordEnabled();
            needRestart = true;

            // Temp disable wakeword for Android 7.x.x
            if (Build.VERSION.SDK_INT == Build.VERSION_CODES.N || Build.VERSION.SDK_INT == Build.VERSION_CODES.N_MR1) {
                this.isWakeWordEnabled = false;
                needRestart = false;
            }
        }

        if (prefs.isButtonHidden() != options.getAndroidSettings().isButtonHidden) {
            prefs.setButtonHidden(options.getAndroidSettings().isButtonHidden);
            needRestart = true;
        }

        if (!prefs.getTimeout().equals(options.getAndroidSettings().getTimeout())) {
            prefs.setTimeout(options.getAndroidSettings().getTimeout());
            needRestart = true;
        }

        if (prefs.hideS2TPanel() != options.getAndroidSettings().shouldHideS2TPanel()) {
            prefs.setHideS2TPanel(options.getAndroidSettings().shouldHideS2TPanel());
            needRestart = true;
        }

        if (prefs.isScreenshotsEnabled() != options.getAndroidSettings().isScreenShotsEnabled()) {
            prefs.setScreenshotsEnabled(options.getAndroidSettings().isScreenShotsEnabled());
            needRestart = true;
        }
        
        final String oldModelUrl = Utils.extractFileNameFromUrl(prefs.getWakeWordModelUrl());
        final String newModelUrl = options.getWakeWordModelName();
        if (!oldModelUrl.equalsIgnoreCase(newModelUrl)) {
            new ModelDownloader().download(context, new DownloadListener() {
                @Override
                public void onDownloaded(String url) {
                    prefs.setWakeWordModelUrl(url);
                    modelPath = Utils.getWakeWordModelFile(context, Utils.extractFileNameFromUrl(url)).getAbsolutePath();
                    restart();
                }

                @Override
                public void onDownloadFailed() {
                    AlanLogger.e("Failed to download new model");
                }
            }, options.wakeWordPath);
        }

        if (needRestart) {
            AlanLogger.i("Options changed -- need restart");
            restart();
        }
    }

    private void handleRecognizedEvent(EventRecognised event) {
        if (AlanService.isRunning) {
            context.startService(AlanService.getSpeechIntent(context, event.getText()));
        }
    }

    private void handleTextEvent(EventText event) {
        if (AlanService.isRunning) {
            context.startService(AlanService.getSpeechIntent(context, event.getText()));
        }
    }

    public void takeScreenshot(@Nullable Activity activity, String screenShotUrl) {
        if (prefs.isScreenshotsEnabled()) {
            AlanLogger.i("Taking screenshot");
            String image = screenShooter.shot(activity, screenShotUrl);
            if (image != null) {
                AlanLogger.i("Saved new screenshot: " + image);
            } else {
                AlanLogger.e("Cannot take screenshot");
            }
        } else {
            AlanLogger.w("Screenshots are disabled!");
        }
    }

    public boolean isHintPanelHidden() {
        return prefs.hideS2TPanel();
    }

    public boolean isButtonHidden() {
        return prefs.isButtonHidden();
    }

    public String getCustomLogo() {
        return prefs.getCustomLogo();
    }

    public void setCustomLogo(String customLogo) {
        prefs.setCustomLogo(customLogo);
    }

    public Map<String,String> getCustomLogos() {
        return prefs.getCustomLogos();
    }

    public void setCustomLogos(Map<String,String> customLogos) {
        prefs.setCustomLogos(customLogos);
    }

    private void handleConnectionState(ConnectionState connectionState) {
        if (connectionState == ConnectionState.CONNECTED && isWakeWordEnabled) {
            record();
            speak();
        }
    }

    private void handleDialogState(DialogState dialogState) {
        if (dialogState == DialogState.IDLE && isWakeWordEnabled) {
            record();
        }
    }

    private void uploadPendingScreenshots() {
        UploadThread uploader = new UploadThread(context,
                lastUsedServer);
        uploader.start();
    }

    public boolean restart() {
        return initWithConfig(null);
    }

    /* Called from native*/
    @Keep
    private void onConnectStateChanged(final int connectState) {
        final ConnectionState newState = ConnectionState.fromInt(connectState);
        connectionState = newState;
        AlanLogger.i("Connection state: " + newState.name());
        handleConnectionState(newState);
        switch (newState) {
            case CONNECTING:
            case AUTHORIZING:
                alanState = AlanState.CONNECTING;
                break;
            case CONNECTED:
                alanState = AlanState.ONLINE;
                sendButtonReadyEvent();
                break;
            case IDLE:
                break;
            case CLOSED: alanState = AlanState.OFFLINE;
                break;
        }
        mainHandler.post(new Runnable() {
            @Override
            public void run() {
                for (WeakReference<AlanCallback> callbackRef : callbacks) {
                    AlanCallback alanCallback = callbackRef.get();
                    if (alanCallback != null) {
                        alanCallback.onAlanStateChanged(alanState);
                        alanCallback.onButtonState(alanState);
                    }
                }
            }
        });
    }

    //================================================================================
    // Client Events
    //================================================================================

    private void sendPlayAllowedEvent() {
        if (!hasPlayAllowedEvent) {
            hasPlayAllowedEvent = true;
            try {
                JSONObject params = new JSONObject();
                params.put("playAllowed", true);
                sendClientEvent(params.toString());
                AlanLogger.w("clientEvent \"playAllowed\" is sent");
            } catch (JSONException e) {
                AlanLogger.e("sendPlayAllowedEvent - failed to create json");
            }
        }
    }

    private void sendButtonReadyEvent() {
        if (!hasButtonReadyEvent) {
            hasButtonReadyEvent = true;
            try {
                JSONObject params = new JSONObject();
                params.put("buttonReady", true);
                sendClientEvent(params.toString());
                AlanLogger.w("clientEvent \"buttonReady\" is sent");
            } catch (JSONException e) {
                AlanLogger.e("sendButtonReadyEvent - failed to create json");
            }
        }
    }

    public void sendClientEvent(final String jsonParams) {
        callProjectApiInternal("clientEvent", jsonParams, null, false);
    }

    /* Called from native*/
    @Keep
    private void onDialogStateChanged(final int dialogState) {
        final DialogState newState = DialogState.fromInt(dialogState);
        if (newState == DialogState.IDLE && newState != state ) {
            uploadPendingScreenshots();
        }
        state = newState;
        AlanLogger.i("DialogState: " + newState.name());

        switch (newState) {
            case IDLE:
                alanState = AlanState.ONLINE;
                break;
            case LISTEN:
                record();
                alanState = AlanState.LISTEN;
                if (AlanService.isRunning) {
                    context.startService(AlanService.getSpeechIntent(context, null));
                }
                break;
            case PROCESS:
                speak();
                alanState = AlanState.PROCESS;
                break;
            case REPLY:
                alanState = AlanState.REPLY;
                speak();
                break;
        }
        mainHandler.post(new Runnable() {
            @Override
            public void run() {
                for (WeakReference<AlanCallback> callbackRef : callbacks) {
                    AlanCallback alanCallback = callbackRef.get();
                    if (alanCallback != null) {
                        alanCallback.onAlanStateChanged(alanState);
                        alanCallback.onButtonState(alanState);
                    }
                }
            }
        });
    }

    /* Called from native*/
    @Keep
    private void onEvent(final String event, final String payload) {
        AlanLogger.event("New event: \"" + event + "\", with payload: " + payload);
        mainHandler.post(new Runnable() {
            @Override
            public void run() {
                switch (event) {
                    case EventRecognised.EVENT_NAME:
                        EventRecognised eventRecognised = EventRecognised.fromJson(payload);
                        handleRecognizedEvent(eventRecognised);
                        for (WeakReference<AlanCallback> callbackRef : callbacks) {
                            AlanCallback alanCallback = callbackRef.get();
                            if (alanCallback != null) {
                                alanCallback.onRecognizedEvent(eventRecognised);

                                JsonObject e = new Gson().fromJson(payload, JsonObject.class);
                                e.addProperty("name", "recognized");
                                String p = e.toString();
                                alanCallback.onEvent(p);
                            }
                        }
                        for (WeakReference<AlanCallback> callbackRef : callbacks) {
                            alanState = AlanState.PROCESS;
                            AlanCallback alanCallback = callbackRef.get();
                            if (alanCallback != null) {
                                alanCallback.onButtonState(alanState);
                            }
                        }
                        break;
                    case EventText.EVENT_NAME:
                        EventText eventText = EventText.fromJson(payload);
                        handleTextEvent(eventText);
                        for (WeakReference<AlanCallback> callbackRef : callbacks) {
                            AlanCallback alanCallback = callbackRef.get();
                            if (alanCallback != null) {
                                alanCallback.onTextEvent(eventText);

                                JsonObject e = new Gson().fromJson(payload, JsonObject.class);
                                e.addProperty("name", "text");
                                String p = e.toString();
                                alanCallback.onEvent(p);
                            }
                        }
                        break;
                    case EventOptions.EVENT_NAME:
                        EventOptions eventOptions = EventOptions.fromJson(payload);
                        handleOptions(eventOptions);
                        for (WeakReference<AlanCallback> callbackRef : callbacks) {
                            AlanCallback alanCallback = callbackRef.get();
                            if (alanCallback != null) {
                                try {
                                    alanCallback.onOptionsReceived(eventOptions);
                                } catch (Exception e) {
                                    AlanLogger.e(e);
                                }
                            }
                        }
                        break;
                    case EventCommand.EVENT_NAME:
                        for (WeakReference<AlanCallback> callbackRef : callbacks) {
                            AlanCallback alanCallback = callbackRef.get();
                            if (alanCallback != null) {
                                alanCallback.onCommandReceived(EventCommand.fromPayload(payload));

                                alanCallback.onCommand(EventCommand.fromPayload(payload));
                            }
                        }
                        break;
                    case EventParsed.EVENT_NAME:
                        for (WeakReference<AlanCallback> callbackRef : callbacks) {
                            AlanCallback alanCallback = callbackRef.get();
                            if (alanCallback != null) {
                                alanCallback.onParsedEvent(EventParsed.fromJson(payload));

                                JsonObject e = new Gson().fromJson(payload, JsonObject.class);
                                e.addProperty("name", "parsed");
                                String p = e.toString();
                                alanCallback.onEvent(p);
                            }
                        }
                        break;
                    case EventDialogId.EVENT_NAME:
                        EventDialogId eventDialogId = EventDialogId.fromJson(payload);
                        lastUsedDialog = eventDialogId.getDialogId();
                        break;
                    default:
                        for (WeakReference<AlanCallback> callbackRef : callbacks) {
                            AlanCallback alanCallback = callbackRef.get();
                            if (alanCallback != null) {
                                alanCallback.onEvent(event, payload);
                            }
                        }
                }
            }
        });
    }

    /* Called from native*/
    @Keep
    private void onError(final String error) {
        AlanLogger.e("Native error: " + error);
        mainHandler.post(new Runnable() {
            @Override
            public void run() {
                for (WeakReference<AlanCallback> callbackRef : callbacks) {
                    AlanCallback alanCallback = callbackRef.get();
                    if (alanCallback != null) {
                        alanCallback.onError(error);
                    }
                }
            }
        });
    }

    @Deprecated
    /**
     * Use `initWithConfig` method instead
     */
    public boolean init(@Nullable String server, String projectId, String dialogId, String authParams) {
        AlanConfig config = AlanConfig.builder()
                .setServer(server)
                .setProjectId(projectId)
                .setDialogId(dialogId)
                .setDataObject(authParams)
                .build();
        return initWithConfig(config);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method instead
     */
    public boolean init(@Nullable String server, String projectId, String dialogId) {
        return init(this.server, projectId, dialogId, null);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method instead
     */
    public boolean init(String projectId, String dialogId) {
        return init(server, projectId, dialogId, null);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method instead
     */
    public boolean init(String projectId) {
        return init(server, projectId, null, null);
    }

    public boolean initWithConfig(@Nullable AlanConfig config) {

        hasButtonReadyEvent = false;
        hasPlayAllowedEvent = false;

        stop();

        state = DialogState.IDLE;

        if (config != null) {
            lastUsedServer = config.getServer();
            lastUsedProject = config.getProjectId();
            lastUsedDialog = config.getDialogId();
            lastUsedAuth = config.getDataObject();
        }

        AlanLogger.i("Connecting to the " + lastUsedServer +
                ", with projectId: " + lastUsedProject +
                ", dialogId: " + lastUsedDialog +
                "\nAuthJson: " + lastUsedAuth);

        if (lastUsedProject == null) {
            AlanLogger.e("Project id is null! Aborting connection");
            isInited = false;
            return isInited;
        }

        prefs.updatePath(this.context, lastUsedServer, lastUsedProject);

        modelPath = prefs.getWakeWordModelUrl();

        if (modelPath == null || modelPath.isEmpty()) {
            modelPath = FileUtils.copyTfModel(context);
            prefs.setWakeWordModelUrl(modelPath);
        }

        this.isWakeWordEnabled = prefs.isWakeWordEnabled();

        String model = isWakeWordEnabled ? modelPath : null;

        // Temp disable wakeword for Android 7.x.x
        if (Build.VERSION.SDK_INT == Build.VERSION_CODES.N || Build.VERSION.SDK_INT == Build.VERSION_CODES.N_MR1) {
            model = null;
        }

        String sdkVersion = BuildConfig.VERSION_NAME;
        if (PLATFORM_VERSION_SUFFIX != null &&
            !PLATFORM_VERSION_SUFFIX.isEmpty()
        ) {
            sdkVersion += ":" + PLATFORM_VERSION_SUFFIX;
        }
        String platform = "android";
        if (PLATFORM_SUFFIX != null && !PLATFORM_SUFFIX.isEmpty()
        ) {
            platform += ":" + PLATFORM_SUFFIX;
        }

        isInited = initNative(lastUsedServer,
                lastUsedProject,
                lastUsedDialog,
                lastUsedAuth,
                model,
                prefs.getUUID(),
                sdkVersion,
                platform,
                context.getPackageName(),
                AudioConfig.SAMPLE_RATE
                );

        if (isInited) {
            sendPlayAllowedEvent();
        }

        return isInited;
    }

    public static void enableLogging(boolean shouldLog) {
        AlanLogger.SHOULD_LOG = shouldLog;
    }

    @Override
    public void setVisualState(String visualState) {
        if (!isInited()) {
            throw new IllegalStateException("Init Alan first!");
        }
        setParamsNative(visualState);
    }

    @Deprecated
    /*
     * User setVisualState instead
     */
    public void setVisuals(String visuals) {
        setVisualState(visuals);
    }

    public boolean isInited() {
        return isInited;
    }

    public String getVersion() {
        String sdkVersion = "SDK version: " + BuildConfig.VERSION_NAME;
        String coreVersion = "AlanBase: " + getVersionNative();
        String wakeWordName = "WakeWord model: " + Utils.extractFileNameFromUrl(prefs.getWakeWordModelUrl());
        return sdkVersion + "\n" + coreVersion + "\n" + wakeWordName;
    }

    @Override
    public void activate() {
        if (!isInited()) {
            AlanLogger.w("Performing stop when sdk is not inited yet");
            return;
        }
        if (connectionState != ConnectionState.CONNECTED) {
            AlanLogger.i("Alan is not connected to backend");
            return;
        }
        turn(true);
        record();
        speak();
    }

    @Deprecated
    /*
     * User activate() instead
     */
    public void turnOn() {
        activate();
    }

    @Override
    public void deactivate() {
        if (!isInited()) {
            AlanLogger.w("Performing stop when sdk is not inited yet");
            return;
        }
        if (!isWakeWordEnabled) {
            stopRecording();
        }
        stopPlaying();
        if (connectionState != ConnectionState.CONNECTED) {
            AlanLogger.i("Alan is not connected to backend");
            return;
        }
        turn(false);
    }

    @Override
    public boolean isActive() {
        return alanState == AlanState.LISTEN ||
               alanState == AlanState.PROCESS ||
               alanState == AlanState.REPLY;
    }

    @Deprecated
    /*
     * User deactivate() instead
     */
    public void turnOff() {
        deactivate();
    }

    public void toggle() {
        if (!isInited()) {
            AlanLogger.i("Alan is not inited!");
            return;
        }

        if (connectionState != ConnectionState.CONNECTED) {
            AlanLogger.i("Alan is not connected to backend");
            return;
        }

        if (state != DialogState.IDLE) {
            deactivate();
        } else {
            activate();
        }
    }

    public void registerCallback(AlanCallback callback) {
        callbacks.add(new WeakReference<>(callback));
    }

    public void unregisterCallback(AlanCallback callback) {
        Iterator<WeakReference<AlanCallback>> iterator = callbacks.iterator();
        while (iterator.hasNext()) {
            WeakReference<AlanCallback> callbackWeakReference = iterator.next();
            if (callbackWeakReference.get().equals(callback)) {
                iterator.remove();
                break;
            }
        }
    }

    public void clearCallbacks() {
        callbacks.clear();
    }

    public DialogState getDialogState() {
        return state;
    }

    public ConnectionState getConnectionState() {
        return connectionState;
    }

    @Override
    public void playText(String text) {
        callProjectApiInternal("play", "{\"text\":\"" + text + "\"}", null, false);
    }

    @Override
    public void playText(String text, @Nullable ScriptMethodCallback callback) {
        callProjectApiInternal("play", "{\"text\":\"" + text + "\"}", callback, false);
    }

    @Override
    public void playCommand(String data, @Nullable ScriptMethodCallback callback) {
        callProjectApiInternal("play", "{\"data\":{\"data\":" + data + "}}", callback, false);
    }

    @Deprecated
    /*
     * User playCommand instead
     */
    public void playData(String data, @Nullable ScriptMethodCallback callback) {
        callProjectApiInternal("play", "{\"data\":" + data + "}", callback, false);
    }

    @Override
    public void callProjectApi(String method, String jsonParams) {
        callProjectApi(method, jsonParams, null);
    }

    @Override
    public void callProjectApi(final String method,
                               final String jsonParams,
                               @Nullable final ScriptMethodCallback callback) {
        if (method.startsWith("script::")) {
            callProjectApiInternal(method, jsonParams, callback, false);
        } else {
            callProjectApiInternal("script::" + method, jsonParams, callback, false);
        }
    }

    private void callProjectApiInternal(final String method,
                                        final String jsonParams,
                                        @Nullable final ScriptMethodCallback callback,
                                        final boolean shouldActivate) {
        if (!isInited()) {
            AlanLogger.i("Alan is not inited!");
            return;
        }
        if (alanState == AlanState.CONNECTING || alanState == AlanState.OFFLINE || alanState == AlanState.UNKNOWN) {
            AlanLogger.d("Skipping action. " + method + ", Alan is not connected");
            return;
        }
        String params = jsonParams;
        if (params == null || params.isEmpty()) {
            params = "{}";
        }
        if (shouldActivate) {
            activate();
        }
        final String finalParams = params;
        mainHandler.postDelayed(new Runnable() {
            @Override
            public void run() {
                callScript(method, finalParams, callback);
            }
        }, 30);
    }

    @Deprecated
    /*
     * Use callProjectApi function instead
     */
    public void call(String method, String jsonParams) {
        callProjectApi(method, jsonParams, null);
    }

    @Deprecated
    /*
     * Use callProjectApi function instead
     */
    public void call(final String method, final String jsonParams, @Nullable final ScriptMethodCallback callback) {
        callProjectApi(method, jsonParams, callback);
    }


    public static void openQRScanner(Activity activity) {
        activity.startActivityForResult(new Intent(activity, QRScanActivity.class),
                REQUEST_CODE_QR);
    }

    public static @Nullable
    SdkParams handleActivityResult(int requestCode,
                                   int resultCode,
                                   @Nullable Intent data) {
        if (requestCode != REQUEST_CODE_QR) {
            return null;
        }
        if (resultCode == Activity.RESULT_OK && data != null) {
            String barcode = data.getStringExtra("BARCODE");
            return extractParamsFromBarcode(barcode);
        } else {
            return null;
        }
    }

    @Nullable
    public static SdkParams extractParamsFromBarcode(String barcode) {
        AlanLogger.i("Got barcode: " + barcode);
        if (barcode.endsWith(".json")) {
            SdkParams barcodeParams = new SdkParams();
            barcodeParams.videoJson = barcode;
            return barcodeParams;
        }
        if (barcode.startsWith("https://")) {
            SdkParams barcodeParams = new SdkParams();
            if (barcode.contains("studio.alan-stage.app")) {
                barcodeParams.server = STAGE_ENDPOINT;
                barcodeParams.projectId = barcode.substring(30);
            } else {
                barcodeParams.server = PROD_ENDPOINT;
                barcodeParams.projectId = barcode.substring(24);
            }

            return barcodeParams;
        } else {
            try {
                Gson gson = new Gson();
                return gson.fromJson(barcode, SdkParams.class);
            } catch (Exception e) {
                AlanLogger.e("Failed to parse barcode");
                return null;
            }
        }
    }

    public void skipMillis(int millis) {
        recorder.skipMillis(millis);
    }

    public void setScreenshotAddon(@Nullable ScreenshotAddon screenshotAddon) {
        if (screenShooter != null) {
            screenShooter.setAddon(screenshotAddon);
        }
    }

    public void onBackground() {
        if (!prefs.keepMicrophoneInBackground()) {
            deactivate();
            stopRecording();
        }
    }

    public void onForeground() {
        if (isWakeWordEnabled) {
            record();
        }
    }

    // NATIVE
    @Keep
    private native boolean callScript(String method, String jsonParams, ScriptMethodCallback callback);

    @Keep
    private native boolean setParamsNative(String params);

    @Keep
    private native boolean initNative(String server,
                                      String projectId,
                                      String dialogId,
                                      String authJson,
                                      String modelPath,
                                      String uuid,
                                      String sdkVersion,
                                      String platform,
                                      String appName,
                                      int sampleRate);

    @Keep
    private native void resetNative();

    @Keep
    private native String getVersionNative();

    @Keep
    private native void turn(boolean value);

    @Keep
    private native void stopNative();

}
