package com.alan.alansdk.button;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Point;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.util.Log;
import android.view.Display;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.WindowManager;
import android.webkit.WebView;
import android.widget.FrameLayout;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.alan.alansdk.Alan;
import com.alan.alansdk.AlanActivateCompletionCallback;
import com.alan.alansdk.AlanCallback;
import com.alan.alansdk.AlanConfig;
import com.alan.alansdk.AlanSDK;
import com.alan.alansdk.AlanState;
import com.alan.alansdk.BuildConfig;
import com.alan.alansdk.CheckPermissionsActivity;
import com.alan.alansdk.PermissionListener;
import com.alan.alansdk.R;
import com.alan.alansdk.ScriptMethodCallback;
import com.alan.alansdk.alanbase.ConnectionState;
import com.alan.alansdk.alanbase.SdkParams;
import com.alan.alansdk.button.animations.ButtonXAnimation;
import com.alan.alansdk.button.animations.HintHideAnimation;
import com.alan.alansdk.logging.AlanLogger;
import com.alan.alansdk.prefs.AlanPrefs;
import com.alan.alansdk.qr.BarcodeEvent;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;

public class AlanButton extends FrameLayout implements AlanSDK {

    private final static String TAG = "AlanButton";

    private AlanActivateCompletionCallback activateCompletionCallback = null;

    public static final int BUTTON_LEFT = 1;
    public static final int BUTTON_RIGHT = 2;

    private StateHandler stateHandler;

    public View button;
    private int buttonPosition;

    private View hintPanelLeft;
    private TextView hintTextLeft;
    private View hintPanelRight;
    private TextView hintTextRight;

    private GradientButton gradientBg;

    private Point displaySize;
    private Alan sdk;
    private AlanStateListener stateListener;
    private ColorManager colorManager;

    public AlanButtonInteractionListener buttonInteractionListener;

    private HintHideAnimation hintHideAnimation;
    private ButtonXAnimation buttonXAnimation;

    private boolean showHintPanel = true;
    private boolean showAlanButton = true;
    private boolean needsInitialLayout = true;

    /// Flag indicates that "micAllowed" is sent
    private boolean hasMicAllowedEvent;
    /// Flag indicates that "firstClick" is sent
    private boolean hasFirstTapEvent;

    View activeHintPanel;
    TextView activeHintText;

    public boolean stopStick = false;

    private Set<AlanCallback> buttonCallbacks = new LinkedHashSet<>();

    public AlanButton(final Context context, AttributeSet attrs) {
        super(context, attrs);

        TypedArray a = context.getTheme().obtainStyledAttributes(
                attrs,
                R.styleable.AlanButton,
                0, 0);

        try {
            buttonPosition = a.getInt(R.styleable.AlanButton_button_horizontal_align, BUTTON_RIGHT);
        } finally {
            a.recycle();
        }

        LayoutInflater inflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        addView(inflater.inflate(R.layout.alan_button, null));

        gradientBg = findViewById(R.id.button_background);

        WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        Display display = wm.getDefaultDisplay();
        displaySize = new Point();
        display.getSize(displaySize);

        buttonInteractionListener = new AlanButtonInteractionListener() {
            @Override
            public void onQRTime() {
                if (BuildConfig.QR_ENABLED) {
                    Alan.openQRScanner(getActivity());
                } else {
                    AlanLogger.w("QR scanner is disabled");
                }
            }
        };

        colorManager = new ColorManager(getContext());
        stateListener = new AlanStateListener(this, colorManager);
        EventBus.getDefault().register(this);

        init();

        setWillNotDraw(false);
    }

    @Nullable
    @Override
    protected Parcelable onSaveInstanceState() {
        AlanLogger.e("on save instance");
        if (sdk != null) {
            sdk.onBackground();
        }
        return super.onSaveInstanceState();
    }

    @Override
    protected void onRestoreInstanceState(Parcelable state) {
        super.onRestoreInstanceState(state);
        AlanLogger.e("on restore instance");
        if (sdk != null) {
            sdk.onForeground();
        }
    }

    @Subscribe(threadMode = ThreadMode.ASYNC)
    public void onBarcodeEvent(BarcodeEvent event) {
        if (!BuildConfig.QR_ENABLED) {
            return;
        }
        SdkParams data = Alan.extractParamsFromBarcode(event.getPayload());
        if (data == null || data.videoJson != null) {
            //handle video tool case in the activity code
            return;
        }
        sdk.unregisterCallback(stateListener);
        initWithConfig(data.toConfig());
    }

    private void setSDK(Alan sdk) {
        this.sdk = sdk;
    }

    /**
     * Connects AlanButton view to AlanTutor backend.
     * After executing this method AlanButton view is ready to work.
     * @param config AlanConfig object.
     *               At least `projectKey` field should be set in order to connect with backend
     * @return true if init is successful
     */
    public boolean initWithConfig(AlanConfig config) {

        hasMicAllowedEvent = false;
        hasFirstTapEvent = false;

        setSDK(Alan.getInstance(getContext().getApplicationContext()));
        sdk.clearCallbacks();
        sdk.registerCallback(stateListener);

        checkAudioPermissions();

        return sdk.initWithConfig(config);
    }

    /**
     * Init AlanButton view with existing SDK object.
     * Normally you should prefer `initWithConfig` method
     * @param alan `Alan` SDK object
     */
    public void initWithSDK(Alan alan) {
        if (sdk != null) {
            sdk.deactivate();
            sdk.clearCallbacks();
            sdk = null;
        }
        setSDK(alan);
        sdk.registerCallback(stateListener);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method
     */
    public boolean initSDK(@Nullable String server,
                           String projectId,
                           @Nullable String dialogId,
                           @Nullable String authJson) {
        AlanConfig config = AlanConfig.builder()
                .setServer(server)
                .setProjectId(projectId)
                .setDialogId(dialogId)
                .setDataObject(authJson)
                .build();
        return initWithConfig(config);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method
     */
    public boolean initSDK(String server, String projectId, @Nullable String dialogId) {
        return initSDK(null, projectId, dialogId, null);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method
     */
    public boolean initSDK(String projectId, String dialogId) {
        return initSDK(null, projectId, dialogId, null);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method
     */
    public boolean initSDK(String projectId) {
        return initSDK(null, projectId, null, null);
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        if (sdk != null) {
            sdk.unregisterCallback(stateListener);
        }
        EventBus.getDefault().unregister(this);
    }

    private void init() {
        stateHandler = new StateHandler(getContext().getApplicationContext(), this, colorManager);

        button = findViewById(R.id.button_container);
        buttonXAnimation = new ButtonXAnimation(button);

        hintPanelLeft = findViewById(R.id.hint_panel_left);
        hintTextLeft = findViewById(R.id.hint_text);
        hintPanelRight = findViewById(R.id.hint_panel_right);
        hintTextRight = findViewById(R.id.hint_text_right);

        hintPanelRight.setTag("RIGHT");
        hintPanelLeft.setTag("LEFT");

        needsInitialLayout = true;

        setElevation(20.0f);

        setSDK(Alan.getInstance(getContext().getApplicationContext()));

        updateButtonVisibility();

        button.setOnTouchListener(new ButtonTouchListener());

        if (sdk.getConnectionState() != ConnectionState.CONNECTED) {
            gradientBg.setColors(colorManager.getConnectingColor().getFirstColor(), colorManager.getConnectingColor().getSecondColor());
            setState(AlanState.CONNECTING, true);
        } else {
            gradientBg.setColors(colorManager.getIdleColor().getFirstColor(), colorManager.getIdleColor().getSecondColor());
            setState(AlanState.ONLINE, true);
        }
        sdk.registerCallback(stateListener);
    }

    private float oldX = -100;

    void reset() {
        if (stateHandler.getState() == AlanState.ONLINE) {
            stateHandler.setIdleState();
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        if (needsInitialLayout && getWidth() > 0 && getHeight() > 0) {
            int buttonSize = sdk.getButtonSize();
            updateButtonSize(buttonSize);

            button.setX(getWidth() - button.getWidth());
            oldX = button.getX();
            needsInitialLayout = false;
            Map<String,String> customLogos = sdk.getCustomLogos();
            stateHandler.setCustomLogos(customLogos);

            if (checkAudioPermissions()) {
                sendMicAllowedEvent();
            }
        }
        if (stopStick) {
            oldX = button.getX();
            return;
        }
        if (button.getX() != oldX || (activeHintText == null && showHintPanel)) {
            resetHintPanelState();
            stickToEdge(buttonPosition);
        }
    }

    private void reLayoutChildren(final View view) {
        post(new Runnable() {
            @Override
            public void run() {
                view.measure(
                        View.MeasureSpec.makeMeasureSpec(view.getMeasuredWidth(), View.MeasureSpec.EXACTLY),
                        View.MeasureSpec.makeMeasureSpec(view.getMeasuredHeight(), View.MeasureSpec.EXACTLY));
                view.layout(view.getLeft(), view.getTop(), view.getRight(), view.getBottom());
            }
        });
    }

    public void setHintPanelVisibility(boolean hintVisibility) {
        showHintPanel = hintVisibility;
    }

    private void toggleAlan() {
        if (sdk == null) {
            return;
        }

        if (!checkAudioPermissions()) {
            requestAudioPermissions();
            activateCompletionCallback = new AlanActivateCompletionCallback() {
                @Override
                public void activateResult(boolean didActivate) {
                }
            };
            return;
        }
        else {
            sendMicAllowedEvent();
        }

        sdk.toggle();

        sendFirstTapEvent();
    }

    private boolean checkAudioPermissions() {
        return ContextCompat.checkSelfPermission(getContext(), Manifest.permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
    }

    public void requestAudioPermissions() {
        CheckPermissionsActivity.permissionListener = new PermissionListener() {
            @Override
            public void permissionResult(boolean hasPermission) {
                if (activateCompletionCallback != null) {
                    if (hasPermission) {
                        activate();
                    }
                    activateCompletionCallback.activateResult(hasPermission);
                    activateCompletionCallback = null;
                }
            }
        };
        Context context = getContext();
        context.startActivity(new Intent(context, CheckPermissionsActivity.class));
    }

    private Activity getActivity() {
        Context context = getContext();
        while (context instanceof ContextWrapper) {
            if (context instanceof Activity) {
                return (Activity) context;
            }
            context = ((ContextWrapper) context).getBaseContext();
        }
        return null;
    }

    @Override
    public boolean performClick() {
        toggleAlan();
        return super.performClick();
    }

    @Override
    public boolean performLongClick() {
        return super.performLongClick();
    }

    private final static long QR_PRESS_TIME = TimeUnit.SECONDS.toMillis(5);
    private final static float DRAG_TRESHOLD = 100;
    private boolean clickState = false;
    private boolean dragState = false;
    private float startX = 0.0f;
    private float startY = 0.0f;
    private float deltaX = 0.0f;
    private float deltaY = 0.0f;

    class QrTimerTask extends TimerTask {
        @Override
        public void run() {
            isQrMode = true;
            AlanLogger.i("" + QR_PRESS_TIME + " seconds passed");
            if (buttonInteractionListener != null) {
                buttonInteractionListener.onQRTime();
            }
        }
    }

    private boolean isQrMode = false;
    private QrTimerTask qrTimerTask;
    private Timer qrTimer;
    private long time1 = 0;

    private void showHint() {
        WebView webView = findViewById(R.id.webview);
        webView.loadData("<!DOCTYPE HTML><html><head><meta charset=\\\"UTF-8\\\"><meta name=\\\"viewport\\\" content=\\\"width=device-width, initial-scale=1.0\\\"><title></title></head><body><p style=\\\"font-weight:bold;font-size:150;color:white;font-family:Arial, Helvetica, sans-serif\\\">This is what you can ask Alan: <br><br> • What is bitcoin price? <br> • What is bitcoin price for past week? <br>• What is bitcoin price for past month? <br> • What is bitcoin price in dollars?</p></body></html>",
                null,
                null);
    }

    class ButtonTouchListener implements OnTouchListener {

        @Override
        public boolean onTouch(View v, MotionEvent event) {
            switch (event.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    clickState = true;
                    stateHandler.setTouchedState();
                    startX = event.getRawX();
                    startY = event.getRawY();

                    deltaX = event.getRawX() - button.getX();
                    deltaY = event.getRawY() - getY();

                    qrTimer = new Timer();
                    qrTimerTask = new QrTimerTask();
                    qrTimer.schedule(qrTimerTask, QR_PRESS_TIME);
                    time1 = System.currentTimeMillis();
                    break;
                case MotionEvent.ACTION_UP:
                    stateHandler.setUntouchedState();
                    if (dragState) {
                        stopStick = false;
                        stickToEdge();
                    }
                    if (clickState) {
                        clickState = false;
                        if (!isQrMode) {
                            if (System.currentTimeMillis() - time1 > 1500) {
                                showHint();
                            } else {
                                performClick();
                            }
                        }
                        isQrMode = false;
                    }
                    dragState = false;
                    qrTimer.cancel();
                    break;
                case MotionEvent.ACTION_MOVE:
                    if (dragState ||
                            Math.abs(event.getRawX() - startX) >= DRAG_TRESHOLD ||
                            Math.abs(event.getRawY() - startY) >= DRAG_TRESHOLD) {
                        dragState = true;
                        float newX = Math.max(0, Math.min(event.getRawX() - deltaX, displaySize.x - button.getWidth()));
                        float newY = Math.max(0, Math.min(event.getRawY() - deltaY, displaySize.y - getHeight()));
                        setButtonPosition(newX, newY);
                        clickState = false;
                        qrTimer.cancel();
                        hideHintPanel();
                    }
                    break;
            }

            return true;
        }
    }

    private void setButtonPosition(float x, float y) {
        button.setX(x);
        setY(y);
    }

    private void stickToEdge() {
        int screenWidth = getContext().getResources().getDisplayMetrics().widthPixels;
        if (button.getX() + button.getWidth() / 2 < screenWidth / 2) {
            buttonPosition = BUTTON_LEFT;
        } else {
            buttonPosition = BUTTON_RIGHT;
        }
        stickToEdge(buttonPosition);
    }

    private void stickToEdge(int align) {
        buttonPosition = align;
        if (stopStick) {
            return;
        }
        switch (align) {
            case BUTTON_LEFT:
                buttonXAnimation.animate(0);
                break;
            case BUTTON_RIGHT:
                buttonXAnimation.animate(getWidth() - button.getWidth());
                break;
            default:
                AlanLogger.e("Unknown button alignment");
        }
    }

    public void setButtonAlign(int buttonAlign) {
        stickToEdge(buttonAlign);
    }

    public void setState(AlanState alanState) {
        if (alanState == AlanState.ONLINE) {
            updateButtonVisibility();
            hideHintPanel();
        }
        setState(alanState, false);
    }

    public void setDisconnectedState() {
        stateHandler.setDisconnectedState();
    }

    public void setState(AlanState alanState, boolean force) {
        stateHandler.setState(alanState);
        post(new Runnable() {
            @Override
            public void run() {
                invalidate();
            }
        });
    }

    public AlanState getState() {
        return stateHandler.getState();
    }

    public void updateButtonVisibility() {
        showHintPanel = !sdk.isHintPanelHidden();
        showAlanButton = !sdk.isButtonHidden();
        if (!showAlanButton) {
            hideButton();
            showHintPanel = false;
        }
        else {
            showButton();
        }
        if (!showHintPanel) {
            hideHintPanel();
        }
        else {
            showHintPanel();
        }
    }

    private int getButtonSizeWithDimension(int buttonSize) {
        int dp = buttonSize + AlanPrefs.DEFAULT_BUTTON_OFFSET;
        final float scale = getContext().getResources().getDisplayMetrics().density;
        int buttonSizeWithDimension = (int) (dp * scale + 0.5f);
        return buttonSizeWithDimension;
    }

    public void resetButtonSize() {
        updateButtonSize(AlanPrefs.DEFAULT_BUTTON_SIZE);
    }

    public void updateButtonSize(int buttonSize) {
        int buttonSizeWithDimension = getButtonSizeWithDimension(buttonSize);
        View view = findViewById(R.id.button_container);
        LayoutParams layoutParams = (LayoutParams) view.getLayoutParams();
        layoutParams.width = buttonSizeWithDimension;
        layoutParams.height = buttonSizeWithDimension;
        view.setLayoutParams(layoutParams);

        final Handler handler = new Handler(Looper.getMainLooper());
        handler.postDelayed(new Runnable() {
            @Override
            public void run() {
                stickToEdge();
            }
        }, 100);
    }

    public void updateLogos(Map<String,String> customLogos) {
        stateHandler.setCustomLogos(customLogos);
    }

    public void hideHintPanel() {
        if (activeHintPanel == null) {
            return;
        }
        if (activeHintPanel.getAlpha() == 1.0f) {
            hintHideAnimation.startAnimation();
        }
        if (activeHintPanel.getVisibility() != View.GONE) {
            activeHintPanel.setVisibility(View.GONE);
        }
    }

    void showHintPanel() {
        AlanState state = stateHandler.getState();
        if (state == AlanState.OFFLINE || state == AlanState.CONNECTING || state == AlanState.ONLINE) {
            return;
        }
        if (activeHintPanel == null) {
            return;
        }
        activeHintPanel.setAlpha(1.0f);
        if (activeHintPanel.getVisibility() != View.VISIBLE) {
            activeHintPanel.setVisibility(View.VISIBLE);
        }
    }

    private void resetHintPanelState() {
        oldX = button.getX();
        if (!showHintPanel) {
            return;
        }
        if (button.getX() < 100) {
            activeHintPanel = hintPanelRight;
            activeHintText = hintTextRight;
        } else {
            activeHintPanel = hintPanelLeft;
            activeHintText = hintTextLeft;
        }
        hintHideAnimation = new HintHideAnimation(activeHintPanel);
        reLayoutChildren(this);
    }

    public void hideButton() {
        setVisibility(View.GONE);
    }

    public boolean hintPanelVisible() {
        AlanState state = stateHandler.getState();
        if (state == AlanState.OFFLINE || state == AlanState.CONNECTING || state == AlanState.ONLINE) {
            return false;
        }
        return showHintPanel;
    }

    public void showButton() {
        if (!sdk.isInited()) {
            Log.w(TAG, "Button is disabled, skipping show");
            return; //Button is disabled
        }
        setVisibility(View.VISIBLE);
    }

    public void disableButton() {
        hideButton();
    }

    public Alan getSDK() {
        return sdk;
    }

    public void registerCallback(AlanCallback callback) {
        if (sdk != null) {
            buttonCallbacks.add(callback);
            sdk.registerCallback(callback);
        }
    }

    public void removeCallback(AlanCallback callback) {
        if (sdk != null) {
            buttonCallbacks.remove(callback);
            sdk.unregisterCallback(callback);
        }
    }

    public void clearCallbacks() {
        for (AlanCallback callback : buttonCallbacks) {
            removeCallback(callback);
        }
    }


    //================================================================================
    // Client Events
    //================================================================================

    private void sendMicAllowedEvent() {
        if (!hasMicAllowedEvent) {
            hasMicAllowedEvent = true;
            try {
                JSONObject params = new JSONObject();
                params.put("micAllowed", true);
                sendClientEvent(params.toString());
                AlanLogger.w("clientEvent \"micAllowed\" is sent");
            } catch (JSONException e) {
                AlanLogger.e("sendMicAllowedEvent - failed to create json");
            }
        }
    }

    private void sendFirstTapEvent() {
        if (!hasFirstTapEvent) {
            hasFirstTapEvent = true;
            try {
                JSONObject params = new JSONObject();
                params.put("firstClick", true);
                sendClientEvent(params.toString());
                AlanLogger.w("clientEvent \"firstClick\" is sent");
            } catch (JSONException e) {
                AlanLogger.e("sendFirstTapEvent - failed to create json");
            }
        }
    }

    private void sendClientEvent(final String jsonParams) {
        sdk.sendClientEvent(jsonParams);
    }

    //================================================================================
    // PROXY METHODS
    //================================================================================

    @Override
    public void callProjectApi(String method, String jsonParams) {
        callProjectApi(method, jsonParams, null);
    }

    @Override
    public void callProjectApi(final String method, final String jsonParams, @Nullable final ScriptMethodCallback callback) {
        if (sdk != null) {
            sdk.callProjectApi(method, jsonParams, callback);
        }
    }

    @Override
    public void playText(String text, @Nullable ScriptMethodCallback callback) {
        if (sdk != null) {
            sdk.playText(text, callback);
        }
    }

    @Override
    public void playText(String text) {
        playText(text, null);
    }

    @Override
    public void playCommand(String data, @Nullable ScriptMethodCallback callback) {
        if (sdk != null) {
            sdk.playCommand(data, callback);
        }
    }

    @Override
    public void setVisualState(String visualState) {
        if (sdk != null) {
            sdk.setVisualState(visualState);
        }
    }

    @Override
    public void activate(AlanActivateCompletionCallback... activateCallback) {
        if (!checkAudioPermissions()) {
            if (activateCallback.length > 0) {
                activateCompletionCallback = activateCallback[0];
            } else {
                activateCompletionCallback = new AlanActivateCompletionCallback() {
                    @Override
                    public void activateResult(boolean didActivate) {
                    }
                };
            }
            requestAudioPermissions();
        } else if (sdk != null) {
            sdk.activate(activateCallback);
            if (activateCallback.length > 0) {
                activateCompletionCallback = activateCallback[0];
                activateCompletionCallback.activateResult(true);
                activateCompletionCallback = null;
            }
        }
    }

    @Override
    public void deactivate() {
        if (sdk != null) {
            sdk.deactivate();
        }
    }

    @Override
    public boolean isActive() {
        if (sdk == null) {
            return false;
        }
        return sdk.alanState == AlanState.LISTEN ||
                sdk.alanState == AlanState.PROCESS ||
                sdk.alanState == AlanState.REPLY;
    }

}