package com.alan.alansdk.button;

import android.app.Activity;
import android.content.Context;
import android.content.ContextWrapper;
import android.graphics.Bitmap;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import com.alan.alansdk.Alan;
import com.alan.alansdk.AlanState;
import com.alan.alansdk.R;
import com.alan.alansdk.alanbase.AudioConfig;
import com.alan.alansdk.button.animations.BackgroundAnimation;
import com.alan.alansdk.button.animations.LoaderAnimation;
import com.alan.alansdk.button.animations.ButtonColorAnimation;
import com.alan.alansdk.button.animations.InnerBgListenAnimation;
import com.alan.alansdk.button.animations.LogoAnimation;
import com.alan.alansdk.button.animations.OnAnimationEnd;
import com.alan.alansdk.button.animations.SpeakingBackAnimation;
import com.bumptech.glide.Glide;
import com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.bumptech.glide.request.RequestOptions;
import com.bumptech.glide.request.target.DrawableImageViewTarget;

import java.util.Map;

public class StateHandler {

    private WakeLockManager wakeLockManager;
    private AlanState currentState;
    private SoundManager soundManager;
    private ViewGroup buttonContainer;

    private View shadow;
    private GradientButton backgroundView;
    private View innerBgTriangle;
    private ImageView innerBgCircle;
    private ImageView logoView1;
    private ImageView logoView2;
    private View logoContainer;
    private ImageView loaderView;

    private ColorManager colorManager;

    private LogoAnimation logoAnimation;
    private BackgroundAnimation backgroundAnimation;
    private SpeakingBackAnimation triangleAnimation;
    private SpeakingBackAnimation circleAnimation;
    private InnerBgListenAnimation innerBgListenAnimation;
    private ButtonColorAnimation buttonColorAnimation;
    private LoaderAnimation buttonLoaderAnimation;

    private String customLogoIdle;
    private String customLogoListen;
    private String customLogoProcess;
    private String customLogoReply;

    public StateHandler(Context context, AlanButton buttonView, ColorManager colorManager) {
        customLogoIdle = null;
        customLogoListen = null;
        customLogoProcess = null;
        customLogoReply = null;

        wakeLockManager = new WakeLockManager(context);
        soundManager = new SoundManager(context);
        buttonContainer = buttonView;

        shadow = buttonContainer.findViewById(R.id.shadow);
        backgroundView = buttonContainer.findViewById(R.id.button_background);
        logoView1 = buttonContainer.findViewById(R.id.logo_1);
        logoView2 = buttonContainer.findViewById(R.id.logo_2);
        innerBgTriangle = buttonContainer.findViewById(R.id.inner_bg_triangle);
        innerBgCircle = buttonContainer.findViewById(R.id.inner_bg_circle);
        buttonContainer = buttonContainer.findViewById(R.id.button_container);
        loaderView = buttonContainer.findViewById(R.id.button_loader);

        logoAnimation = new LogoAnimation(logoView1, logoView2, false);
        backgroundAnimation = new BackgroundAnimation(backgroundView);
        buttonLoaderAnimation = new LoaderAnimation(loaderView);
        triangleAnimation = new SpeakingBackAnimation(innerBgTriangle, SpeakingBackAnimation.STATE.SPEAKING_BACK, false);
        circleAnimation = new SpeakingBackAnimation(innerBgCircle, SpeakingBackAnimation.STATE.IDLE, false);
        innerBgListenAnimation = new InnerBgListenAnimation(innerBgTriangle);
        buttonColorAnimation = new ButtonColorAnimation(backgroundView);

        this.colorManager = colorManager;
    }

    public void setCustomLogos(Map<String, String> customLogos) {
        customLogoIdle = null;
        customLogoListen = null;
        customLogoProcess = null;
        customLogoReply = null;

        logoAnimation.stopAnimation();

        if (customLogos == null || customLogos.size() == 0) {
            logoAnimation = new LogoAnimation(logoView1, logoView2, false);
            triangleAnimation = new SpeakingBackAnimation(innerBgTriangle, SpeakingBackAnimation.STATE.SPEAKING_BACK, false);
            circleAnimation = new SpeakingBackAnimation(innerBgCircle, SpeakingBackAnimation.STATE.IDLE, false);
            return;
        }
        else {
            logoAnimation = new LogoAnimation(logoView1, logoView2, true);
            triangleAnimation = new SpeakingBackAnimation(innerBgTriangle, SpeakingBackAnimation.STATE.SPEAKING_BACK, true);
            circleAnimation = new SpeakingBackAnimation(innerBgCircle, SpeakingBackAnimation.STATE.IDLE, true);
        }

        for (Map.Entry<String, String> entry : customLogos.entrySet()) {
            String key = entry.getKey();
            if (key.equals("logoIdle")) {
                String customLogo = entry.getValue();
                if (customLogo != null && customLogo.length() > 0) {
                    customLogoIdle = customLogo;
                }
            }
            else if (key.equals("logoListen")) {
                String customLogo = entry.getValue();
                if (customLogo != null && customLogo.length() > 0) {
                    customLogoListen = customLogo;
                }
            }
            else if (key.equals("logoProcess")) {
                String customLogo = entry.getValue();
                if (customLogo != null && customLogo.length() > 0) {
                    customLogoProcess = customLogo;
                }
            }
            else if (key.equals("logoReply")) {
                String customLogo = entry.getValue();
                if (customLogo != null && customLogo.length() > 0) {
                    customLogoReply = customLogo;
                }
            }
        }
    }

    public void setCustomLogo(String customLogo) {
        if (customLogo != null && customLogo.length() > 0) {
            DrawableImageViewTarget target = new DrawableImageViewTarget(logoView1);
            Glide.with(buttonContainer
                    .getContext())
                    .load(customLogo)
                    .fitCenter()
                    .apply(new RequestOptions().fitCenterTransform().circleCropTransform())
                    .diskCacheStrategy(DiskCacheStrategy.ALL)
                    .into(target);
        }
        else {
            logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
            logoView2.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
        }
    }

    public void setState(AlanState alanState) {
        if (currentState == alanState) {
            return;
        }
        if (currentState == AlanState.REPLY && alanState == AlanState.LISTEN) {
            Alan.getInstance(backgroundView.getContext()).skipMillis(AudioConfig.SKIP_MILLIS_ON_START);
            soundManager.playNext();
        }
        if (currentState == AlanState.LISTEN && alanState == AlanState.ONLINE) {
            soundManager.playOff();
        }
        if (currentState == AlanState.ONLINE && alanState == AlanState.LISTEN) {
            Alan.getInstance(backgroundView.getContext()).skipMillis(AudioConfig.SKIP_MILLIS_ON_START);
            soundManager.playOn();
        }
        currentState = alanState;
        switch (alanState) {
            case ONLINE:
                setIdleState();
                break;
            case OFFLINE:
                setDisconnectedState();
                break;
            case CONNECTING:
                setConnectingState();
                break;
            case LISTEN:
                setListenState();
                break;
            case PROCESS:
                setUnderstoodState();
                break;
            case REPLY:
                setSpeakBackState();
                break;
            case NO_PERMISSION:
                setNoPermissionState();
                break;
            case UNKNOWN:
                break;
        }
    }

    void setNoPermissionState() {
        buttonColorAnimation.start(colorManager.getDisconnectedColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.GONE);

        logoAnimation.stopAnimation();
        backgroundAnimation.stopAnimation();
        triangleAnimation.stopAnimation();
        circleAnimation.stopAnimation();
        innerBgListenAnimation.stopAnimation();
        buttonLoaderAnimation.stopAnimation();

        logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_no_mic));
        logoView1.setAlpha(1.0f);
        logoView2.setAlpha(0.0f);

        wakeLockManager.releaseWakeLock();
    }

    void setConnectingState() {
        buttonColorAnimation.start(colorManager.getConnectingColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        buttonLoaderAnimation.startAnimation();
        logoAnimation.stopAnimation();
        backgroundAnimation.stopAnimation();
        triangleAnimation.stopAnimation();
        circleAnimation.stopAnimation();
        innerBgListenAnimation.stopAnimation();

        if (customLogoIdle != null) {
            setCustomLogo(customLogoIdle);
        }
        else {
            logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
        }

        innerBgTriangle.setVisibility(View.GONE);
    }

    void setDisconnectedState() {
        buttonColorAnimation.start(colorManager.getDisconnectedColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.GONE);

        logoAnimation.stopAnimation();
        backgroundAnimation.stopAnimation();
        triangleAnimation.stopAnimation();
        circleAnimation.stopAnimation();
        innerBgListenAnimation.stopAnimation();
        buttonLoaderAnimation.stopAnimation();

        logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_no_network));
        logoView1.setAlpha(1.0f);
        logoView2.setAlpha(0.0f);

        wakeLockManager.releaseWakeLock();
    }

    void setIdleState() {
        buttonColorAnimation.start(colorManager.getIdleColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.GONE);

        logoAnimation.stopAnimation();
        backgroundAnimation.startNormalAnimation();
        triangleAnimation.startRespondingAnimation();
        circleAnimation.startSpeakingBackAnimation();
        innerBgListenAnimation.stopAnimation();
        buttonLoaderAnimation.stopAnimation();

        if (customLogoIdle != null) {
            setCustomLogo(customLogoIdle);
        }
        else {
            logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
        }

        logoView1.setAlpha(1.0f);
        logoView2.setAlpha(0.0f);

        wakeLockManager.releaseWakeLock();
    }

    private void setListenState() {
        wakeLockManager.acquireWakeLock();

        buttonColorAnimation.start(colorManager.getListenColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.VISIBLE);
        logoAnimation.startAnimation();
        backgroundAnimation.startRespondingAnimation();
        triangleAnimation.startRespondingAnimation();
        circleAnimation.startSpeakingBackAnimation();
        innerBgListenAnimation.startAnimation();
        buttonLoaderAnimation.stopAnimation();

        if (customLogoListen != null) {
            setCustomLogo(customLogoListen);
        }
        else {
            logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
        }
    }

    private void setUnderstoodState() {
        buttonColorAnimation.start(colorManager.getProcessColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        if (customLogoProcess != null) {
            setCustomLogo(customLogoProcess);
        }
        else {
            logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
        }

        innerBgTriangle.setVisibility(View.VISIBLE);
    }

    private void setSpeakBackState() {
        buttonColorAnimation.start(colorManager.getReplyColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.VISIBLE);

        backgroundAnimation.startNormalAnimation();
        triangleAnimation.startSpeakingBackAnimation();
        circleAnimation.startRespondingAnimation();
        innerBgListenAnimation.stopAnimation();
        buttonLoaderAnimation.stopAnimation();

        if (customLogoReply != null) {
            setCustomLogo(customLogoReply);
        }
        else {
            logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
        }
    }

    void setTouchedState() {
        if (currentState != AlanState.ONLINE) {
            innerBgTriangle.setVisibility(View.GONE);
        }
        logoView1.setScaleX(0.9f);
        logoView1.setScaleY(0.9f);
        logoView2.setScaleX(0.9f);
        logoView2.setScaleY(0.9f);
        shadow.setVisibility(View.GONE);
        backgroundView.setScaleX(0.9f);
        backgroundView.setScaleY(0.9f);
    }


    void setUntouchedState() {
        if (currentState != AlanState.ONLINE) {
            innerBgTriangle.setVisibility(View.VISIBLE);
        }
        logoView1.setScaleX(1.0f);
        logoView1.setScaleY(1.0f);
        logoView2.setScaleX(1.0f);
        logoView2.setScaleY(1.0f);
        shadow.setVisibility(View.VISIBLE);
        backgroundView.setScaleX(1.0f);
        backgroundView.setScaleY(1.0f);
    }

    public AlanState getState() {
        return currentState;
    }
}
