package com.alan.alansdk.button;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Point;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcelable;
import android.util.AttributeSet;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.TextView;

import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import com.alan.alansdk.Alan;
import com.alan.alansdk.AlanActivateCompletionCallback;
import com.alan.alansdk.AlanCallback;
import com.alan.alansdk.AlanConfig;
import com.alan.alansdk.AlanSDK;
import com.alan.alansdk.AlanState;
import com.alan.alansdk.BuildConfig;
import com.alan.alansdk.CheckPermissionsActivity;
import com.alan.alansdk.PermissionListener;
import com.alan.alansdk.R;
import com.alan.alansdk.ScriptMethodCallback;
import com.alan.alansdk.alanbase.ConnectionState;
import com.alan.alansdk.alanbase.SdkParams;
import com.alan.alansdk.button.animations.ButtonXAnimation;
import com.alan.alansdk.button.animations.HintHideAnimation;
import com.alan.alansdk.button.animations.PopupAlphaAnimation;
import com.alan.alansdk.button.animations.PopupXAnimation;
import com.alan.alansdk.logging.AlanLogger;
import com.alan.alansdk.prefs.AlanPrefs;
import com.alan.alansdk.qr.BarcodeEvent;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;

public class AlanButton extends FrameLayout implements AlanSDK {

    private final static String TAG = "AlanButton";

    private AlanActivateCompletionCallback activateCompletionCallback = null;

    private float popupThreshold = 0.3f;
    private int popupState = 0;
    public static final int POPUP_CLOSE = 0;
    public static final int POPUP_OPEN = 1;
    public static final int POPUP_FROM_LEFT = 2;
    public static final int POPUP_FROM_RIGHT = 3;
    public static final int POPUP_TO_LEFT = 4;
    public static final int POPUP_TO_RIGHT = 5;

    public static final int BUTTON_LEFT = 1;
    public static final int BUTTON_RIGHT = 2;

    private StateHandler stateHandler;
    private int previousScreenWidth = 0;

    public View button;
    public View buttonContainer;
    public View buttonPopup;
    public View buttonPopupBackground;
    private int buttonPosition;

    private View hintPanelLeft;
    private TextView hintTextLeft;
    private View hintPanelRight;
    private TextView hintTextRight;

    private GradientButton gradientBg;

    private Point displaySize;
    private Alan sdk;
    private AlanStateListener stateListener;
    private ColorManager colorManager;

    public AlanButtonInteractionListener buttonInteractionListener;

    private HintHideAnimation hintHideAnimation;
    private ButtonXAnimation buttonXAnimation;
    private PopupXAnimation popupXAnimation;
    private PopupAlphaAnimation popupAlphaAnimation;

    private boolean showHintPanel = true;
    private boolean showAlanButton = true;
    private boolean needsInitialLayout = true;

    /// Flag indicates that "micAllowed" is sent
    private boolean hasMicAllowedEvent;
    /// Flag indicates that "firstClick" is sent
    private boolean hasFirstTapEvent;

    View activeHintPanel;
    TextView activeHintText;

    public boolean stopStick = false;

    private Set<AlanCallback> buttonCallbacks = new LinkedHashSet<>();

    public AlanButton(final Context context, AttributeSet attrs) {
        super(context, attrs);

        TypedArray a = context.getTheme().obtainStyledAttributes(
                attrs,
                R.styleable.AlanButton,
                0, 0);

        try {
            buttonPosition = a.getInt(R.styleable.AlanButton_button_horizontal_align, BUTTON_RIGHT);
        } finally {
            a.recycle();
        }

        LayoutInflater inflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        addView(inflater.inflate(R.layout.alan_button, null));

        gradientBg = findViewById(R.id.button_background);

        int screenWidth = getContext().getResources().getDisplayMetrics().widthPixels;
        int screenHeight = getContext().getResources().getDisplayMetrics().heightPixels;
        displaySize = new Point(screenWidth, screenHeight);

        buttonInteractionListener = new AlanButtonInteractionListener() {
            @Override
            public void onQRTime() {
                if (BuildConfig.QR_ENABLED) {
                    Alan.openQRScanner(getActivity());
                } else {
                    AlanLogger.w("QR scanner is disabled");
                }
            }
        };

        colorManager = new ColorManager(getContext());
        stateListener = new AlanStateListener(this, colorManager);
        EventBus.getDefault().register(this);

        init();

        setWillNotDraw(false);
    }

    @Nullable
    @Override
    protected Parcelable onSaveInstanceState() {
        AlanLogger.i("on save instance");
        if (sdk != null) {
            sdk.onBackground();
        }
        return super.onSaveInstanceState();
    }

    @Override
    protected void onRestoreInstanceState(Parcelable state) {
        super.onRestoreInstanceState(state);
        AlanLogger.i("on restore instance");
        if (sdk != null) {
            sdk.onForeground();
        }
    }

    @Subscribe(threadMode = ThreadMode.ASYNC)
    public void onBarcodeEvent(BarcodeEvent event) {
        if (!BuildConfig.QR_ENABLED) {
            return;
        }
        SdkParams data = Alan.extractParamsFromBarcode(event.getPayload());
        if (data == null || data.videoJson != null) {
            //handle video tool case in the activity code
            return;
        }
        sdk.unregisterCallback(stateListener);
        initWithConfig(data.toConfig());
    }

    private void setSDK(Alan sdk) {
        this.sdk = sdk;
    }

    /**
     * Connects AlanButton view to AlanTutor backend.
     * After executing this method AlanButton view is ready to work.
     * @param config AlanConfig object.
     *               At least `projectKey` field should be set in order to connect with backend
     * @return true if init is successful
     */
    public boolean initWithConfig(AlanConfig config) {

        hasMicAllowedEvent = false;
        hasFirstTapEvent = false;

        setSDK(Alan.getInstance(getContext().getApplicationContext()));
        sdk.clearCallbacks();
        sdk.registerCallback(stateListener);

        checkAudioPermissions();

        return sdk.initWithConfig(config);
    }

    /**
     * Init AlanButton view with existing SDK object.
     * Normally you should prefer `initWithConfig` method
     * @param alan `Alan` SDK object
     */
    public void initWithSDK(Alan alan) {
        if (sdk != null) {
            sdk.deactivate();
            sdk.clearCallbacks();
            sdk = null;
        }
        setSDK(alan);
        sdk.registerCallback(stateListener);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method
     */
    public boolean initSDK(@Nullable String server,
                           String projectId,
                           @Nullable String dialogId,
                           @Nullable String authJson) {
        AlanConfig config = AlanConfig.builder()
                .setServer(server)
                .setProjectId(projectId)
                .setDialogId(dialogId)
                .setDataObject(authJson)
                .build();
        return initWithConfig(config);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method
     */
    public boolean initSDK(String server, String projectId, @Nullable String dialogId) {
        return initSDK(null, projectId, dialogId, null);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method
     */
    public boolean initSDK(String projectId, String dialogId) {
        return initSDK(null, projectId, dialogId, null);
    }

    @Deprecated
    /**
     * Use `initWithConfig` method
     */
    public boolean initSDK(String projectId) {
        return initSDK(null, projectId, null, null);
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        if (sdk != null) {
            sdk.unregisterCallback(stateListener);
        }
        EventBus.getDefault().unregister(this);
    }

    private void init() {
        stateHandler = new StateHandler(getContext().getApplicationContext(), this, colorManager);

        buttonContainer = findViewById(R.id.container);
        button = findViewById(R.id.button_container);
        buttonXAnimation = new ButtonXAnimation(button);

        hintPanelLeft = findViewById(R.id.hint_panel_left);
        hintTextLeft = findViewById(R.id.hint_text);
        hintPanelRight = findViewById(R.id.hint_panel_right);
        hintTextRight = findViewById(R.id.hint_text_right);

        hintPanelRight.setTag("RIGHT");
        hintPanelLeft.setTag("LEFT");

        needsInitialLayout = true;

        setElevation(20.0f);

        setSDK(Alan.getInstance(getContext().getApplicationContext()));

        updateButtonVisibility();
        setupPopup();

        button.setOnTouchListener(new ButtonTouchListener());

        if (sdk.getConnectionState() != ConnectionState.CONNECTED) {
            gradientBg.setColors(colorManager.getConnectingColor().getFirstColor(), colorManager.getConnectingColor().getSecondColor());
            stateHandler.setState(AlanState.CONNECTING);
        } else {
            gradientBg.setColors(colorManager.getIdleColor().getFirstColor(), colorManager.getIdleColor().getSecondColor());
            stateHandler.setState(AlanState.ONLINE);
        }
        sdk.registerCallback(stateListener);
    }

    private float oldX = -100;

    void reset() {
        if (stateHandler.getState() == AlanState.ONLINE) {
            stateHandler.setIdleState();
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        if (needsInitialLayout && getWidth() > 0 && getHeight() > 0) {
            int buttonSize = sdk.getButtonSize();
            updateButtonSize(buttonSize);

            button.setX(getWidth() - button.getWidth());
            oldX = button.getX();
            needsInitialLayout = false;
            Map<String,String> customLogos = sdk.getCustomLogos();
            stateHandler.setCustomLogos(customLogos);

            if (checkAudioPermissions()) {
                sendMicAllowedEvent();
            } else {
                int screenWidth = getContext().getResources().getDisplayMetrics().widthPixels;
                int screenHeight = getContext().getResources().getDisplayMetrics().heightPixels;
                if (previousScreenWidth != 0 && previousScreenWidth != screenWidth) {
                    stickToEdge(buttonPosition);
                }
                previousScreenWidth = screenWidth;
                displaySize = new Point(screenWidth, screenHeight);
                buttonPopup.setX(displaySize.x);
            }
        } else {
            int screenWidth = getContext().getResources().getDisplayMetrics().widthPixels;
            int screenHeight = getContext().getResources().getDisplayMetrics().heightPixels;
            if (previousScreenWidth != 0 && previousScreenWidth != screenWidth) {
                stickToEdge(buttonPosition);
            }
            previousScreenWidth = screenWidth;
            displaySize = new Point(screenWidth, screenHeight);
            buttonPopup.setX(displaySize.x);
        }
        if (stopStick) {
            oldX = button.getX();
            return;
        }
        if (button.getX() != oldX || (activeHintText == null && showHintPanel)) {
            resetHintPanelState();
            stickToEdge(buttonPosition);
        }
    }

    private void reLayoutChildren(final View view) {
        post(new Runnable() {
            @Override
            public void run() {
                view.measure(
                        View.MeasureSpec.makeMeasureSpec(view.getMeasuredWidth(), View.MeasureSpec.EXACTLY),
                        View.MeasureSpec.makeMeasureSpec(view.getMeasuredHeight(), View.MeasureSpec.EXACTLY));
                view.layout(view.getLeft(), view.getTop(), view.getRight(), view.getBottom());
            }
        });
    }

    public void setHintPanelVisibility(boolean hintVisibility) {
        showHintPanel = hintVisibility;
    }

    private void toggleAlan() {
        if (sdk == null) {
            return;
        }

        if (!checkAudioPermissions()) {
            requestAudioPermissions();
            activateCompletionCallback = new AlanActivateCompletionCallback() {
                @Override
                public void activateResult(boolean didActivate) {
                }
            };
            return;
        }
        else {
            sendMicAllowedEvent();
        }

        sdk.toggle();

        sendFirstTapEvent();
    }

    private boolean checkAudioPermissions() {
        return ContextCompat.checkSelfPermission(getContext(), Manifest.permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
    }

    public void requestAudioPermissions() {
        if (sdk != null) {
            sdk.setMicRequest();
        }
        CheckPermissionsActivity.permissionListener = new PermissionListener() {
            @Override
            public void permissionResult(boolean hasPermission) {
                if (activateCompletionCallback != null) {
                    deactivate();
                    if (hasPermission) {
                        activate();
                    } else {
                        stateHandler.setState(AlanState.NO_PERMISSION);
                    }
                    activateCompletionCallback.activateResult(hasPermission);
                    activateCompletionCallback = null;
                }
            }
        };
        Context context = getContext();
        context.startActivity(new Intent(context, CheckPermissionsActivity.class));

        sendMicPermissionPromptEvent();
    }

    private Activity getActivity() {
        Context context = getContext();
        while (context instanceof ContextWrapper) {
            if (context instanceof Activity) {
                return (Activity) context;
            }
            context = ((ContextWrapper) context).getBaseContext();
        }
        return null;
    }

    @Override
    public boolean performClick() {
        toggleAlan();
        return super.performClick();
    }

    @Override
    public boolean performLongClick() {
        return super.performLongClick();
    }

    private final static long QR_PRESS_TIME = TimeUnit.SECONDS.toMillis(5);
    private final static float DRAG_TRESHOLD = 25;
    private boolean clickState = false;
    private boolean dragState = false;
    private float startX = 0.0f;
    private float startY = 0.0f;
    private float startPopupX = 0.0f;
    private float deltaX = 0.0f;
    private float deltaY = 0.0f;

    class QrTimerTask extends TimerTask {
        @Override
        public void run() {
            isQrMode = true;
            AlanLogger.i("" + QR_PRESS_TIME + " seconds passed");
            if (buttonInteractionListener != null) {
                buttonInteractionListener.onQRTime();
            }
        }
    }

    private boolean isQrMode = false;
    private QrTimerTask qrTimerTask;
    private Timer qrTimer;
    private long time1 = 0;

    class ButtonTouchListener implements OnTouchListener {

        @Override
        public boolean onTouch(View v, MotionEvent event) {
            switch (event.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    clickState = true;
                    stateHandler.setTouchedState();
                    startX = event.getRawX();
                    startY = event.getRawY();

                    deltaX = event.getRawX() - button.getX();
                    deltaY = event.getRawY() - buttonContainer.getY();

                    qrTimer = new Timer();
                    qrTimerTask = new QrTimerTask();
                    qrTimer.schedule(qrTimerTask, QR_PRESS_TIME);
                    time1 = System.currentTimeMillis();
                    break;
                case MotionEvent.ACTION_UP:
                    stateHandler.setUntouchedState();
                    if (dragState) {
                        stopStick = false;
                        stickToEdge();
                        setEndPopupPosition();
                    }
                    if (clickState) {
                        clickState = false;
                        if (!isQrMode) {
                            if (System.currentTimeMillis() - time1 > 1500) {
                                //
                            } else {
                                performClick();
                            }
                        }
                        isQrMode = false;
                    }
                    dragState = false;
                    qrTimer.cancel();
                    break;
                case MotionEvent.ACTION_MOVE:
                    if (dragState || Math.abs(event.getRawX() - startX) >= DRAG_TRESHOLD || Math.abs(event.getRawY() - startY) >= DRAG_TRESHOLD) {
                        if (!dragState && popupState != POPUP_OPEN) {
                            if (buttonPosition == BUTTON_RIGHT) {
                                setPopupPosition(displaySize.x, buttonPopup.getY());
                                popupState = POPUP_FROM_RIGHT;
                            } else {
                                setPopupPosition(-displaySize.x, buttonPopup.getY());
                                popupState = POPUP_FROM_LEFT;
                            }
                        }
                        dragState = true;

                        float newX = Math.max(0, Math.min(event.getRawX() - deltaX, displaySize.x - button.getWidth()));
                        float newY = Math.max(0, Math.min(event.getRawY() - deltaY, displaySize.y - button.getHeight()));
                        setButtonPosition(newX, newY);

                        if (popupState == POPUP_FROM_RIGHT) {
                            setPopupBackgroundAlpha(1 - event.getRawX() / displaySize.x);
                            setPopupPosition(event.getRawX(), buttonPopup.getY());
                        } else if (popupState == POPUP_FROM_LEFT) {
                            setPopupBackgroundAlpha(event.getRawX() / displaySize.x);
                            setPopupPosition(-displaySize.x + event.getRawX(), buttonPopup.getY());
                        }

                        clickState = false;
                        qrTimer.cancel();
                        hideHintPanel();
                    }
                    break;
            }
            return true;
        }
    }

    private void setButtonPosition(float x, float y) {
        button.setX(x);
        buttonContainer.setY(y);
    }

    private void stickToEdge() {
        int screenWidth = getContext().getResources().getDisplayMetrics().widthPixels;
        if (button.getX() + button.getWidth() / 2 < screenWidth / 2) {
            buttonPosition = BUTTON_LEFT;
        } else {
            buttonPosition = BUTTON_RIGHT;
        }
        stickToEdge(buttonPosition);
    }

    private void stickToEdge(int align) {
        buttonPosition = align;
        if (stopStick) {
            return;
        }
        switch (align) {
            case BUTTON_LEFT:
                buttonXAnimation.animate(0);
                break;
            case BUTTON_RIGHT:
                buttonXAnimation.animate(getWidth() - button.getWidth());
                break;
            default:
                AlanLogger.e("Unknown button alignment");
        }
    }

    public void setButtonAlign(int buttonAlign) {
        stickToEdge(buttonAlign);
    }

    public void setState(AlanState alanState) {
        if (sdk == null) {
            return;
        }
        if (alanState == AlanState.ONLINE) {
            updateButtonVisibility();
            hideHintPanel();
        }
        if (!checkAudioPermissions() && sdk.isMicRequest()) {
            stateHandler.setState(AlanState.NO_PERMISSION);
        } else {
            stateHandler.setState(alanState);
        }
    }

    public void setDisconnectedState() {
        stateHandler.setDisconnectedState();
    }

    public AlanState getState() {
        return stateHandler.getState();
    }

    public void updateButtonVisibility() {
        showHintPanel = !sdk.isHintPanelHidden();
        showAlanButton = !sdk.isButtonHidden();
        if (!showAlanButton) {
            hideButton();
            showHintPanel = false;
        }
        else {
            showButton();
        }
        if (!showHintPanel) {
            hideHintPanel();
        }
        else {
            showHintPanel();
        }
    }

    private int getButtonSizeWithDimension(int buttonSize) {
        int dp = buttonSize + AlanPrefs.DEFAULT_BUTTON_OFFSET;
        final float scale = getContext().getResources().getDisplayMetrics().density;
        int buttonSizeWithDimension = (int) (dp * scale + 0.5f);
        return buttonSizeWithDimension;
    }

    public void resetButtonSize() {
        updateButtonSize(AlanPrefs.DEFAULT_BUTTON_SIZE);
    }

    public void updateButtonSize(int buttonSize) {
        int buttonSizeWithDimension = getButtonSizeWithDimension(buttonSize);
        int defaultSizeWithDimension = getButtonSizeWithDimension(AlanPrefs.DEFAULT_BUTTON_SIZE);
        View view = findViewById(R.id.button_container);
        LayoutParams layoutParams = (LayoutParams) view.getLayoutParams();
        layoutParams.width = buttonSizeWithDimension;
        layoutParams.height = buttonSizeWithDimension;
        view.setLayoutParams(layoutParams);

        View container = findViewById(R.id.container);
        LayoutParams containerLayoutParams = (LayoutParams) container.getLayoutParams();

        containerLayoutParams.height = Math.max(defaultSizeWithDimension, buttonSizeWithDimension);
        container.setLayoutParams(containerLayoutParams);

        View leftText = findViewById(R.id.hint_panel_left);
        LayoutParams leftTextLayoutParams = (LayoutParams) leftText.getLayoutParams();
        leftTextLayoutParams.setMarginEnd(buttonSizeWithDimension);
        leftText.setLayoutParams(leftTextLayoutParams);

        View rightText = findViewById(R.id.hint_panel_right);
        LayoutParams rightTextLayoutParams = (LayoutParams) rightText.getLayoutParams();
        rightTextLayoutParams.setMarginStart(buttonSizeWithDimension);
        rightText.setLayoutParams(rightTextLayoutParams);

        final Handler handler = new Handler(Looper.getMainLooper());
        handler.postDelayed(new Runnable() {
            @Override
            public void run() {
                stickToEdge();
            }
        }, 100);
    }

    public void updateLogos(Map<String,String> customLogos) {
        stateHandler.setCustomLogos(customLogos);
    }

    public void hideHintPanel() {
        if (activeHintPanel == null) {
            return;
        }
        if (activeHintPanel.getAlpha() == 1.0f) {
            hintHideAnimation.startAnimation();
        }
        if (activeHintPanel.getVisibility() != View.GONE) {
            activeHintPanel.setVisibility(View.GONE);
        }
    }

    void showHintPanel() {
        AlanState state = stateHandler.getState();
        if (state == AlanState.OFFLINE || state == AlanState.CONNECTING || state == AlanState.ONLINE) {
            return;
        }
        if (activeHintPanel == null) {
            return;
        }
        activeHintPanel.setAlpha(1.0f);
        if (activeHintPanel.getVisibility() != View.VISIBLE) {
            activeHintPanel.setVisibility(View.VISIBLE);
        }
    }

    private void resetHintPanelState() {
        oldX = button.getX();
        if (!showHintPanel) {
            return;
        }
        if (button.getX() < 100) {
            activeHintPanel = hintPanelRight;
            activeHintText = hintTextRight;
        } else {
            activeHintPanel = hintPanelLeft;
            activeHintText = hintTextLeft;
        }
        hintHideAnimation = new HintHideAnimation(activeHintPanel);
        reLayoutChildren(this);
    }

    public void hideButton() {
        setVisibility(View.GONE);
    }

    public boolean hintPanelVisible() {
        AlanState state = stateHandler.getState();
        if (state == AlanState.OFFLINE || state == AlanState.CONNECTING || state == AlanState.ONLINE) {
            return false;
        }
        return showHintPanel;
    }

    public void showButton() {
        if (!sdk.isInited()) {
            Log.w(TAG, "Button is disabled, skipping show");
            return; //Button is disabled
        }
        setVisibility(View.VISIBLE);
    }

    public void disableButton() {
        hideButton();
    }

    public Alan getSDK() {
        return sdk;
    }

    public void registerCallback(AlanCallback callback) {
        if (sdk != null) {
            buttonCallbacks.add(callback);
            sdk.registerCallback(callback);
        }
    }

    public void removeCallback(AlanCallback callback) {
        if (sdk != null) {
            buttonCallbacks.remove(callback);
            sdk.unregisterCallback(callback);
        }
    }

    public void clearCallbacks() {
        for (AlanCallback callback : buttonCallbacks) {
            removeCallback(callback);
        }
    }

    //================================================================================
    // Wakeword enable/disable/status
    //================================================================================

    public void setWakewordEnabled(boolean enabled) {
        if (sdk == null) {
            return;
        }
        sdk.setWakewordEnabled(enabled);
    }

    public boolean getWakewordEnabled() {
        if (sdk == null) {
            return false;
        }
        return sdk.getWakewordEnabled();
    }

    //================================================================================
    // Alan Popup
    //================================================================================

    class PopupTouchListener implements OnTouchListener {
        @Override
        public boolean onTouch(View v, MotionEvent event) {
            if (popupState != POPUP_OPEN && popupState != POPUP_TO_LEFT && popupState != POPUP_TO_RIGHT) {
                return true;
            }
            float d = startPopupX - event.getRawX();
            switch (event.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    startPopupX = event.getRawX();
                    break;
                case MotionEvent.ACTION_UP:
                    if (popupState == POPUP_TO_LEFT && d < displaySize.x * popupThreshold) {
                        popupState = POPUP_FROM_LEFT;
                    } else if (popupState == POPUP_TO_RIGHT && -d < displaySize.x * popupThreshold) {
                        popupState = POPUP_FROM_RIGHT;
                    }
                    setEndPopupPosition();
                    break;
                case MotionEvent.ACTION_MOVE:
                    if (d > 0) {
                        setPopupBackgroundAlpha(event.getRawX() / displaySize.x);
                        popupState = POPUP_TO_LEFT;
                    } else {
                        setPopupBackgroundAlpha(1 - event.getRawX() / displaySize.x);
                        popupState = POPUP_TO_RIGHT;
                    }
                    setPopupPosition(-d, buttonPopup.getY());
                    break;
            }
            return true;
        }
    }

    private void setupPopup() {
        buttonPopupBackground = findViewById(R.id.alan_popup_background);
        buttonPopup = findViewById(R.id.alan_popup);
        buttonPopup.setX(displaySize.x);
        buttonPopup.setOnTouchListener(new PopupTouchListener());
        popupXAnimation = new PopupXAnimation(buttonPopup);
        popupAlphaAnimation = new PopupAlphaAnimation(buttonPopupBackground);
    }

    private void setEndPopupPosition() {
        if (sdk != null && !sdk.isPopupEnabled()) {
            return;
        }
        switch (popupState) {
            case POPUP_FROM_LEFT:
                if (displaySize.x + buttonPopup.getX() < displaySize.x * popupThreshold) {
                    popupXAnimation.animate(-displaySize.x);
                    popupAlphaAnimation.animate(0);
                    popupState = POPUP_CLOSE;
                } else {
                    popupXAnimation.animate(0);
                    popupAlphaAnimation.animate(1);
                    popupState = POPUP_OPEN;
                }
                break;
            case POPUP_FROM_RIGHT:
                if (buttonPopup.getX() > displaySize.x - displaySize.x * popupThreshold) {
                    popupXAnimation.animate(displaySize.x);
                    popupAlphaAnimation.animate(0);
                    popupState = POPUP_CLOSE;
                } else {
                    popupXAnimation.animate(0);
                    popupAlphaAnimation.animate(1);
                    popupState = POPUP_OPEN;
                }
                break;
            case POPUP_TO_RIGHT:
                popupXAnimation.animate(displaySize.x);
                popupAlphaAnimation.animate(0);
                popupState = POPUP_CLOSE;
                break;
            case POPUP_TO_LEFT:
                popupXAnimation.animate(-displaySize.x);
                popupAlphaAnimation.animate(0);
                popupState = POPUP_CLOSE;
                break;
        }
    }

    private void setPopupPosition(float x, float y) {
        if (sdk != null && !sdk.isPopupEnabled()) {
            return;
        }
        buttonPopup.setX(x);
        buttonPopup.setY(y);
    }

    private void setPopupBackgroundAlpha(float a) {
        if (sdk != null && !sdk.isPopupEnabled()) {
            return;
        }
        buttonPopupBackground.setAlpha(a);
    }

    //================================================================================
    // Client Events
    //================================================================================

    private void sendMicPermissionPromptEvent() {
        try {
            JSONObject params = new JSONObject();
            params.put("micPermissionPrompt", true);
            sendClientEvent(params.toString());
            AlanLogger.w("clientEvent \"micPermissionPrompt\" is sent");
        } catch (JSONException e) {
            AlanLogger.e("sendMicPermissionPromptEvent - failed to create json");
        }
    }

    private void sendMicAllowedEvent() {
        if (!hasMicAllowedEvent) {
            hasMicAllowedEvent = true;
            try {
                JSONObject params = new JSONObject();
                params.put("micAllowed", true);
                sendClientEvent(params.toString());
                AlanLogger.w("clientEvent \"micAllowed\" is sent");
            } catch (JSONException e) {
                AlanLogger.e("sendMicAllowedEvent - failed to create json");
            }
        }
    }

    private void sendFirstTapEvent() {
        if (!hasFirstTapEvent) {
            hasFirstTapEvent = true;
            try {
                JSONObject params = new JSONObject();
                params.put("firstClick", true);
                sendClientEvent(params.toString());
                AlanLogger.w("clientEvent \"firstClick\" is sent");
            } catch (JSONException e) {
                AlanLogger.e("sendFirstTapEvent - failed to create json");
            }
        }
    }

    private void sendClientEvent(final String jsonParams) {
        sdk.sendClientEvent(jsonParams);
    }

    //================================================================================
    // PROXY METHODS
    //================================================================================

    @Override
    public void callProjectApi(String method, String jsonParams) {
        callProjectApi(method, jsonParams, null);
    }

    @Override
    public void callProjectApi(final String method, final String jsonParams, @Nullable final ScriptMethodCallback callback) {
        if (sdk != null) {
            sdk.callProjectApi(method, jsonParams, callback);
        }
    }

    @Override
    public void sendText(String text) {
        if (sdk != null) {
            sdk.sendText(text);
        }
    }

    @Override
    public void sendText(String text, @Nullable ScriptMethodCallback callback) {
        if (sdk != null) {
            sdk.sendText(text, callback);
        }
    }

    @Override
    public void playText(String text, @Nullable ScriptMethodCallback callback) {
        if (sdk != null) {
            sdk.playText(text, callback);
        }
    }

    @Override
    public void playText(String text) {
        playText(text, null);
    }

    @Override
    public void playCommand(String data, @Nullable ScriptMethodCallback callback) {
        if (sdk != null) {
            sdk.playCommand(data, callback);
        }
    }

    @Override
    public void setVisualState(String visualState) {
        if (sdk != null) {
            sdk.setVisualState(visualState);
        }
    }

    @Override
    public void activate(AlanActivateCompletionCallback... activateCallback) {
        if (!checkAudioPermissions()) {
            if (activateCallback.length > 0) {
                activateCompletionCallback = activateCallback[0];
            } else {
                activateCompletionCallback = new AlanActivateCompletionCallback() {
                    @Override
                    public void activateResult(boolean didActivate) {
                    }
                };
            }
            requestAudioPermissions();
        } else if (sdk != null) {
            sdk.activate(activateCallback);
            if (activateCallback.length > 0) {
                activateCompletionCallback = activateCallback[0];
                activateCompletionCallback.activateResult(true);
                activateCompletionCallback = null;
            }
        }
    }

    @Override
    public void deactivate() {
        if (sdk != null) {
            sdk.deactivate();
        }
    }

    @Override
    public boolean isActive() {
        if (sdk == null) {
            return false;
        }
        return sdk.isActive();
    }

}