package com.alan.alansdk.alanbase.recorder;

import android.annotation.TargetApi;
import android.media.AudioFormat;
import android.media.AudioRecord;
import android.media.MediaRecorder;
import android.os.Build;

import com.alan.alansdk.Utils;
import com.alan.alansdk.alanbase.AudioConfig;
import com.alan.alansdk.logging.AlanLogger;

import java.util.Arrays;

@TargetApi(Build.VERSION_CODES.M)
public class RecorderApi23 extends RecorderThread {

    private int bufferSize = 0;
    private float[] floatBuffer;

    private short[] shortBuffer;
    private int samplesToSkip = 0;

    public RecorderApi23(RecordDumpListener callback) {
        super(callback);
    }

    @Override
    protected void skip(int samples) {
        samplesToSkip = samples;
    }

    @Override
    protected boolean prepareRecorder() {
        bufferSize  = AudioRecord.getMinBufferSize(
                AudioConfig.SAMPLE_RATE,
                AudioFormat.CHANNEL_IN_MONO,
                AudioConfig.AUDIO_FORMAT
        );

        if (bufferSize == AudioRecord.ERROR || bufferSize == AudioRecord.ERROR_BAD_VALUE) {
            bufferSize = AudioConfig.SAMPLE_RATE * 2;
        }
        AlanLogger.i("Recorder thread - use buffer size - " + bufferSize);

        if (AudioConfig.AUDIO_FORMAT == AudioFormat.ENCODING_PCM_FLOAT) {
            floatBuffer = new float[bufferSize];
        } else {
            shortBuffer = new short[bufferSize];
        }

        record = new AudioRecord(
                MediaRecorder.AudioSource.MIC,
                AudioConfig.SAMPLE_RATE,
                AudioFormat.CHANNEL_IN_MONO,
                AudioConfig.AUDIO_FORMAT,
                bufferSize
        );

        if (record.getState() != AudioRecord.STATE_INITIALIZED) {
            AlanLogger.e("Audio Record can't initialize!");
            return false;
        }

        shouldContinue = true;
        return shouldContinue;
    }

    @Override
    protected void writeFrames() {
        if (AudioConfig.AUDIO_FORMAT == AudioFormat.ENCODING_PCM_FLOAT) {
            if (floatBuffer.length > 0) {
                if (samplesToSkip > 0) {
                    int skip = Math.min(floatBuffer.length, samplesToSkip);
                    samplesToSkip -= skip;
                    if (skip >= floatBuffer.length) {
                        return;
                    } else {
                        float[] skippedBuffer = Arrays.copyOfRange(floatBuffer, skip, floatBuffer.length);
                        callback.onRecordDump(skippedBuffer);
                    }
                } else {
                    callback.onRecordDump(floatBuffer);
                }
            }
        } else {
            if (shortBuffer.length > 0) {
                if (samplesToSkip > 0) {
                    int skip = Math.min(shortBuffer.length, samplesToSkip);
                    samplesToSkip -= skip;
                    if (skip >= shortBuffer.length) {
                        return;
                    } else {
                        short[] skippedBuffer = Arrays.copyOfRange(shortBuffer, skip, shortBuffer.length);
                        callback.onRecordDump(Utils.shortsToFloat(skippedBuffer));
                    }
                } else {
                    callback.onRecordDump(Utils.shortsToFloat(shortBuffer));
                }
            }
            if (shortBuffer.length > 0) {
                callback.onRecordDump(Utils.shortsToFloat(shortBuffer));
            }
        }
    }

    @Override
    protected int readBuffer() {
        if (AudioConfig.AUDIO_FORMAT == AudioFormat.ENCODING_PCM_FLOAT) {
            return record.read(floatBuffer, 0, floatBuffer.length, AudioRecord.READ_BLOCKING);
        } else {
            return record.read(shortBuffer, 0, shortBuffer.length);
        }
    }
}
