package com.alan.alansdk.button;

import android.app.Activity;
import android.content.Context;
import android.content.ContextWrapper;
import android.graphics.Typeface;
import android.text.Spannable;
import android.text.SpannableString;
import android.text.style.ForegroundColorSpan;
import android.view.View;

import androidx.annotation.ColorRes;
import androidx.annotation.NonNull;

import com.alan.alansdk.AlanCallback;
import com.alan.alansdk.AlanState;
import com.alan.alansdk.R;
import com.alan.alansdk.events.EventOptions;
import com.alan.alansdk.events.EventParsed;
import com.alan.alansdk.events.EventRecognised;

public class AlanStateListener extends AlanCallback {

    private AlanButton button;
    private ColorManager colorManager;
    private HintHiderTask hintHiderTask;

    AlanStateListener(AlanButton button, ColorManager colorManager) {
        this.button = button;
        this.colorManager = colorManager;
        hintHiderTask = new HintHiderTask(button);
        hintHiderTask.start();
    }

    @Override
    public void onAlanStateChanged(@NonNull AlanState alanState) {
        super.onAlanStateChanged(alanState);
        button.setState(alanState);
    }

    @Override
    public void onOptionsReceived(final EventOptions eventOptions) {
        super.onOptionsReceived(eventOptions);
        if (eventOptions.getAndroidSettings() != null) {
            if (eventOptions.getAndroidSettings().isButtonHidden) {
                button.disableButton();
            } else {
                button.showButtonInternal();
            EventOptions.AndroidSettings.ButtonOptions options = eventOptions.getAndroidSettings().buttonOptions;
            if (options != null) {
                colorManager.setIdleColor(options.idleColor.background);
                colorManager.setListenColor(options.listenColor.background);
                colorManager.setProcessColor(options.processColor.background);
                colorManager.setReplyColor(options.replyColor.background);
            }
                button.reset();
            }
        }
    }

    private Activity getActivityFromView(View v) {
        Context context = v.getContext();
        while (context instanceof ContextWrapper) {
            if (context instanceof Activity) {
                return (Activity)context;
            }
            context = ((ContextWrapper)context).getBaseContext();
        }
        return null;
    }

    @Override
    public void onRecognizedEvent(final EventRecognised eventRecognised) {
        super.onRecognizedEvent(eventRecognised);
        if (button.hintPanelVisible() && button.activeHintText != null) {
            button.showHintPanel();
            button.activeHintText.setText(eventRecognised.getText());
            button.activeHintText.setTextColor(getColor(R.color.hint_recognized_text));
            button.activeHintText.setTypeface(null, Typeface.ITALIC);
            button.activeHintPanel.invalidate();
            hintHiderTask.set();
        }

        if (eventRecognised.isFinal()) {
            if (eventRecognised.getReqId() != null &&
                    !eventRecognised.getReqId().isEmpty()) {
                button.getHandler().postDelayed(
                        new Runnable() {
                            @Override
                            public void run() {
                                button
                                        .getSDK()
                                        .takeScreenshot(getActivityFromView(button), eventRecognised.getScreenshotUrl());
                            }
                        }, 100
                );
            }
        }
    }

    @Override
    public void onParsedEvent(final EventParsed eventParsed) {
        super.onParsedEvent(eventParsed);
        if (button.hintPanelVisible()) {
            button.showHintPanel();
            button.activeHintText.setText(generateHintMarkdown(eventParsed));
            button.activeHintText.setTextColor(getColor(R.color.hint_final_text));
            button.activeHintText.setTypeface(null, Typeface.NORMAL);
            button.activeHintPanel.invalidate();
            hintHiderTask.set();
        }

//        if (eventParsed.reqId != null && !eventParsed.reqId.isEmpty()) {
//            button.getSDK().takeScreenshot(getActivityFromView(button), eventParsed.reqId);
//        }
    }

    private SpannableString generateHintMarkdown(EventParsed eventParsed) {
        SpannableString spannable = new SpannableString(eventParsed.getText());
        for (int i = 0; i < eventParsed.getEntities().size(); i++ ) {
            EventParsed.ParsedEntity entity = eventParsed.getEntities().get(i);
            if (entity.index == -1 || entity.value.isEmpty()) {
                continue;
            }
            int color = entity.type.equals("ITEM") ? R.color.hint_entity_item : R.color.hint_entity_word;
            spannable.setSpan(new ForegroundColorSpan(getColor(color)),
                    entity.index,
                    entity.index + entity.value.length(),
                    Spannable.SPAN_EXCLUSIVE_INCLUSIVE);
        }
        return spannable;
    }

    private int getColor(@ColorRes int colorRes) {
        return button.getContext().getResources().getColor(colorRes);
    }

}
