package com.alan.alansdk.button;

import android.app.Activity;
import android.content.Context;
import android.content.ContextWrapper;
import android.graphics.Bitmap;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import com.alan.alansdk.Alan;
import com.alan.alansdk.AlanState;
import com.alan.alansdk.R;
import com.alan.alansdk.alanbase.AudioConfig;
import com.alan.alansdk.button.animations.BackgroundAnimation;
import com.alan.alansdk.button.animations.ButtonColorAnimation;
import com.alan.alansdk.button.animations.InnerBgListenAnimation;
import com.alan.alansdk.button.animations.LogoAnimation;
import com.alan.alansdk.button.animations.OnAnimationEnd;
import com.alan.alansdk.button.animations.SpeakingBackAnimation;
import com.alan.alansdk.prefs.DownloadImageTask;
import com.alan.alansdk.prefs.BitmapCircleMask;
import com.alan.alansdk.prefs.FileCache;
import com.alan.alansdk.Utils;

public class StateHandler {

    private WakeLockManager wakeLockManager;
    private AlanState currentState;
    private SoundManager soundManager;
    private ViewGroup buttonContainer;

    private View shadow;
    private GradientButton backgroundView;
    private View innerBgTriangle;
    private ImageView innerBgCircle;
    private ImageView logoView1;
    private ImageView logoView2;
    private View logoContainer;

    private ColorManager colorManager;

    private LogoAnimation logoAnimation;
    private BackgroundAnimation backgroundAnimation;
    private SpeakingBackAnimation triangleAnimation;
    private SpeakingBackAnimation circleAnimation;
    private InnerBgListenAnimation innerBgListenAnimation;
    private ButtonColorAnimation buttonColorAnimation;

    private String customLogoUrl;

    public StateHandler(Context context, AlanButton buttonView, ColorManager colorManager) {
        customLogoUrl = null;

        wakeLockManager = new WakeLockManager(context);
        soundManager = new SoundManager(context);
        buttonContainer = buttonView;

        shadow = buttonContainer.findViewById(R.id.shadow);
        backgroundView = buttonContainer.findViewById(R.id.button_background);
        logoView1 = buttonContainer.findViewById(R.id.logo_1);
        logoView2 = buttonContainer.findViewById(R.id.logo_2);
        innerBgTriangle = buttonContainer.findViewById(R.id.inner_bg_triangle);
        innerBgCircle = buttonContainer.findViewById(R.id.inner_bg_circle);
        buttonContainer = buttonContainer.findViewById(R.id.button_container);

        logoAnimation = new LogoAnimation(logoView1, logoView2, customLogoUrl);
        backgroundAnimation = new BackgroundAnimation(backgroundView);
        triangleAnimation = new SpeakingBackAnimation(innerBgTriangle, SpeakingBackAnimation.STATE.SPEAKING_BACK);
        circleAnimation = new SpeakingBackAnimation(innerBgCircle, SpeakingBackAnimation.STATE.IDLE);
        innerBgListenAnimation = new InnerBgListenAnimation(innerBgTriangle);
        buttonColorAnimation = new ButtonColorAnimation(backgroundView);

        this.colorManager = colorManager;
    }

    public void setCustomLogo(String customLogo) {
        if (customLogo != null && customLogo.length() > 0) {
            customLogoUrl = customLogo;
            Bitmap cachedBitmap = FileCache.getInstance().retrieveBitmapFromCache(Utils.getActivityFromView(buttonContainer), customLogo);
            if (cachedBitmap != null) {
                Bitmap circleBitmap = BitmapCircleMask.GetBitmapClippedCircle(cachedBitmap);
                logoView1.setImageBitmap(circleBitmap);
                logoView2.setImageBitmap(circleBitmap);
            }
            else {
                new DownloadImageTask((ImageView) buttonContainer.findViewById(R.id.logo_1)).execute(customLogo);
                new DownloadImageTask((ImageView) buttonContainer.findViewById(R.id.logo_2)).execute(customLogo);
            }
        }
        else {
            customLogoUrl = null;
            logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
            logoView2.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
        }
        logoAnimation = new LogoAnimation(logoView1, logoView2, customLogoUrl);
    }

    public void setState(AlanState alanState) {
        if (currentState == AlanState.REPLY && alanState == AlanState.LISTEN) {
            Alan.getInstance(backgroundView.getContext()).skipMillis(AudioConfig.SKIP_MILLIS_ON_START);
            soundManager.playNext();
        }
        if (currentState == AlanState.LISTEN && alanState == AlanState.ONLINE) {
            soundManager.playOff();
        }
        if (currentState == AlanState.ONLINE && alanState == AlanState.LISTEN) {
            Alan.getInstance(backgroundView.getContext()).skipMillis(AudioConfig.SKIP_MILLIS_ON_START);
            soundManager.playOn();
        }
        currentState = alanState;
        switch (alanState) {
            case ONLINE:
                setIdleState();
                break;
            case OFFLINE:
            case CONNECTING:
                setDisconnectedState();
                break;
            case LISTEN:
                setListenState();
                break;
            case PROCESS:
                setUnderstoodState();
                break;
            case REPLY:
                setSpeakBackState();
                break;
            case UNKNOWN:
                break;
        }
    }

    void setDisconnectedState() {
        buttonColorAnimation.start(colorManager.getDisconnectedColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.GONE);

        logoAnimation.stopAnimation();
        backgroundAnimation.stopAnimation();
        triangleAnimation.stopAnimation();
        circleAnimation.stopAnimation();
        innerBgListenAnimation.stopAnimation();

        logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_no_network));
        logoView1.setAlpha(1.0f);
        logoView2.setAlpha(0.0f);

        wakeLockManager.releaseWakeLock();
    }

    void setIdleState() {
        buttonColorAnimation.start(colorManager.getIdleColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.GONE);

        logoAnimation.stopAnimation();
        backgroundAnimation.startNormalAnimation();
        triangleAnimation.startRespondingAnimation();
        circleAnimation.startSpeakingBackAnimation();
        innerBgListenAnimation.stopAnimation();

        if (customLogoUrl != null) {
            setCustomLogo(customLogoUrl);
        }
        else {
            logoView1.setImageDrawable(buttonContainer.getContext().getResources().getDrawable(R.drawable.button_logo_state_01));
        }

        logoView1.setAlpha(1.0f);
        logoView2.setAlpha(0.0f);

        wakeLockManager.releaseWakeLock();
    }

    private void setListenState() {
        wakeLockManager.acquireWakeLock();

        buttonColorAnimation.start(colorManager.getListenColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.VISIBLE);
        logoAnimation.startAnimation();
        backgroundAnimation.startRespondingAnimation();
        triangleAnimation.startRespondingAnimation();
        circleAnimation.startSpeakingBackAnimation();
        innerBgListenAnimation.startAnimation();
    }

    private void setUnderstoodState() {
        buttonColorAnimation.start(colorManager.getProcessColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.VISIBLE);
    }

    private void setSpeakBackState() {
        buttonColorAnimation.start(colorManager.getReplyColor(), new OnAnimationEnd() {
            @Override
            public void onAnimationEnd() {
            }
        });

        innerBgTriangle.setVisibility(View.VISIBLE);

        backgroundAnimation.startNormalAnimation();
        triangleAnimation.startSpeakingBackAnimation();
        circleAnimation.startRespondingAnimation();
        innerBgListenAnimation.stopAnimation();
    }

    void setTouchedState() {
        if (currentState != AlanState.ONLINE) {
            innerBgTriangle.setVisibility(View.GONE);
        }
        logoView1.setScaleX(0.9f);
        logoView1.setScaleY(0.9f);
        logoView2.setScaleX(0.9f);
        logoView2.setScaleY(0.9f);
        shadow.setVisibility(View.GONE);
        backgroundView.setScaleX(0.9f);
        backgroundView.setScaleY(0.9f);
    }


    void setUntouchedState() {
        if (currentState != AlanState.ONLINE) {
            innerBgTriangle.setVisibility(View.VISIBLE);
        }
        logoView1.setScaleX(1.0f);
        logoView1.setScaleY(1.0f);
        logoView2.setScaleX(1.0f);
        logoView2.setScaleY(1.0f);
        shadow.setVisibility(View.VISIBLE);
        backgroundView.setScaleX(1.0f);
        backgroundView.setScaleY(1.0f);
    }

    public AlanState getState() {
        return currentState;
    }
}
