package app.appnomix.sdk.internal

import android.accessibilityservice.AccessibilityService
import android.content.BroadcastReceiver
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.view.accessibility.AccessibilityEvent
import android.view.accessibility.AccessibilityWindowInfo
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.domain.CouponCoordinator
import app.appnomix.sdk.internal.domain.browsers.BrowserResolver
import app.appnomix.sdk.internal.domain.browsers.DeviceBrowser
import app.appnomix.sdk.internal.domain.model.Coupon
import app.appnomix.sdk.internal.domain.model.Demand
import app.appnomix.sdk.internal.ui.PopupDisplay
import app.appnomix.sdk.internal.ui.WebviewWindow
import app.appnomix.sdk.internal.utils.SLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.time.Instant
import kotlin.time.Duration.Companion.seconds

private const val MILLIS_BETWEEN_EVENTS = 300

internal class CouponsAccessibilityService : AccessibilityService(), CouponAction {
    private val clipboardManager: ClipboardManager by lazy {
        getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
    }
    private val coordinator: CouponCoordinator by lazy { Deps.coordinator() }
    private var screenOffReceiver: BroadcastReceiver? = null
    private val scope = CoroutineScope(Dispatchers.Main)
    private val popupWindow = PopupDisplay(this)
    private var lastProcessedEventTs = 0L

    private var itemBeingApplied: Any? = null
    private var currentlyMatchedItem: Any? = null
    private val sdkConfig: SdkConfig by lazy { Deps.sdkConfig() }
    private lateinit var webviewWindow: WebviewWindow

    init {
        this.screenOffReceiver = object : BroadcastReceiver() {
            override fun onReceive(context: Context, intent: Intent?) {
                if (intent?.action == Intent.ACTION_SCREEN_OFF) {
                    popupWindow.hide()
                }
            }
        }
    }

    override fun onCreate() {
        super.onCreate()
        webviewWindow = WebviewWindow(this)
    }

    override fun onAccessibilityEvent(event: AccessibilityEvent) {
        SLog.i("event: ${AccessibilityEvent.eventTypeToString(event.eventType)}")
        when (event.eventType) {
            AccessibilityEvent.TYPE_WINDOW_STATE_CHANGED, AccessibilityEvent.TYPE_WINDOW_CONTENT_CHANGED -> {
                if (Instant.now().toEpochMilli() - lastProcessedEventTs > MILLIS_BETWEEN_EVENTS) {
                    lastProcessedEventTs = Instant.now().toEpochMilli()
                    processEvent()
                }
            }

            AccessibilityEvent.TYPE_VIEW_SCROLLED -> {
                webviewWindow.updateScroll(event.scrollX, event.scrollY)
            }

            AccessibilityEvent.TYPE_VIEW_CLICKED -> {
                processClickEvent(event)
            }
            else -> {
                SLog.i("event: ${event.eventType} with id: ${event.source?.contentDescription}")
            }
        }
    }

    private fun processClickEvent(event: AccessibilityEvent) {
        try {
            val text = event.source?.text
            val contentDescription = event.source?.contentDescription
            val viewIdResourceName = event.source?.viewIdResourceName
            val className = event.source?.className
            SLog.i("clicked text: $text")
            SLog.i("clicked contentDescription: $contentDescription")
            SLog.i("clicked viewIdResourceName: $viewIdResourceName")
            SLog.i("clicked className: $className")

            webviewWindow.click(viewIdResourceName, contentDescription, text, className)
        } catch (t: Throwable) {
            SLog.e("something went wrong when processing click", t)
        }
    }

    private fun processEvent() {
        try {
            val rootNode = rootInActiveWindow ?: return
            val packageName = rootNode.packageName ?: ""
            val browser = BrowserResolver.getBrowser(packageName.toString())
            if (browser == null) {
                popupWindow.hide()
                return
            }

            val isValidBrowser = BrowserResolver.supportedBrowserPackages().contains(packageName)
            val addressText = browser.getAddressText(rootNode) ?: ""
            webviewWindow.update(addressText, browser.locateWebviewBounds(rootNode))

            val matchedItems = coordinator.matchAgainstUrl(addressText)
            val hasMatchedItem = matchedItems.isNotEmpty()
            currentlyMatchedItem = matchedItems.firstOrNull()

            val hasMatchedCoupons = currentlyMatchedItem is Coupon

            if (isValidBrowser
                && browser.isIdle(rootNode)
                && hasMatchedItem
                && !isKeyboardOpened()
            ) {
                val item = matchedItems.first()
                if (coordinator.shouldAlwaysApply(item)) {
                    applyItem(
                        item = item,
                        browser = browser,
                        currentUrl = addressText
                    )
                } else {
                    onEventMatched(
                        item = item,
                        addressText = addressText,
                        onItemAction = {
                            when (item) {
                                is Demand -> {
                                    coordinator.markAsAlwaysApply(item)
                                }

                                is Coupon -> {
                                    applyItem(
                                        item = item,
                                        browser = browser,
                                        currentUrl = addressText
                                    )
                                }
                            }
                        },
                        onItemIgnore = {
                            coordinator.snoozeItem(matchedItems.first())
                        })
                }

                if (hasMatchedCoupons) {
                    if (browser.isOnCheckoutPage(addressText)) {
                        val applied = browser.applyCouponCodeOnce(
                            node = rootNode,
                            coupon = (currentlyMatchedItem as Coupon).code,
                            url = addressText
                        )
                        if (applied) {
                            popupWindow.hide()
                        }
                    }
                }
            } else if (itemBeingApplied == null) {
                popupWindow.hide()
            }
        } catch (e: Exception) {
            SLog.e("something went wrong", e)
        }
    }

    private fun applyItem(
        item: Any,
        browser: DeviceBrowser?,
        currentUrl: String? = null
    ) {
        itemBeingApplied = item
        coordinator.snoozeItem(item)
        var urlToOpen: String? = null
        when (item) {
            is Demand -> {
                urlToOpen = item.redirectUrl
            }

            is Coupon -> {
                urlToOpen = item.clickUrl
                val clip = ClipData.newPlainText("coupon", item.code)
                clipboardManager.setPrimaryClip(clip)
            }
        }
        urlToOpen?.let {
            if (browser != null) {
                browser.openUrl(
                    context = this@CouponsAccessibilityService,
                    destinationUrl = urlToOpen,
                    currentUrl = currentUrl,
                    sdkConfig = sdkConfig
                )
            } else {
                DeviceBrowser.openUrl(
                    context = this@CouponsAccessibilityService,
                    url = urlToOpen
                )
            }
        }

        scope.launch {
            // simulate an artificial 'applying' operation
            delay(2.seconds)
            itemBeingApplied = null
            popupWindow.hide()
        }
    }

    private fun onEventMatched(
        item: Any,
        addressText: String,
        onItemAction: () -> Unit,
        onItemIgnore: () -> Unit
    ) {
        SLog.d("matched $addressText")
        popupWindow.show(
            item = item,
            onActivateCoupon = onItemAction,
            onIgnoreCoupon = onItemIgnore
        )
    }

    override fun applyCoupon(id: String) {
        coordinator.getById(id)?.let {
            applyItem(it, null)
        }
    }

    override fun dismissActiveItem() {
        currentlyMatchedItem?.let {
            coordinator.snoozeItem(it)
        }
    }

    override fun onInterrupt() {
    }

    private fun isKeyboardOpened(): Boolean {
        val windowInfoList = windows
        for (windowInfo in windowInfoList) {
            if (windowInfo.type == AccessibilityWindowInfo.TYPE_INPUT_METHOD) {
                return true
            }
        }
        return false
    }

    override fun onServiceConnected() {
        super.onServiceConnected()
        CouponAction.ACTIVE_INSTANCE = this
        val filter = IntentFilter(Intent.ACTION_SCREEN_OFF)
        this.registerReceiver(this.screenOffReceiver, filter)
    }

    override fun onDestroy() {
        super.onDestroy()
        CouponAction.ACTIVE_INSTANCE = null
        try {
            this.unregisterReceiver(screenOffReceiver)
        } catch (ignored: Exception) {
        }
    }
}
