package app.appnomix.sdk.internal.data.network

import DomainDataDto
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.data.network.model.ConfigContentDto
import app.appnomix.sdk.internal.data.network.model.CouponDto
import app.appnomix.sdk.internal.data.network.model.DemandDto
import app.appnomix.sdk.internal.data.network.model.TimestampSerializer
import app.appnomix.sdk.internal.utils.SLog
import com.squareup.picasso.BuildConfig
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.engine.android.Android
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.defaultRequest
import io.ktor.client.plugins.logging.LogLevel
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.request.get
import io.ktor.client.request.headers
import io.ktor.http.ContentType
import io.ktor.http.URLProtocol
import io.ktor.http.contentType
import io.ktor.http.path
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import kotlinx.serialization.modules.SerializersModule
import kotlinx.serialization.modules.polymorphic
import java.time.LocalDateTime
import kotlin.time.Duration.Companion.seconds

val jsonEngine = Json {
    ignoreUnknownKeys = true
    prettyPrint = true
    isLenient = true
    encodeDefaults = true
    serializersModule = SerializersModule {
        contextual(LocalDateTime::class, TimestampSerializer())
        polymorphic(DomainDataDto::class) {
            defaultDeserializer { DomainDataDto.Unknown.serializer() }
        }
    }
}

class SaversLeagueApi(
    private val defaultHost: String = "api.appnomix.app"
) {
    private val sdkConfig: SdkConfig by lazy { Deps.sdkConfig() }

    private val client = HttpClient(Android) {
        engine {
            connectTimeout = 30.seconds.inWholeMilliseconds.toInt()
            socketTimeout = 30.seconds.inWholeMilliseconds.toInt()
        }
        install(Logging) {
            logger = object : io.ktor.client.plugins.logging.Logger {
                override fun log(message: String) {
                    SLog.i(message)
                }
            }
            level = if (BuildConfig.DEBUG) LogLevel.ALL else LogLevel.HEADERS
        }
        defaultRequest {
            url {
                protocol = URLProtocol.HTTPS
                host = defaultHost
                path("api/v1/")
            }
            contentType(ContentType.Application.Json)
        }
        install(ContentNegotiation) {
            json(jsonEngine)
        }
    }

    suspend fun getConfig(): ConfigContentDto {
        val response: ConfigContentDto =
            client.get("client/configuration") {
                headers {
                    SLog.d("Fetching config for clientId: ${sdkConfig.clientId} / ${sdkConfig.authToken}")
                    headers.append("client-id", sdkConfig.clientId)
                    headers.append("auth-token", sdkConfig.authToken)
                }
            }
                .body()
        return response
    }

    suspend fun getCoupons(countryCode: String?, forHomepage: Boolean): DataResponse<CouponDto> {
        val response: DataResponse<CouponDto> =
            client.get("promo/${sdkConfig.campaignId}") {
                url {
                    countryCode?.let {
                        parameters.append("country_code", countryCode)
                    }
//                    parameters.append("country_code", "us")
                    parameters.append("homepage", forHomepage.toString())
                }
            }.body()
        return response
    }

    suspend fun getOnDemandRedirects(countryCode: String?): DataResponse<DemandDto> {
        val response: DataResponse<DemandDto> =
            client.get("demand/${sdkConfig.campaignId}") {
                url {
                    countryCode?.let {
                        parameters.append("country_code", countryCode)
                    }
//                    parameters.append("country_code", "us")
                }
            }.body()
        return response
    }

    fun getCouponImageUrl(brandName: String): String {
        return "https://i.v2i8b.com/$brandName"
    }

    suspend fun getBrand(baseDomain: String): DataResponse<DomainDataDto> {
        return client.get("coupon/config/$baseDomain") {
            url {
                sdkConfig.campaignId?.let { campaignId ->
                    parameters.append("campaign_id", campaignId)
                }
            }
        }.body()
    }
}


@Serializable
data class DataResponse<T>(
    @SerialName("data") val data: List<T>
)
