package app.appnomix.sdk.internal.domain.browsers

import android.content.Context
import android.content.Intent
import android.net.Uri
import android.view.accessibility.AccessibilityNodeInfo
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.utils.SLog

internal class ChromeBrowser : DeviceBrowser() {

    override fun browserPackage() = "com.android.chrome"

    override fun getAddressText(nodeInfo: AccessibilityNodeInfo): String? {
        return findInfos(nodeInfo = nodeInfo, tag = "${browserPackage()}:id/url_bar")
            .firstOrNull()
            ?.text
            ?.toString()
    }

    override fun openUrl(
        context: Context,
        currentUrl: String?,
        destinationUrl: String,
        sdkConfig: SdkConfig
    ) {
        try {
            val updatedDestinationUrl = if (currentUrl != null) replaceRedirectUrlFromFullUrl(
                fullUrl = destinationUrl,
                currentUrl = currentUrl,
                sdkConfig = sdkConfig
            ) else destinationUrl
            val intent = Intent(Intent.ACTION_VIEW, Uri.parse(updatedDestinationUrl))
            intent.setPackage(browserPackage())
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(intent)
        } catch (e: Exception) {
            SLog.e("Unable to open url $destinationUrl", e)
        }
    }

    override fun isIdle(nodeInfo: AccessibilityNodeInfo): Boolean {
        return !isSettingsActive(nodeInfo) && !isShareActive(nodeInfo) && !isOnHomePage(nodeInfo)
    }

    private fun isOnHomePage(nodeInfo: AccessibilityNodeInfo): Boolean {
        return findInfos(nodeInfo, "${browserPackage()}:id/lens_camera_button").isNotEmpty()
                && findInfos(nodeInfo, "${browserPackage()}:id/voice_search_button").isNotEmpty()
    }

    private fun isShareActive(nodeInfo: AccessibilityNodeInfo): Boolean {
        return findInfos(nodeInfo, "${browserPackage()}:id/share_sheet_scrollview").isNotEmpty()
    }

    private fun isSettingsActive(nodeInfo: AccessibilityNodeInfo): Boolean {
        val newTabNodes = findInfos(nodeInfo, "${browserPackage()}:id/new_tab_menu_id")
        val newTabNodesAndroid7 = findInfos(nodeInfo, "${browserPackage()}:id/new_tab_button")
        val allTabsNodes = findInfos(nodeInfo, "${browserPackage()}:id/tab_list_view")
        val goForwardButton = findInfos(nodeInfo, "${browserPackage()}:id/button_one")
        val bookmarkButton = findInfos(nodeInfo, "${browserPackage()}:id/button_two")
        val siteInfoButton = findInfos(nodeInfo, "${browserPackage()}:id/button_three")
        val refreshButton = findInfos(nodeInfo, "${browserPackage()}:id/button_four")
        val tabSwitcherToolbar = findInfos(nodeInfo, "${browserPackage()}:id/tab_switcher_toolbar")
        val searchHistoryCloseNode = findInfos(nodeInfo, "${browserPackage()}:id/close_menu_id")

        return (newTabNodes.isEmpty() && (allTabsNodes.isEmpty() || !findByClass(
            allTabsNodes,
            "android.widget.GridView"
        )) &&
                newTabNodesAndroid7.isEmpty() && (goForwardButton.isEmpty() || bookmarkButton.isEmpty() ||
                siteInfoButton.isEmpty() || refreshButton.isEmpty()) && searchHistoryCloseNode.isEmpty() &&
                tabSwitcherToolbar.isEmpty()).not()
    }

}