package app.appnomix.sdk.internal.domain.model

import app.appnomix.sdk.external.ExternalCoupon
import app.appnomix.sdk.internal.Deps
import java.time.LocalDateTime

data class Coupon(
    val id: String,
    val code: String,
    val endDate: LocalDateTime,
    val startDate: LocalDateTime,
    val clickUrl: String,
    val brandDomain: String,
    val isForHomepage: Boolean,
    val brandImageUrl: String?,
    val description: String,
    val snoozeTime: LocalDateTime?,
    val validCountriesCodes: List<String>,
) {
    private val regex: Regex by lazy {
        Regex(
            pattern = "^(?:https?://(?:www\\.)?)?(.*\\.)*$brandDomain.*",
            options = setOf(RegexOption.IGNORE_CASE)
        )
    }

    fun matchesTarget(address: String): Boolean {
        return regex.matches(address)
    }

    fun matchesLocation(countryCode: String?): Boolean {
        if (countryCode == null) {
            return true
        }
        return validCountriesCodes.find {
            it.lowercase() == countryCode.lowercase()
        } != null
    }

    fun isValid(): Boolean {
        return LocalDateTime.now() in startDate..endDate
    }

    fun isSnoozed(): Boolean {
        if (snoozeTime == null) return false
        return LocalDateTime.now()
            .isBefore(snoozeTime.plusSeconds(Deps.sdkConfig().popupSnoozeTimeMillis))
    }
}

fun Coupon.toExternalCoupon(): ExternalCoupon {
    return ExternalCoupon(
        id = this.id,
        name = this.brandDomain,
        description = this.description,
        baseDomain = this.brandDomain,
        redirectUrl = this.clickUrl,
        code = this.code,
        thumbnailUrl = this.brandImageUrl
    )
}