package app.appnomix.sdk.internal.ui

import android.content.Context
import android.graphics.Bitmap
import android.util.AttributeSet
import android.view.accessibility.AccessibilityNodeInfo
import android.webkit.WebView
import android.webkit.WebViewClient
import app.appnomix.sdk.internal.utils.SLog
import io.ktor.util.collections.ConcurrentMap
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.json.JSONArray
import java.time.Instant

class OverlayWebview(context: Context, attrs: AttributeSet?) : WebView(context, attrs) {
    private val scope = CoroutineScope(Dispatchers.Default)
    private val accessibilityNodeMap = ConcurrentMap<HtmlElementKey, AccessibilityNodeInfo>()

    init {
        getSettings().javaScriptEnabled = true
        getSettings().domStorageEnabled = true
        webViewClient = object : WebViewClient() {
            override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                super.onPageStarted(view, url, favicon)
                SLog.i("onPageStarted: $url")
            }

            override fun onPageFinished(view: WebView, url: String?) {
                super.onPageFinished(view, url)
                SLog.i("onPageFinished: $url")
                accessibilityNodeMap.clear()
                transformPageContentToAccessibilityNodes()
            }
        }
    }

    private fun transformPageContentToAccessibilityNodes() {
        evaluateJavascript(
            """
        (function() {
            var elements = $ELEMENT_JS_SELECTOR
            var result = [];
            for (var i = 0; i < elements.length; i++) {
                var element = elements[i];
                var id = element.id.trim() || null;
                var ariaLabel = element.getAttribute("aria-label") || null;
                var title = element.getAttribute("title") || null;
                var role = element.getAttribute("role") || null;
                var text = element.innerText.trim() || ariaLabel;
                result.push({id: id, text: text, ariaLabel: ariaLabel, title: title});
            }
            return JSON.stringify(result);
        })();
        """
        ) { json ->
            scope.launch {
                if (json.isNullOrEmpty()) return@launch
                try {
                    val cleanedJson = json.replace("""^"(.*)"$""".toRegex(), "$1")
                        .replace("""\\(.)""".toRegex(), "$1")
                        .trim()

                    if (cleanedJson.isEmpty()) return@launch

                    val elements = JSONArray(cleanedJson)
                    for (i in 0 until elements.length()) {
                        val element = elements.getJSONObject(i)
                        val id = if (element.isNull("id")) null else element.getString("id")
                        val text = if (element.isNull("text")) null else element.getString("text")
                        val title =
                            if (element.isNull("title")) null else element.getString("title")
                        val ariaLabel =
                            if (element.isNull("ariaLabel")) null else element.getString("ariaLabel")

                        val textToUse = text?.replace("\n", " ")

                        if (!textToUse.isNullOrEmpty()) {
                            val key = HtmlElementKey(
                                id = id,
                                text = textToUse,
                                title = title,
                                ariaLabel = ariaLabel
                            )
                            val nodeInfo = createAccessibilityNodeInfo(id, textToUse)
                            accessibilityNodeMap[key] = nodeInfo
                        }
                    }
                    SLog.i("mapped: ${accessibilityNodeMap.keys}")
                } catch (e: Exception) {
                    SLog.e("something went wrong while parsing dom: $json", e)
                }
            }
        }
    }

    private fun createAccessibilityNodeInfo(id: String?, text: String?): AccessibilityNodeInfo {
        val nodeInfo = AccessibilityNodeInfo.obtain()
        nodeInfo.text = text
        nodeInfo.viewIdResourceName = id
        return nodeInfo
    }

    fun click(
        viewIdResourceName: String?,
        contentDescription: CharSequence?,
        text: CharSequence?
    ) {
        try {
            val contentDescriptionString = contentDescription.toString().lowercase()
            val textString = text.toString().lowercase()

            // first, match by accessibility text
            var matches = accessibilityNodeMap.filterValues {
                (textString.isNotEmpty() && it.text.toString().lowercase() == textString) ||
                        (!viewIdResourceName.isNullOrEmpty() && it.viewIdResourceName.lowercase() == viewIdResourceName)
            }.keys.toList()

            if (matches.isEmpty()) {
                // then, try to match by html tags
                matches = accessibilityNodeMap.keys.filter {
                    val potentialTextMatches = listOfNotNull(
                        it.text?.lowercase(),
                        it.title?.lowercase(),
                        it.ariaLabel?.lowercase()
                    )

                    (it.id != null && viewIdResourceName.equals(it.id, ignoreCase = true)) ||
                            potentialTextMatches.contains(contentDescriptionString) ||
                            potentialTextMatches.contains(textString)
                }
            }

            if (matches.size > 1) {
                SLog.i("match for (viewIdResourceName=$viewIdResourceName, contentDescription=$contentDescription, text=$text) has too many matches ($matches). skipping.")
            } else if (matches.size == 1) {
                val htmlTag = matches.first()
                SLog.i("match for (viewIdResourceName=$viewIdResourceName, contentDescription=$contentDescription, text=$text) is: $htmlTag")

                val textToMatch = htmlTag.text
                val idToMatch = htmlTag.id
                evaluateJavascript(
                    """
        (function() {
            var buttons = $ELEMENT_JS_SELECTOR
            for (var i = 0; i < buttons.length; i++) {
                var button = buttons[i];
                // Replace '\n' characters with ' ' in button's innerText
                var buttonText = button.innerText.replace(/\n/g, ' ');
                
                if (button.id.trim() === '$idToMatch' || buttonText.trim() === '$textToMatch') {
                    button.click();
                    return true; // Indicate success
                }
            }
            return false; // Indicate failure
        })();
    """.trimIndent()
                ) {
                    val success = it?.toBoolean() ?: false
                    if (success) {
                        SLog.i("match for (viewIdResourceName=$viewIdResourceName, contentDescription=$contentDescription, text=$text) was clicked")
                    } else {
                        SLog.i("match for (viewIdResourceName=$viewIdResourceName, contentDescription=$contentDescription, text=$text) was NOT clicked")
                    }
                }
            } else {
                SLog.i("match for (viewIdResourceName=$viewIdResourceName, contentDescription=$contentDescription, text=$text) is MISSING")
            }

        } catch (e: Exception) {
            SLog.e("something went wrong while trying to click", e)
        }
    }

    private companion object {
        const val ELEMENT_JS_SELECTOR =
            "document.querySelectorAll('button', [role='button']);"
    }
}

private data class HtmlElementKey(
    val generatedId: Int = Instant.now().nano,
    val id: String?,
    val text: String?,
    val ariaLabel: String?,
    val title: String?,
)