package app.appnomix.sdk.internal.ui

import android.content.Context
import android.graphics.PixelFormat
import android.graphics.Rect
import android.os.Build
import android.view.LayoutInflater
import android.view.View
import android.view.WindowInsets
import android.view.WindowManager
import app.appnomix.sdk.R
import app.appnomix.sdk.internal.domain.browsers.ensureHttpsPrefix
import app.appnomix.sdk.internal.utils.SLog

const val DEBUG_WEBVIEW_ENABLED = false

internal class WebviewWindow(private val context: Context) {
    private val windowManager: WindowManager by lazy {
        context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
    }
    private var webView: OverlayWebview? = null
    private var lastBounds: Rect? = null
    private var lastUrl: String? = null
    private var windowInsets: WindowInsets? = null

    fun update(
        url: String,
        bounds: Rect?
    ) {
        if (!DEBUG_WEBVIEW_ENABLED) return
        if (url.isEmpty()) return
        if (bounds == lastBounds && url == lastUrl) return
        if (bounds == null) {
            if (webView?.parent != null)
                windowManager.removeView(webView)
            return
        }
        if (webView == null) {
            webView =
                LayoutInflater.from(context).inflate(R.layout.webview, null) as? OverlayWebview
            webView?.settings?.javaScriptEnabled = true
        }

        val topOffset = windowInsets?.topInset ?: 0
        val bottomOffset = windowInsets?.bottomInset ?: 0

        val params = WindowManager.LayoutParams(
            bounds.width(),
            bounds.height(),
            bounds.left,
            bounds.top - topOffset - bottomOffset,
            WindowManager.LayoutParams.TYPE_ACCESSIBILITY_OVERLAY,
            WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE
                    or WindowManager.LayoutParams.FLAG_NOT_TOUCHABLE,
            PixelFormat.TRANSLUCENT
        )

        if (webView?.parent != null) {
            windowManager.removeView(webView)
        }

        webView?.setOnApplyWindowInsetsListener { view, insets ->
            if (windowInsets == null || (insets.topInset > 0 || insets.bottomInset > 0))
                windowInsets = insets
            insets
        }
        windowManager.addView(webView, params)

        webView?.loadUrl(url.ensureHttpsPrefix())
        lastUrl = url
        lastBounds = bounds
    }

    fun click(
        viewIdResourceName: String?,
        contentDescription: CharSequence?,
        text: CharSequence?,
        className: CharSequence?,
    ) {
        if (!DEBUG_WEBVIEW_ENABLED) return
        webView?.click(
            viewIdResourceName = viewIdResourceName,
            contentDescription = contentDescription,
            text = text
        )
    }

    fun updateScroll(x: Int, y: Int) {
        if (!DEBUG_WEBVIEW_ENABLED) return
        SLog.i("scroll to: x=$x y=$y")
        webView?.scrollTo(x, y)
    }

    fun hide() {
        webView?.visibility = View.GONE
    }
}

@Suppress("DEPRECATION")
val WindowInsets.topInset: Int
    get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
        this.getInsets(WindowInsets.Type.systemBars()).top
    } else {
        this.systemWindowInsetTop
    }

@Suppress("DEPRECATION")
val WindowInsets.bottomInset: Int
    get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
        this.getInsets(WindowInsets.Type.systemBars()).bottom
    } else {
        this.systemWindowInsetBottom
    }
