package app.appnomix.sdk.internal.ui.onboarding

import android.app.Dialog
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.ImageView
import android.widget.TextView
import androidx.activity.enableEdgeToEdge
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import app.appnomix.sdk.R
import app.appnomix.sdk.external.CouponsSdkFacade
import app.appnomix.sdk.external.SdkRequirement
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.data.SdkConfig

private const val PROMINENT_DIALOG_ENABLED = false

class OnboardingActivity : AppCompatActivity() {
    private val sdkConfig: SdkConfig by lazy { Deps.sdkConfig() }
    private var currentOnboardingScreen: OnboardingScreenType? = null
    private var contentContainer: ViewGroup? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_onboarding)
        enableEdgeToEdge()

        val mainContainer = findViewById<View>(R.id.main)
        contentContainer = mainContainer.findViewById(R.id.content_layout)
        ViewCompat.setOnApplyWindowInsetsListener(mainContainer) { v, insets ->
            val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom)
            insets
        }
        if (intent.action == TO_ACCESSIBILITY_STEP_ACTION) {
            sdkConfig.onboardingFinished = false
            currentOnboardingScreen = OnboardingScreenType.GET_STARTED_2
        }
        moveToNextScreen()
    }

    override fun onResume() {
        super.onResume()
        moveToLastScreenIfPossible(currentOnboardingScreen)
    }

    private fun moveToLastScreenIfPossible(fromScreen: OnboardingScreenType?) {
        val unmetRequirements = CouponsSdkFacade.unmetRequirements()
        if (fromScreen in listOf(
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_EXPLAINER,
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_1,
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2,
            )
            && !unmetRequirements.contains(SdkRequirement.ACCESSIBILITY)
        ) {
            currentOnboardingScreen = OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2
            moveToNextScreen()
        }
    }

    private fun moveToNextScreen() {
        val nextScreenOrder = currentOnboardingScreen?.order?.plus(1) ?: 0
        val nextScreen = OnboardingScreenType.entries.find { it.order == nextScreenOrder }
        moveToLastScreenIfPossible(nextScreen)

        if (nextScreen != null) {
            val screenView = createView(nextScreen)
            contentContainer?.removeAllViews()
            contentContainer?.addView(screenView)
            currentOnboardingScreen = nextScreen
        } else {
            endOnboarding()
        }
    }

    private fun createView(screenType: OnboardingScreenType): View {
        val view = View.inflate(this, R.layout.onboarding, null)
        ONBOARDING_SCREENS[screenType]?.let { screenData ->
            val logoImageView = view.findViewById<ImageView>(R.id.logo_imageView)
            val titleTextView = view.findViewById<TextView>(R.id.title_textView)
            val contentViewContainer = view.findViewById<ViewGroup>(R.id.template_layout)
            val primaryButton = view.findViewById<Button>(R.id.primary_button)
            val secondaryButton = view.findViewById<Button>(R.id.secondary_button)

            sdkConfig.appIconRes?.let {
                val result = kotlin.runCatching { ContextCompat.getDrawable(this, it) }
                val drawable = result.getOrDefault(packageManager.getApplicationIcon(packageName))
                logoImageView?.setImageDrawable(drawable)
            }

            primaryButton.setText(screenData.mainButtonTitleRes)
            primaryButton.setOnClickListener {
                if (currentOnboardingScreen?.order == OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2.order) {
                    showProminentDialog()
                } else {
                    handlePrimaryAction(screenType)
                }
            }

            if (screenData.secondaryButtonTitleRes != null) {
                val content = screenData.contentWithAppName(
                    context = this,
                    stringRes = screenData.secondaryButtonTitleRes
                )
                secondaryButton.text = content
                if (screenData.secondaryButtonTextColorRes != null) {
                    secondaryButton.setTextColor(
                        ContextCompat.getColor(
                            this,
                            screenData.secondaryButtonTextColorRes
                        )
                    )
                }
                secondaryButton.visibility = View.VISIBLE
                secondaryButton.setOnClickListener {
                    handleSecondaryAction(screenType)
                }

                secondaryButton.setBackgroundResource(screenData.secondaryButtonDrawableRes)
            } else {
                secondaryButton.visibility = View.GONE
            }

            screenData.titleRes?.let {
                val content = screenData.contentWithAppName(context = this, stringRes = it)
                titleTextView.text = content
            }

            val contentLayoutResToUse = when (screenData.content) {
                is OnboardingContent.ImageText -> R.layout.onboarding_content_image_text
                is OnboardingContent.TextImage -> R.layout.onboarding_content_text_image
                is OnboardingContent.TextWithMore -> R.layout.onboarding_content_text_action
                is OnboardingContent.Image -> R.layout.onboarding_content_image
            }
            val contentLayoutToUse = View.inflate(this, contentLayoutResToUse, null)

            configureContentView(view = contentLayoutToUse, screenData = screenData)
            contentViewContainer.addView(contentLayoutToUse)
        }
        return view
    }

    private fun handlePrimaryAction(screenType: OnboardingScreenType) {
        when (screenType) {
            OnboardingScreenType.GET_STARTED_2 -> {
                sdkConfig.alwaysFollowDemands = true
            }

            else -> {}
        }
        moveToNextScreen()
    }

    private fun handleSecondaryAction(screenType: OnboardingScreenType) {
        if (screenType in listOf(
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_EXPLAINER,
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_1,
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2,
            )
        ) {
            endOnboarding()
        } else {
            moveToNextScreen()
        }
    }

    private fun endOnboarding() {
        finish()
        sdkConfig.onboardingFinished = true
    }

    private fun configureContentView(view: View, screenData: OnboardingScreenData) {
        when (screenData.content) {
            is OnboardingContent.Image -> {
                val imageView: ImageView = view.findViewById(R.id.onboarding_imageView)
                imageView.setImageDrawable(
                    ContextCompat.getDrawable(
                        this,
                        screenData.content.imageDrawableRes
                    )
                )
            }

            is OnboardingContent.ImageText -> {
                val imageView: ImageView = view.findViewById(R.id.onboarding_imageView)
                val textView: TextView = view.findViewById(R.id.description_textView)
                imageView.setImageDrawable(
                    ContextCompat.getDrawable(
                        this,
                        screenData.content.imageDrawableRes
                    )
                )
                val content = screenData.contentWithAppName(context = this, stringRes = screenData.content.textRes)
                textView.text = content
            }

            is OnboardingContent.TextImage -> {
                val imageView: ImageView = view.findViewById(R.id.onboarding_imageView)
                val textView: TextView = view.findViewById(R.id.description_textView)
                imageView.setImageDrawable(
                    ContextCompat.getDrawable(
                        this,
                        screenData.content.imageDrawableRes
                    )
                )
                val content = screenData.contentWithAppName(context = this, stringRes = screenData.content.textRes)
                textView.text = content
            }

            is OnboardingContent.TextWithMore -> {
                val descriptionTextView: TextView = view.findViewById(R.id.description_textView)
                val actionTextView: TextView = view.findViewById(R.id.action_textView)
                actionTextView.setOnClickListener {
                    if (actionTextView.text == getString(R.string.more_information)) {
                        actionTextView.text = getString(R.string.less)
                        descriptionTextView.maxLines = Int.MAX_VALUE
                    } else {
                        actionTextView.text = getString(R.string.more_information)
                        descriptionTextView.maxLines = 7
                    }
                }
                descriptionTextView.setText(screenData.content.textRes)
            }
        }
    }

    private fun showProminentDialog() {
        if (!PROMINENT_DIALOG_ENABLED) {
            CouponsSdkFacade.goToToAccessibilitySettings(this)
            return
        }
        val alert = Dialog(this)
        val layout = View.inflate(this, R.layout.dialog_layout, null)
        layout.findViewById<Button>(R.id.negative_button).setOnClickListener {
            alert.dismiss()
            endOnboarding()
        }
        layout.findViewById<Button>(R.id.positive_button).setOnClickListener {
            alert.dismiss()
            CouponsSdkFacade.goToToAccessibilitySettings(this)
        }
        alert.setContentView(layout)
        alert.setCancelable(true)
        alert.window?.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        alert.show()
    }

    companion object {
        const val TO_ACCESSIBILITY_STEP_ACTION = "action_to_accessibility"
    }
}
