package app.appnomix.sdk.internal.data

import android.content.Context
import android.content.SharedPreferences
import androidx.ads.identifier.AdvertisingIdClient
import androidx.annotation.DrawableRes
import app.appnomix.sdk.internal.data.local.DynamicConfig
import app.appnomix.sdk.internal.utils.LocalLocationProvider
import app.appnomix.sdk.internal.utils.SLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.UUID
import kotlin.time.Duration.Companion.hours

class SdkConfig(
    private val context: Context,
    private val localLocationProvider: LocalLocationProvider
) {
    private var dynamicConfig: DynamicConfig? = null

    private val prefs: SharedPreferences by lazy {
        context.getSharedPreferences(
            PREFS_NAME, Context.MODE_PRIVATE
        )
    }

    val campaignId: String?
        get() = dynamicConfig?.campaignId

    val dataRefreshPeriodMillis: Long
        get() = dynamicConfig?.dataRefreshPeriodMillis ?: 12.hours.inWholeMilliseconds

    val popupSnoozeTimeMillis: Long
        get() = dynamicConfig?.popupSnoozeTimeMillis ?: 24.hours.inWholeMilliseconds

    val countryCode: String?
        get() = localLocationProvider.getAddress()?.countryCode?.lowercase()

    var alwaysFollowDemands: Boolean
        get() = prefs.getBoolean(ALWAYS_FOLLOW_DEMANDS_KEY, false)
        set(value) {
            prefs.edit().putBoolean(ALWAYS_FOLLOW_DEMANDS_KEY, value).apply()
        }

    val adId: String
        get() = prefs.getString(AD_ID_KEY, null) ?: ""

    val installationId: String
        get() = prefs.getString(INSTALLATION_ID_KEY, null) ?: ""

    var onboardingFinished: Boolean
        get() = prefs.getBoolean(ONBOARDING_FINISHED_KEY, false)
        set(value) {
            prefs.edit().putBoolean(ONBOARDING_FINISHED_KEY, value).apply()
        }

    @DrawableRes
    var appIconRes: Int? = null

    var clientId: String
        get() = prefs.getString(CLIENT_ID, "") ?: ""
        set(value) {
            prefs.edit().putString(CLIENT_ID, value).apply()
        }

    var authToken: String
        get() = prefs.getString(AUTH_TOKEN, "") ?: ""
        set(value) {
            prefs.edit().putString(AUTH_TOKEN, value).apply()
        }

    fun updateWithProperties(dynamicConfig: DynamicConfig) {
        this.dynamicConfig = dynamicConfig
        generateTrackingId()
    }

    private fun generateTrackingId() {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val addId = prefs.getString(AD_ID_KEY, null)
                val installationId = prefs.getString(INSTALLATION_ID_KEY, null)
                if (addId.isNullOrEmpty()) {
                    if (AdvertisingIdClient.isAdvertisingIdProviderAvailable(context)) {
                        val advertisingIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(context)
                        prefs.edit()
                            .putString(AD_ID_KEY, advertisingIdInfo.get().id)
                            .apply()
                    } else if (installationId.isNullOrEmpty()) {
                        prefs.edit()
                            .putString(INSTALLATION_ID_KEY, UUID.randomUUID().toString())
                            .apply()
                    }
                }
            } catch (t: Throwable) {
                SLog.e("something went wrong while generating tracking id")
            }
        }
    }

    companion object {
        const val PREFS_NAME = "sl_prefs"
        private const val ALWAYS_FOLLOW_DEMANDS_KEY = "always_follow_demands"
        private const val INSTALLATION_ID_KEY = "installation_id"
        private const val AD_ID_KEY = "ad_id"
        private const val ONBOARDING_FINISHED_KEY = "onboarding_finished"
        private const val CLIENT_ID = "client_id"
        private const val AUTH_TOKEN = "auth_token"
    }
}
