package app.appnomix.sdk.internal.data.network

import android.content.Context
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.NetworkType
import androidx.work.PeriodicWorkRequest
import androidx.work.PeriodicWorkRequestBuilder
import androidx.work.WorkerParameters
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.data.local.toDynamicConfig
import app.appnomix.sdk.internal.domain.CouponsRepo
import app.appnomix.sdk.internal.utils.SLog
import java.util.concurrent.TimeUnit

class CouponsSyncWorker(
    appContext: Context,
    workerParams: WorkerParameters
) : CoroutineWorker(appContext, workerParams) {
    private val repo: CouponsRepo by lazy { Deps.repo() }
    private val api: SaversLeagueApi by lazy { Deps.api() }
    private val sdkConfig: SdkConfig by lazy { Deps.sdkConfig() }

    override suspend fun doWork(): Result {
        try {
            SLog.i("preparing to fetch sdk data")
            fetchConfig()
            fetchCoupons(forHomepage = true)
            fetchCoupons(forHomepage = false)
            fetchDemands()
        } catch (e: Throwable) {
            SLog.e("Something went wrong with $NAME", e)
            return Result.failure()
        }
        return Result.success()
    }

    private suspend fun fetchCoupons(forHomepage: Boolean = false) {
        repo.fetchCoupons(countryCode = sdkConfig.countryCode, forHomepage = forHomepage)
    }

    private suspend fun fetchDemands() {
        repo.fetchDemands(sdkConfig.countryCode)
    }

    private suspend fun fetchConfig() {
        val configContent = api.getConfig()
        sdkConfig.updateWithProperties(configContent.toDynamicConfig())
    }

    companion object {
        const val NAME = "coupons-sync-worker"
        fun create(repeatIntervalHours: Long): PeriodicWorkRequest {
            val constraints =
                Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()
            return PeriodicWorkRequestBuilder<CouponsSyncWorker>(
                repeatIntervalHours,
                TimeUnit.HOURS
            )
                .setConstraints(constraints)
                .build()
        }
    }
}
