package app.appnomix.sdk.internal.data.network

import DomainDataDto
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.data.SdkConfig
import app.appnomix.sdk.internal.data.network.model.AnalyticsEventsDto
import app.appnomix.sdk.internal.data.network.model.ConfigContentDto
import app.appnomix.sdk.internal.data.network.model.CouponDto
import app.appnomix.sdk.internal.data.network.model.DemandDto
import app.appnomix.sdk.internal.data.network.model.PageInfoDto
import app.appnomix.sdk.internal.data.network.model.TimestampSerializer
import app.appnomix.sdk.internal.utils.SLog
import io.ktor.client.HttpClient
import io.ktor.client.HttpClientConfig
import io.ktor.client.call.body
import io.ktor.client.engine.android.Android
import io.ktor.client.engine.android.AndroidEngineConfig
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.defaultRequest
import io.ktor.client.plugins.logging.LogLevel
import io.ktor.client.plugins.logging.Logger
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.request.bearerAuth
import io.ktor.client.request.get
import io.ktor.client.request.headers
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.client.statement.HttpResponse
import io.ktor.http.ContentType
import io.ktor.http.URLProtocol
import io.ktor.http.contentType
import io.ktor.http.path
import io.ktor.serialization.kotlinx.json.json
import io.ktor.util.reflect.TypeInfo
import io.ktor.util.reflect.typeInfo
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import kotlinx.serialization.modules.SerializersModule
import kotlinx.serialization.modules.polymorphic
import java.time.LocalDateTime
import kotlin.time.Duration.Companion.seconds

private val LOG_FULL_REQUEST = true

val APP_HOSTS = listOf(
    "appnomix.app",
    "app.appnomix.app",
    "i.v2i8b.com",
    "r.v2i8b.com",
    "linksynergy.com",
    "click.linksynergy.com"
)

val jsonEngine = Json {
    ignoreUnknownKeys = true
    prettyPrint = true
    isLenient = true
    encodeDefaults = true
    serializersModule = SerializersModule {
        contextual(LocalDateTime::class, TimestampSerializer())
        polymorphic(DomainDataDto::class) {
            defaultDeserializer { DomainDataDto.Unknown.serializer() }
        }
    }
}

class SaversLeagueApi(
    private val defaultHost: String = "api.appnomix.app",
) {
    private val sdkConfig: SdkConfig by lazy { Deps.sdkConfig() }

    private val baseClient = HttpClient(Android) {
        engine()
        logging()
        defaultRequest {
            url {
                protocol = URLProtocol.HTTPS
                host = defaultHost
            }
            bearerAuth(sdkConfig.authToken)
            contentType(ContentType.Application.Json)
        }
        contentNegotiation()
    }

    private val clientV1 = HttpClient(Android) {
        engine()
        logging()
        defaultRequest {
            url {
                protocol = URLProtocol.HTTPS
                host = defaultHost
                path("api/v1/")
            }
            bearerAuth(sdkConfig.authToken)
            contentType(ContentType.Application.Json)
        }
        contentNegotiation()
    }

    private val clientV2 = HttpClient(Android) {
        engine()
        logging()
        defaultRequest {
            url {
                protocol = URLProtocol.HTTPS
                host = defaultHost
                path("api/v2/")
            }
            bearerAuth(sdkConfig.authToken)
            contentType(ContentType.Application.Json)
        }
        contentNegotiation()
    }

    private fun HttpClientConfig<AndroidEngineConfig>.engine() {
        engine {
            connectTimeout = 30.seconds.inWholeMilliseconds.toInt()
            socketTimeout = 30.seconds.inWholeMilliseconds.toInt()
        }
    }

    private fun HttpClientConfig<AndroidEngineConfig>.logging() {
        install(Logging) {
            logger = object : Logger {
                override fun log(message: String) {
                    SLog.d(message)
                }
            }
            level = if (LOG_FULL_REQUEST) LogLevel.ALL else LogLevel.HEADERS
        }
    }

    private fun HttpClientConfig<AndroidEngineConfig>.contentNegotiation() {
        install(ContentNegotiation) {
            json(jsonEngine)
        }
    }


    suspend fun getConfig(): ConfigContentDto {
        val response: ConfigContentDto =
            clientV1.get("client/configuration") {
                headers {
                    SLog.d("Fetching config for clientId: ${sdkConfig.clientId} / ${sdkConfig.authToken}")
                    headers.append("client-id", sdkConfig.clientId)
                    headers.append("auth-token", sdkConfig.authToken)
                }
            }
                .body()
        return response
    }

    suspend fun getCoupons(
        forHomepage: Boolean
    ): DataResponse<CouponDto, PageInfoDto> {
        val response: DataResponse<CouponDto, PageInfoDto> =
            clientV2.get("promo/${sdkConfig.campaignId}") {
                url {
                    sdkConfig.countryCodeOverride?.let {
                        parameters.append("country_codes", it)
                    }
                    parameters.append("homepage", forHomepage.toString())
                }
            }.body()
        return response
    }

    suspend fun sendAnalyticsEvents(analyticsEvents: AnalyticsEventsDto): HttpResponse =
        baseClient.post(sdkConfig.analyticsUrlPath ?: "api/v1/analytics/event") {
            setBody(analyticsEvents)
        }

    suspend fun getOnDemandRedirects(): DataResponse<DemandDto, PageInfoDto> {
        val response: DataResponse<DemandDto, PageInfoDto> =
            clientV2.get("demand/${sdkConfig.campaignId}") {
                url {
                    sdkConfig.countryCodeOverride?.let {
                        parameters.append("country_codes", it)
                    }
                }
            }.body()
        return response
    }

    suspend inline fun <reified T : Any> getNextPageData(url: String): DataResponse<T, PageInfoDto> {
        val type = typeInfo<DataResponse<T, PageInfoDto>>()
        return getNextPageData(url, type)
    }

    suspend fun <T> getNextPageData(url: String, type: TypeInfo): DataResponse<T, PageInfoDto> {
        return try {
            clientV2.get(url).body(type)
        } catch (t: Throwable) {
            throw t
        }
    }

    fun getCouponImageUrl(brandName: String): String {
        return "https://i.v2i8b.com/$brandName"
    }

    suspend fun getBrand(baseDomain: String): DomainDataDto {
        return clientV1.get("coupon/config/$baseDomain") {
            url {
                sdkConfig.campaignId?.let { campaignId ->
                    parameters.append("campaign_id", campaignId)
                }
                sdkConfig.countryCodeOverride?.let {
                    parameters.append("country_code", it)
                }
            }
        }.body()
    }
}


@Serializable
data class DataResponse<T, R>(
    @SerialName("data") val data: List<T>,
    @SerialName("meta") val meta: R? = null
)
