package app.appnomix.sdk.internal.ui

import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.graphics.PixelFormat
import android.graphics.Typeface
import android.graphics.drawable.Drawable
import android.text.Spannable
import android.text.SpannableString
import android.text.style.StyleSpan
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.view.WindowManager
import android.widget.Button
import android.widget.ImageView
import android.widget.TextView
import app.appnomix.sdk.R
import app.appnomix.sdk.internal.domain.model.Coupon
import java.util.Locale
import java.util.concurrent.ConcurrentHashMap

internal data class PopupView(
    val view: View,
    val params: WindowManager.LayoutParams,
    val updateAction: ((data: PopupViewData) -> Unit)? = null
)

internal class PopupViewFactory(private val context: Context) {
    private val cache = ConcurrentHashMap<PopupViewType, PopupView>()

    fun create(
        type: PopupViewType,
        data: PopupViewData? = null,
        windowManager: WindowManager,
        onCloseAction: () -> Unit = {},
        onConfirmAction: (Any?) -> Unit = {},
        onDismissAction: () -> Unit = {},
    ): PopupView {
        return when (type) {
            PopupViewType.AUTO_APPLY_ASK -> {
                PopupView(
                    view = createAutoApplyAskView(
                        confirmAction = onConfirmAction,
                        dismissAction = onDismissAction
                    ),
                    params = getDefaultLayoutParams()
                )
            }

            PopupViewType.TRYING_COUPONS -> {
                cache.getOrPut(type) {
                    val tryingView = TryingCouponsView(context, getAppLogo())
                    PopupView(
                        view = tryingView.create(
                            data as PopupViewData,
                            onDismissAction
                        ),
                        params = getDefaultLayoutParams(),
                        updateAction = { tryingView.configure(it) }
                    )
                }
            }

            PopupViewType.COUPONS_FOUND -> {
                val view = CouponsFoundView(context, getAppLogo())
                PopupView(
                    view = view.create(
                        data = data as PopupViewData,
                        confirmAction = onConfirmAction,
                        dismissAction = onDismissAction
                    ),
                    params = getDefaultLayoutParams(),
                    updateAction = { }
                )
            }

            PopupViewType.STOP_AND_APPLY_CONFIRMATION -> {
                PopupView(
                    view = createStopAndApplyConfirmationView(
                        data = data as PopupViewData,
                        dismissAction = onDismissAction,
                        confirmAction = onConfirmAction,
                    ),
                    params = getDefaultLayoutParams()
                )
            }

            PopupViewType.APPLY_BEST_COUPON -> {
                val view = ApplyBestCouponView(context, getAppLogo())
                PopupView(
                    view = view.create(
                        data = data as PopupViewData,
                        confirmAction = onConfirmAction,
                    ),
                    params = getDefaultLayoutParams(),
                    updateAction = { }
                )
            }

            PopupViewType.NO_SAVINGS_FOUND -> {
                PopupView(
                    view = createNoCouponsFoundView(confirmAction = onConfirmAction),
                    params = getDefaultLayoutParams()
                )
            }

            PopupViewType.FAB_MESSAGE -> {
                val view = FabCopyCodeView(context)
                val fabParams = view.getFabLayoutParams(
                    windowManager = windowManager,
                    data = data as PopupViewData
                )
                PopupView(
                    view = view.createFabMessageView(
                        data = data,
                        params = fabParams,
                        windowManager = windowManager,
                        onAction = onConfirmAction,
                        onAutoDismiss = onDismissAction
                    ),
                    params = fabParams
                )
            }
        }
    }

    private fun getDefaultLayoutParams(): WindowManager.LayoutParams {
        return WindowManager.LayoutParams(
            WindowManager.LayoutParams.MATCH_PARENT,
            WindowManager.LayoutParams.WRAP_CONTENT,
            WindowManager.LayoutParams.TYPE_ACCESSIBILITY_OVERLAY,
            WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE,
            PixelFormat.TRANSLUCENT
        ).apply {
            gravity = Gravity.BOTTOM or Gravity.CENTER_HORIZONTAL
        }
    }

    private fun createStopAndApplyConfirmationView(
        data: PopupViewData?,
        dismissAction: () -> Unit,
        confirmAction: (Any?) -> Unit
    ): View {
        val overlay =
            LayoutInflater.from(context).inflate(R.layout.stop_and_apply_popup_window, null)
        val currencySymbol =
            data?.get<String>(PopupViewData.PopupViewDataType.CURRENCY_SYMBOL) ?: ""
        return overlay.apply {
            val titlePartWithAppName: TextView? = findViewById(R.id.title_part3_textView)
            titlePartWithAppName?.text = resources.getString(
                R.string.with_app,
                context.getString(context.applicationInfo.labelRes)
            )

            val logoImageView: ImageView? = findViewById(R.id.logo_imageView)
            logoImageView?.setImageDrawable(getAppLogo())

            val actionButton: Button? = findViewById(R.id.action_button)
            actionButton?.setOnClickListener { confirmAction(null) }

            val laterButton: Button? = findViewById(R.id.later_button)
            laterButton?.setOnClickListener { dismissAction() }

            val savingsTextView: TextView? = findViewById(R.id.savings_textView)
            data?.get<Double>(PopupViewData.PopupViewDataType.TOTAL_SAVINGS)?.let {
                val savingsValue =
                    "$currencySymbol${String.format(Locale.getDefault(), it.toString())}"
                savingsTextView?.text = savingsValue
            }

            data?.get<Coupon>(PopupViewData.PopupViewDataType.COUPON)?.let {
                val couponDescriptionTextView: TextView? =
                    findViewById(R.id.couponDescription_textView)
                couponDescriptionTextView?.text = it.description
                val couponCodeTextView: TextView? = findViewById(R.id.couponCode_textView)
                couponCodeTextView?.text = it.code
                it.code.copyToClipboard(context)
            }
        }
    }

    private fun createAutoApplyAskView(
        confirmAction: (Any?) -> Unit,
        dismissAction: () -> Unit
    ): View {
        val overlay =
            LayoutInflater.from(context).inflate(R.layout.auto_apply_ask_popup_window, null)
        return overlay.apply {
            val logoImageView: ImageView? = findViewById(R.id.logo_imageView)
            logoImageView?.setImageDrawable(getAppLogo())

            val actionButton: Button? = findViewById(R.id.action_button)
            actionButton?.setOnClickListener { confirmAction(null) }

            val laterButton: Button? = findViewById(R.id.later_button)
            laterButton?.setOnClickListener { dismissAction() }
        }
    }

    private fun createNoCouponsFoundView(
        confirmAction: (Any?) -> Unit
    ): View {
        val overlay =
            LayoutInflater.from(context).inflate(R.layout.no_savings_found_window, null)
        return overlay.apply {
            val logoImageView: ImageView? = findViewById(R.id.logo_imageView)
            logoImageView?.setImageDrawable(getAppLogo())

            val actionButton: Button? = findViewById(R.id.action_button)
            actionButton?.setOnClickListener { confirmAction(null) }
        }
    }


    private fun getAppLogo(): Drawable {
        return context.packageManager.getApplicationIcon(context.packageName)
    }
}


enum class PopupViewType {
    COUPONS_FOUND,
    AUTO_APPLY_ASK,
    TRYING_COUPONS,
    STOP_AND_APPLY_CONFIRMATION,
    APPLY_BEST_COUPON,
    NO_SAVINGS_FOUND,
    FAB_MESSAGE,
}

class PopupViewData(private val cache: ConcurrentHashMap<PopupViewDataType, Any> = ConcurrentHashMap<PopupViewDataType, Any>()) {

    fun <T : Any> set(type: PopupViewDataType, data: T) {
        cache[type] = data
    }

    @Suppress("UNCHECKED_CAST")
    fun <T> get(type: PopupViewDataType): T? {
        return cache[type] as? T
    }

    enum class PopupViewDataType {
        COUPON_LIST,
        CURRENT_COUPON_INDEX_BEING_VALIDATED,
        COUPON,
        TOTAL_SAVINGS,
        CURRENCY_SYMBOL,
        PRIMARY_MESSAGE,
        SECONDARY_MESSAGE,
        FAB_POSITION_DATA,
    }

    fun copy(): PopupViewData {
        return PopupViewData(cache = ConcurrentHashMap(cache))
    }
}

data class FabPositionData(
    var y: Int = -1
)

fun String.copyToClipboard(context: Context) {
    val clipboardManager: ClipboardManager =
        context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
    val clip = ClipData.newPlainText("coupon", this)
    clipboardManager.setPrimaryClip(clip)
}

fun String.boldNumbers(): SpannableString {
    val spannableString = SpannableString(this)
    val regex = "\\d+".toRegex()
    regex.findAll(this).forEach { matchResult ->
        val start = matchResult.range.first
        val end = matchResult.range.last + 1
        spannableString.setSpan(
            /* what = */ StyleSpan(Typeface.BOLD),
            /* start = */ start,
            /* end = */ end,
            /* flags = */ Spannable.SPAN_EXCLUSIVE_EXCLUSIVE
        )
    }

    return spannableString
}
