package app.appnomix.sdk.internal

import android.accessibilityservice.AccessibilityService
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.view.accessibility.AccessibilityEvent
import android.view.accessibility.AccessibilityWindowInfo
import android.webkit.WebView
import app.appnomix.sdk.internal.domain.CouponCodeApplier
import app.appnomix.sdk.internal.domain.browsers.BrowserResolver
import app.appnomix.sdk.internal.domain.browsers.DeviceBrowser.Companion.isHttpUrl
import app.appnomix.sdk.internal.domain.browsers.ensureHttpsPrefix
import app.appnomix.sdk.internal.domain.machine.AppnomixStateMachineFactory
import app.appnomix.sdk.internal.domain.machine.states.TreeInventoryType
import app.appnomix.sdk.internal.ui.PopupDisplay
import app.appnomix.sdk.internal.ui.WebviewWindow
import app.appnomix.sdk.internal.utils.SLog
import java.time.Instant

private const val MILLIS_BETWEEN_EVENTS = 300

internal class CouponsAccessibilityService : AccessibilityService(), CouponCodeApplier {
    private var screenOffReceiver: BroadcastReceiver? = null
    private val popupDisplay = PopupDisplay(this)
    private var lastProcessedEventTs = 0L

    private val stateMachine = AppnomixStateMachineFactory.get(popupDisplay)
    private val webviewWindow: WebviewWindow by lazy { WebviewWindow(this) }
    private var lastUrl: String? = null

    init {
        this.screenOffReceiver = object : BroadcastReceiver() {
            override fun onReceive(context: Context, intent: Intent?) {
                if (intent?.action == Intent.ACTION_SCREEN_OFF) {
                    popupDisplay.hide()
                }
            }
        }
    }

    override fun onCreate() {
        super.onCreate()
        stateMachine.save(TreeInventoryType.POPUP_DISPLAY, popupDisplay)
        stateMachine.save(TreeInventoryType.CONTEXT, applicationContext)
        stateMachine.save(TreeInventoryType.EXTERNAL_COUPON_CODE_APPLIER, this)
    }

    override fun apply(code: String) {
        val rootNode = rootInActiveWindow ?: return
        val packageName = rootNode.packageName ?: ""
        val browser = BrowserResolver.getBrowser(packageName.toString())
        browser?.applyCouponCodeOnce(rootNode, browser.getAddressText(rootNode) ?: "", code)
    }

    override fun onAccessibilityEvent(event: AccessibilityEvent) {
//        SLog.i("event: ${AccessibilityEvent.eventTypeToString(event.eventType)}")
        when (event.eventType) {
            AccessibilityEvent.TYPE_WINDOW_STATE_CHANGED, AccessibilityEvent.TYPE_WINDOW_CONTENT_CHANGED -> {
                if (Instant.now().toEpochMilli() - lastProcessedEventTs > MILLIS_BETWEEN_EVENTS) {
                    lastProcessedEventTs = Instant.now().toEpochMilli()
                    processEvent()
                }
            }

            AccessibilityEvent.TYPE_VIEW_SCROLLED -> {
                webviewWindow.updateScroll(event.scrollX, event.scrollY)
            }

            AccessibilityEvent.TYPE_VIEW_CLICKED -> {
                processClickEvent(event)
            }

            else -> {
//                SLog.i("event: ${event.eventType} with id: ${event.source?.contentDescription}")
            }
        }
    }

    private fun processClickEvent(event: AccessibilityEvent) {
        try {
            val text = event.source?.text
            val contentDescription = event.source?.contentDescription
            val viewIdResourceName = event.source?.viewIdResourceName
            val className = event.source?.className
            SLog.d("clicked text: $text")
            SLog.d("clicked contentDescription: $contentDescription")
            SLog.d("clicked viewIdResourceName: $viewIdResourceName")
            SLog.d("clicked className: $className")

            webviewWindow.click(viewIdResourceName, contentDescription, text, className)
        } catch (t: Throwable) {
            SLog.e("something went wrong when processing click", t)
        }
    }

    private fun processEvent() {
        try {
            val rootNode = rootInActiveWindow ?: return
            val packageName = rootNode.packageName ?: ""
            val browser = BrowserResolver.getBrowser(packageName.toString())

            if (browser == null && packageName != "com.saversleague.coupons") {
                // stop any activity if browser or our app is not in the foreground
                stateMachine.pause()

            } else if (browser != null) {
                stateMachine.save(TreeInventoryType.URL_FOLLOWER, browser)

                val isValidBrowser =
                    BrowserResolver.supportedBrowserPackages().contains(packageName)
                val addressText = browser.getAddressText(rootNode)?.ensureHttpsPrefix() ?: ""
                webviewWindow.update(addressText, browser.locateWebviewBounds(rootNode))

                val webView: WebView = webviewWindow.getWebView() ?: return
                if (isValidBrowser && browser.isIdle(rootNode) && !isKeyboardOpened()) {
                    if (lastUrl != addressText && addressText.isHttpUrl()) {
                        lastUrl = addressText
                        stateMachine.updateUrl(addressText, webView)
                    } else if (stateMachine.isPaused && addressText.isHttpUrl()) {
                        stateMachine.resume()
                    }
                } else {
                    stateMachine.pause()
                }
            }
        } catch (e: Exception) {
            SLog.e("something went wrong", e)
        }
    }

    override fun onInterrupt() {
    }

    private fun isKeyboardOpened(): Boolean {
        val windowInfoList = windows
        for (windowInfo in windowInfoList) {
            if (windowInfo.type == AccessibilityWindowInfo.TYPE_INPUT_METHOD) {
                return true
            }
        }
        return false
    }

    override fun onServiceConnected() {
        super.onServiceConnected()
        val filter = IntentFilter(Intent.ACTION_SCREEN_OFF)
        this.registerReceiver(this.screenOffReceiver, filter)
    }

    override fun onDestroy() {
        super.onDestroy()
        try {
            this.unregisterReceiver(screenOffReceiver)
        } catch (ignored: Exception) {
        }
    }
}
