package app.appnomix.sdk.internal.data.local.dao

import androidx.room.Dao
import androidx.room.Insert
import androidx.room.OnConflictStrategy
import androidx.room.Query
import androidx.room.Transaction
import app.appnomix.sdk.internal.data.local.model.DemandEntity
import app.appnomix.sdk.internal.data.local.model.DemandInteractionEntity
import app.appnomix.sdk.internal.data.local.model.DemandWithInteractionEntity
import java.time.LocalDateTime
import kotlin.time.Duration.Companion.days
import kotlin.time.Duration.Companion.hours

private const val FALLBACK_TRIGGER_SNOOZE_COUNT = 3
private val DEFAULT_SNOOZE_INTERVAL = 14.days
private val FALLBACK_SNOOZE_INTERVAL = 90.days
private val MIN_USE_INTERVAL = 24.hours

@Dao
interface DemandDao {

    @Query("SELECT * FROM demand")
    suspend fun getAll(): List<DemandWithInteractionEntity>

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun saveDemands(items: List<DemandEntity>)

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun saveDemandsSnoozeInformation(items: List<DemandInteractionEntity>)

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    suspend fun insertOrUpdate(demandInteraction: DemandInteractionEntity)

    @Query(
        """
        SELECT * 
        FROM demand 
        WHERE root_domain = :domain COLLATE NOCASE
          AND NOT EXISTS (
              SELECT 1 
              FROM demand_interaction 
              WHERE demand_interaction.brand_domain = demand.root_domain 
                AND (
                    last_use_time IS NULL 
                    OR (strftime('%s', 'now') - strftime('%s', last_use_time)) < :minUseInterval
                )
                AND (
                    snooze_time IS NULL 
                    OR CASE 
                            WHEN snooze_count >= $FALLBACK_TRIGGER_SNOOZE_COUNT 
                            THEN (strftime('%s', 'now') - (strftime('%s', snooze_time) + :fallbackSnoozeSecondsInterval)) < 0
                            ELSE (strftime('%s', 'now') - (strftime('%s', snooze_time) + :defaultSnoozeSecondsInterval)) < 0
                        END
                )
          )
    """
    )
    suspend fun getActiveForDomainAndCountry(
        domain: String,
        fallbackSnoozeSecondsInterval: Long = FALLBACK_SNOOZE_INTERVAL.inWholeSeconds,
        defaultSnoozeSecondsInterval: Long = DEFAULT_SNOOZE_INTERVAL.inWholeSeconds,
        minUseInterval: Long = MIN_USE_INTERVAL.inWholeSeconds,
    ): List<DemandWithInteractionEntity>

    @Query(
        """
        SELECT EXISTS(
            SELECT 1 FROM demand 
            WHERE root_domain = :domain COLLATE NOCASE
              AND NOT EXISTS (
                  SELECT 1 
                  FROM demand_interaction 
                  WHERE demand_interaction.brand_domain = demand.root_domain 
                    AND (
                            (last_use_time NOT NULL 
                            AND (strftime('%s', 'now') - strftime('%s', last_use_time)) < :minUseInterval)
                    
                        OR (
                            snooze_time IS NOT NULL 
                            AND CASE 
                                    WHEN snooze_count >= $FALLBACK_TRIGGER_SNOOZE_COUNT 
                                    THEN (strftime('%s', 'now') - (strftime('%s', snooze_time) + :fallbackSnoozeSecondsInterval)) < 0
                                    ELSE (strftime('%s', 'now') - (strftime('%s', snooze_time) + :defaultSnoozeSecondsInterval)) < 0
                                END
                        )
                    )
              )
          )
        """
    )
    suspend fun hasActiveDemand(
        domain: String,
        fallbackSnoozeSecondsInterval: Long = FALLBACK_SNOOZE_INTERVAL.inWholeSeconds,
        defaultSnoozeSecondsInterval: Long = DEFAULT_SNOOZE_INTERVAL.inWholeSeconds,
        minUseInterval: Long = MIN_USE_INTERVAL.inWholeSeconds,
    ): Boolean

    @Query("DELETE FROM demand")
    suspend fun deleteAll()

    @Query("UPDATE demand_interaction SET last_use_time=:now WHERE brand_domain=:domain")
    suspend fun updateLastUseTime(domain: String, now: LocalDateTime = LocalDateTime.now()): Int

    @Transaction
    suspend fun upsertLastUseTime(
        domain: String,
        now: LocalDateTime = LocalDateTime.now()
    ) {
        val updatedRows = updateLastUseTime(domain, now)
        if (updatedRows == 0) {
            insertOrUpdate(
                DemandInteractionEntity(
                    brandDomain = domain,
                    snoozeTime = now.minusYears(100),
                    snoozeCount = 0,
                    lastUseTime = now,
                )
            )
        }
    }

    @Query("SELECT COUNT(*) FROM demand")
    fun count(): Int
}
