package app.appnomix.sdk.internal.data.network

import android.content.Context
import androidx.work.BackoffPolicy
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.NetworkType
import androidx.work.OneTimeWorkRequest
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkerParameters
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.domain.CouponsRepo
import app.appnomix.sdk.internal.utils.SLog
import java.util.concurrent.TimeUnit

class AppnomixDemandsSyncWorker(
    appContext: Context,
    workerParams: WorkerParameters
) : CoroutineWorker(appContext, workerParams) {
    private val repo: CouponsRepo by lazy { Deps.repo() }

    override suspend fun doWork(): Result {
        if (runAttemptCount >= MAX_RETRY_COUNT) {
            return Result.failure()
        }
        try {
            SLog.i("preparing to fetch demands data")
            fetchDemands()
            SLog.i("demands fetched")
        } catch (e: Throwable) {
            SLog.e("Something went wrong with $NAME", e)
            return Result.retry()
        }
        return Result.success()
    }

    private suspend fun fetchDemands() {
        repo.fetchDemands()
    }

    companion object {
        const val MAX_RETRY_COUNT = 5
        const val NAME = "demands-sync-worker"
        fun create(): OneTimeWorkRequest {
            val constraints =
                Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build()
            return OneTimeWorkRequestBuilder<AppnomixDemandsSyncWorker>()
                .setConstraints(constraints)
                .setBackoffCriteria(BackoffPolicy.EXPONENTIAL, 30, TimeUnit.SECONDS)
                .build()
        }
    }
}
