package app.appnomix.sdk.internal.domain.machine

import android.webkit.WebView
import app.appnomix.sdk.internal.data.network.APP_HOSTS
import app.appnomix.sdk.internal.domain.machine.states.DemandVerifierNode
import app.appnomix.sdk.internal.domain.machine.states.TerminalNode
import app.appnomix.sdk.internal.domain.machine.states.TreeInventory
import app.appnomix.sdk.internal.domain.machine.states.TreeInventoryType
import app.appnomix.sdk.internal.domain.machine.states.TreeNode
import app.appnomix.sdk.internal.domain.machine.states.TreeNodeTransitionListener
import app.appnomix.sdk.internal.domain.machine.states.TreeTransition
import app.appnomix.sdk.internal.domain.machine.states.UiTreeNode
import app.appnomix.sdk.internal.ui.PopupDisplay
import app.appnomix.sdk.internal.utils.SLog
import extractBaseDomain
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import java.time.Instant
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicReference

class AppnomixStateMachine internal constructor(popupDisplay: PopupDisplay) : TreeInventory,
    TreeNodeTransitionListener {
    private val cache = ConcurrentHashMap<TreeInventoryType, Any>()
    private val scope = CoroutineScope(Dispatchers.Default + SupervisorJob())
    private var currentNode: TreeNode? = null
    private var snapshots: MutableMap<TreeNode, SnapshotNodeData> = mutableMapOf()

    private val started = AtomicBoolean(false)
    private val paused = AtomicBoolean(false)

    val isPaused
        get() = paused.get()

    init {
        cache[TreeInventoryType.POPUP_DISPLAY] = popupDisplay
    }

    fun updateUrl(
        url: String,
        trialWebView: WebView
    ) {
        val isRunning = started.get()
        val hasUrlChanged = url != cache[TreeInventoryType.CURRENT_URL]
        val isUrlLocal = extractBaseDomain(url) in APP_HOSTS

        if (isRunning && hasUrlChanged && !isUrlLocal) {
            SLog.i("[aa] url has changed: $url")
            stop()
        } else if (!isRunning && hasUrlChanged && !isUrlLocal) {
            SLog.i("[aa] starting machine for $url")
            started.set(true)
            cache[TreeInventoryType.CURRENT_URL] = url
            cache[TreeInventoryType.WEB_VIEW] = trialWebView

            currentNode = DemandVerifierNode
            currentNode?.transitionListener = this
            currentNode?.inventory = this
            scope.launch {
                currentNode?.execute()
            }
        }
    }

    fun resume(fromNode: TreeNode? = null,
               fromLastSavedData: Boolean = true) {
        val lastSaveNode = snapshots.maxByOrNull { it.value.timestamp }?.key
        currentNode = fromNode ?: lastSaveNode
        SLog.i("[aa] resuming state machine from $currentNode")

        if (currentNode != null) {
            val snapshotData = if (fromLastSavedData && lastSaveNode != null)
                snapshots[lastSaveNode]
            else
                snapshots[currentNode]
            snapshotData?.cache?.let { cache.putAll(it) }
        }

        started.set(true)
        paused.set(false)
        scope.launch {
            currentNode?.execute()
        }
    }

    fun pause() {
        SLog.d("[aa] requesting to pause: $currentNode")
        currentNode?.pause()
        paused.set(true)
    }

    fun stop() {
        SLog.i("[aa] ending state machine")
        pause()
        currentNode = TerminalNode
        scope.launch {
            TerminalNode.execute()
        }
        started.set(false)
        paused.set(false)
        snapshots.clear()
    }

    override fun onResumeRequested(node: TreeNode) {
        resume(node)
    }

    override fun contains(type: TreeInventoryType): Boolean {
        return cache.containsKey(type)
    }

    @Suppress("UNCHECKED_CAST")
    override fun <T> get(type: TreeInventoryType): T? {
        return cache[type] as? T
    }

    override fun <T> save(type: TreeInventoryType, data: T) {
        if (data == null) return
        cache[type] = data
    }

    override fun remove(type: TreeInventoryType) {
        cache.remove(type)
    }

    override fun onReadyForTransition(node: TreeNode, transition: TreeTransition) {
        if (!started.get()) {
            // safety-net, in case there is some hanging/async state that hasn't stopped
            return
        }

        if (currentNode is UiTreeNode && currentNode != node) {
            SLog.i("[aa] $node requested transition to $transition, but ignoring since currentNode is $currentNode")
            return
        }

        val nextNode: TreeNode? = when (transition) {
            TreeTransition.Negative -> node.negativeChildNode()
            TreeTransition.Positive -> node.positiveChildNode()
            TreeTransition.None -> null
        }
        SLog.i("[aa] trying to transition from $node to $nextNode")
        nextNode?.inventory = this
        nextNode?.transitionListener = this

        if (currentNode !is TerminalNode) {
            currentNode?.let {
                snapshots[it] = SnapshotNodeData(
                    cache = cache.toMap(),
                    timestamp = Instant.now().toEpochMilli()
                )
            }
        }

        scope.launch {
            nextNode?.execute()
        }
        currentNode = nextNode
        if (currentNode is TerminalNode) {
            SLog.i("[aa] ending state machine")
            started.set(false)
            paused.set(false)
        }
    }
}

object AppnomixStateMachineFactory {

    private var instance = AtomicReference<AppnomixStateMachine?>()

    fun get(popupDisplay: PopupDisplay): AppnomixStateMachine {
        instance.compareAndSet(null, AppnomixStateMachine(popupDisplay))
        return instance.get() as AppnomixStateMachine
    }

    fun getCached() = instance.get()
}

private class SnapshotNodeData(
    val timestamp: Long,
    val cache: Map<TreeInventoryType, Any>
)