package app.appnomix.sdk.internal.ui

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.animation.AnimatorSet
import android.animation.ObjectAnimator
import android.content.Context
import android.graphics.PixelFormat
import android.os.Build
import android.util.DisplayMetrics
import android.view.Gravity
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import app.appnomix.sdk.R
import kotlin.math.abs
import kotlin.math.min
import kotlin.time.Duration.Companion.seconds

class FabCopyCodeView(
    val context: Context
) {

    private var screenWidth = 0
    private var initialX: Int = 0
    private var initialY: Int = 0
    private var initialTouchX: Float = 0.toFloat()
    private var initialTouchY: Float = 0.toFloat()
    private var isMessageFinishedShowing = false

    fun getFabLayoutParams(
        windowManager: WindowManager,
        data: PopupViewData
    ): WindowManager.LayoutParams {
        val fabPositionData = data.get<FabPositionData>(PopupViewData.PopupViewDataType.FAB_POSITION_DATA)
        val params = WindowManager.LayoutParams(
            WindowManager.LayoutParams.MATCH_PARENT,
            WindowManager.LayoutParams.WRAP_CONTENT,
            WindowManager.LayoutParams.TYPE_ACCESSIBILITY_OVERLAY,
            WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE,
            PixelFormat.TRANSLUCENT
        ).apply {
            gravity = Gravity.BOTTOM or Gravity.START

            fabPositionData?.let {
                y = it.y
            } ?: run {
                y = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) windowManager.currentWindowMetrics.windowInsets.bottomInset else 0
            }
        }

        return params
    }

    fun createFabMessageView(
        data: PopupViewData,
        params: WindowManager.LayoutParams,
        windowManager: WindowManager,
        onAction: (Any?) -> Unit = {},
        onAutoDismiss: () -> Unit = {}
    ): View {
        screenWidth = getScreenWidth(windowManager)

        val primaryText = data.get<String>(PopupViewData.PopupViewDataType.PRIMARY_MESSAGE)
        val secondaryText = data.get<String>(PopupViewData.PopupViewDataType.SECONDARY_MESSAGE)
        val positionData = data.get<FabPositionData>(PopupViewData.PopupViewDataType.FAB_POSITION_DATA)

        val overlay = LayoutInflater.from(context).inflate(R.layout.fab_savings_popup, null)

        overlay.apply {
            val fabGroup = findViewById<View>(R.id.fab_group)
            val textContainer: ViewGroup? = findViewById(R.id.cardView)

            val textLayoutParams = textContainer?.layoutParams as ConstraintLayout.LayoutParams

            textLayoutParams.width = min(
                textContainer.resources.getDimensionPixelSize(R.dimen.fab_text_box_width_size),
                screenWidth - fabGroup.layoutParams.width
            )

            val initialWidth = textContainer.width.toFloat()

            val titleTextView: TextView? = findViewById(R.id.title_textView)
            if (primaryText.isNullOrEmpty()) {
                titleTextView?.visibility = View.GONE
            } else {
                titleTextView?.text = primaryText
                titleTextView?.visibility = View.VISIBLE
            }

            val descriptionTextView: TextView? = findViewById(R.id.description_textView)
            if (secondaryText.isNullOrEmpty()) {
                descriptionTextView?.visibility = View.GONE
            } else {
                descriptionTextView?.text = secondaryText
                descriptionTextView?.visibility = View.VISIBLE
            }

            if (primaryText.isNullOrEmpty() && secondaryText.isNullOrEmpty()) {
                textContainer.visibility = View.GONE
            } else {
                textContainer.visibility = View.VISIBLE
                isMessageFinishedShowing = false

                textContainer.post {
                    val alphaAnimator =
                        ObjectAnimator.ofFloat(textContainer, "alpha", 1f, 0f).apply {
                            duration = 250
                        }

                    val translationAnimator = ObjectAnimator.ofFloat(
                        textContainer,
                        "translationX",
                        0f,
                        initialWidth * -0.1f
                    ).apply {
                        duration = 400
                    }

                    val animatorSet = AnimatorSet()
                    animatorSet.playTogether(alphaAnimator, translationAnimator)
                    animatorSet.addListener(object : AnimatorListenerAdapter() {
                        override fun onAnimationEnd(animation: Animator) {
                            textContainer.visibility = View.GONE

                            if (!isMessageFinishedShowing) {
                                params.width = WindowManager.LayoutParams.WRAP_CONTENT
                                isMessageFinishedShowing = true
                            }

                            onAutoDismiss()
                        }
                    })
                    animatorSet.startDelay = 2.seconds.inWholeMilliseconds
                    animatorSet.start()
                }
            }


            overlay.setOnTouchListener(object : View.OnTouchListener {
                override fun onTouch(v: View, event: MotionEvent): Boolean {
                    when (event.action) {
                        MotionEvent.ACTION_DOWN -> {
                            initialX = params.x
                            initialY = params.y
                            initialTouchX = event.rawX
                            initialTouchY = event.rawY
                            return true
                        }

                        MotionEvent.ACTION_UP -> {
                            val xDiff = event.rawX - initialTouchX
                            val yDiff = event.rawY - initialTouchY

                            if (abs(xDiff) < 10.0f && abs(yDiff) < 10.0f) {
                                onAction(Unit)
                                return true
                            }
                            return true
                        }

                        MotionEvent.ACTION_MOVE -> {
                            if (!isMessageFinishedShowing) return true

                            params.x = 0
                            params.y = initialY - (event.rawY - initialTouchY).toInt()

                            positionData?.let {
                                it.y = params.y
                            }

                            windowManager.updateViewLayout(overlay, params)
                            return true
                        }
                    }
                    return false
                }
            })
        }

        return overlay
    }

    private fun getScreenWidth(windowManager: WindowManager): Int = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
        val bounds = windowManager.currentWindowMetrics.bounds
        bounds.width()
    } else {
        val displayMetrics = DisplayMetrics()
        windowManager.defaultDisplay.getMetrics(displayMetrics)
        displayMetrics.widthPixels
    }

}