package app.appnomix.sdk.internal.ui.onboarding

import android.app.Dialog
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.ImageView
import android.widget.TextView
import androidx.activity.enableEdgeToEdge
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import app.appnomix.sdk.R
import app.appnomix.sdk.external.AppnomixEvent
import app.appnomix.sdk.external.CouponsSdkFacade
import app.appnomix.sdk.external.CouponsSdkFacade.onAppnomixEvent
import app.appnomix.sdk.external.SdkRequirement
import app.appnomix.sdk.internal.Deps
import app.appnomix.sdk.internal.analytics.Analytics
import app.appnomix.sdk.internal.analytics.AnalyticsEvent
import app.appnomix.sdk.internal.analytics.AnalyticsName
import app.appnomix.sdk.internal.data.SdkConfig

private const val PROMINENT_DIALOG_ENABLED = false

class OnboardingActivity : AppCompatActivity() {
    private val sdkConfig: SdkConfig by lazy { Deps.sdkConfig() }
    private var currentOnboardingScreen: OnboardingScreenType? = null
    private var contentContainer: ViewGroup? = null
    private var buttonContainer: ViewGroup? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_onboarding)
        enableEdgeToEdge()

        val mainContainer = findViewById<View>(R.id.main)
        contentContainer = mainContainer.findViewById(R.id.content_layout)
        buttonContainer = mainContainer.findViewById(R.id.button_layout)
        ViewCompat.setOnApplyWindowInsetsListener(mainContainer) { v, insets ->
            val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom)
            insets
        }
        if (intent.action == TO_ACCESSIBILITY_STEP_ACTION) {
            sdkConfig.onboardingFinished = false
            currentOnboardingScreen = OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_1
        }
        onAppnomixEvent(AppnomixEvent.ONBOARDING_STARTED)
        moveToNextScreen()
    }

    override fun onStart() {
        super.onStart()
        Analytics.trackEvent(
            AnalyticsEvent.Builder(
                eventCategory = AnalyticsName.EventCategory.ONBOARDING
            ).apply { name = AnalyticsName.EventType.ONBOARDING_SCREEN_SHOWN }.build()
        )
    }

    override fun onResume() {
        super.onResume()
        updateToLastScreenIfPossible(currentOnboardingScreen)

        val unmetRequirements = CouponsSdkFacade.unmetRequirements()
        if (!unmetRequirements.contains(SdkRequirement.ACCESSIBILITY) &&
            currentOnboardingScreen == OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2) {
            moveToNextScreen()
        }
    }

    private fun updateToLastScreenIfPossible(fromScreen: OnboardingScreenType?) {
        val unmetRequirements = CouponsSdkFacade.unmetRequirements()
        if (fromScreen in listOf(
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_EXPLAINER,
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_1,
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2,
            )
            && !unmetRequirements.contains(SdkRequirement.ACCESSIBILITY)
        ) {
            currentOnboardingScreen = OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2
        }
    }

    private fun moveToNextScreen() {
        updateToLastScreenIfPossible(currentOnboardingScreen)

        val nextScreenOrder = currentOnboardingScreen?.order?.plus(1) ?: 0
        val nextScreen = OnboardingScreenType.entries.find { it.order == nextScreenOrder }

        if (nextScreen != null) {
            val views = createOnboardingView(nextScreen)
            contentContainer?.removeAllViews()
            contentContainer?.addView(views.contentView)

            buttonContainer?.removeAllViews()
            buttonContainer?.addView(views.buttonView)

            currentOnboardingScreen = nextScreen

            if (currentOnboardingScreen == OnboardingScreenType.SETUP_DONE) {
                Analytics.trackEvent(
                    AnalyticsEvent.Builder(
                        eventCategory = AnalyticsName.EventCategory.INSTALL
                    ).apply { name = AnalyticsName.EventType.EXTENSION_ACTIVATED }.build()
                )
            }
        } else {
            endOnboarding()
        }
    }

    private fun createOnboardingView(screenType: OnboardingScreenType): OnboardingViews {
        val onboardingContent = View.inflate(this, R.layout.onboarding, null)
        val onboardingButtons = View.inflate(this, R.layout.onboarding_buttons, null)
        ONBOARDING_SCREENS[screenType]?.let { screenData ->
            val logoImageView = onboardingContent.findViewById<ImageView>(R.id.logo_imageView)
            val titleTextView = onboardingContent.findViewById<TextView>(R.id.title_textView)
            val contentViewContainer =
                onboardingContent.findViewById<ViewGroup>(R.id.template_layout)

            val primaryContentButton = onboardingContent.findViewById<Button>(R.id.primary_button)
            val secondaryContentButton =
                onboardingContent.findViewById<Button>(R.id.secondary_button)
            val copywriteContentTextView =
                this@OnboardingActivity.findViewById<View>(R.id.content_copyright_textView)

            val primaryOnboardingButton =
                onboardingButtons.findViewById<Button>(R.id.primary_button)
            val secondaryOnboardingButton =
                onboardingButtons.findViewById<Button>(R.id.secondary_button)
            val copywriteButtonsTextView =
                this@OnboardingActivity.findViewById<View>(R.id.buttons_copyright_textView)

            val drawable = packageManager.getApplicationIcon(packageName)
            logoImageView?.setImageDrawable(drawable)

            screenData.titleRes?.let {
                val content = screenData.contentWithAppName(context = this, stringRes = it)
                titleTextView.text = content
            }

            val contentLayoutResToUse = when (screenData.content) {
                is OnboardingContent.ImageText -> R.layout.onboarding_content_image_text
                is OnboardingContent.FirstAccessibility -> R.layout.onboarding_access_1
                is OnboardingContent.SecondAccessibility -> R.layout.onboarding_access_2
                is OnboardingContent.TextWithMore -> R.layout.onboarding_content_text_action
                is OnboardingContent.Image -> R.layout.onboarding_content_image
            }
            val contentLayoutToUse = View.inflate(this, contentLayoutResToUse, null)

            val actionTextListener: (ContentState) -> Unit = { contentState ->
                val (primaryButton, secondaryButton) = when (contentState) {
                    ContentState.EXPANDED -> {
                        primaryOnboardingButton.visibility = View.GONE
                        secondaryOnboardingButton.visibility = View.GONE
                        copywriteButtonsTextView.visibility = View.GONE

                        primaryContentButton.visibility = View.VISIBLE
                        secondaryContentButton.visibility = View.VISIBLE
                        copywriteContentTextView.visibility = View.VISIBLE

                        primaryContentButton to secondaryContentButton
                    }

                    ContentState.COLLAPSED -> {
                        primaryContentButton.visibility = View.GONE
                        secondaryContentButton.visibility = View.GONE
                        copywriteContentTextView.visibility = View.GONE

                        primaryOnboardingButton.visibility = View.VISIBLE
                        secondaryOnboardingButton.visibility = View.VISIBLE
                        copywriteButtonsTextView.visibility = View.VISIBLE

                        primaryOnboardingButton to secondaryOnboardingButton
                    }
                }

                primaryButton.setText(screenData.mainButtonTitleRes)
                primaryButton.setText(screenData.mainButtonTitleRes)

                primaryButton.setOnClickListener {
                    if (currentOnboardingScreen?.order == OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2.order) {
                        showProminentDialog()
                    } else {
                        handlePrimaryAction(screenType)
                    }
                }

                if (screenData.secondaryButtonTitleRes != null) {
                    val content = screenData.contentWithAppName(
                        context = this,
                        stringRes = screenData.secondaryButtonTitleRes
                    )
                    secondaryButton.text = content
                    if (screenData.secondaryButtonTextColorRes != null) {
                        secondaryButton.setTextColor(
                            ContextCompat.getColor(
                                this,
                                screenData.secondaryButtonTextColorRes
                            )
                        )
                    }
                    secondaryButton.visibility = View.VISIBLE
                    secondaryButton.setOnClickListener {
                        handleSecondaryAction(screenType)
                    }

                    secondaryButton.setBackgroundResource(screenData.secondaryButtonDrawableRes)
                } else {
                    secondaryButton.visibility = View.INVISIBLE
                }
            }

            actionTextListener(ContentState.COLLAPSED)

            configureContentView(
                view = contentLayoutToUse,
                screenData = screenData,
                actionTextListener = actionTextListener
            )
            contentViewContainer.addView(contentLayoutToUse)
        }
        return OnboardingViews(onboardingContent, onboardingButtons)
    }

    private fun handlePrimaryAction(screenType: OnboardingScreenType) {
        when (screenType) {
            OnboardingScreenType.GET_STARTED_1 -> {
                Analytics.trackEvent(
                    AnalyticsEvent.Builder(
                        eventCategory = AnalyticsName.EventCategory.ONBOARDING
                    ).apply { name = AnalyticsName.EventType.ONBOARDING_GET_STARTED }.build()
                )
            }

            else -> {}
        }
        moveToNextScreen()
    }

    private fun handleSecondaryAction(screenType: OnboardingScreenType) {
        if (screenType in listOf(
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_EXPLAINER,
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_1,
                OnboardingScreenType.ACCESSIBILITY_PERMISSION_STEP_2,
            )
        ) {
            Analytics.trackEvent(
                AnalyticsEvent.Builder(
                    eventCategory = AnalyticsName.EventCategory.ONBOARDING
                ).apply { name = AnalyticsName.EventType.ONBOARDING_SKIP_FOR_NOW }.build()
            )
            endOnboarding()
        } else {
            moveToNextScreen()
        }
    }

    private fun endOnboarding() {
        if (CouponsSdkFacade.isAccessibilityServiceEnabled()) {
            CouponsSdkFacade.onAppnomixEvent(AppnomixEvent.ONBOARDING_FINISHED)
        } else {
            CouponsSdkFacade.onAppnomixEvent(AppnomixEvent.ONBOARDING_ABANDONED)
        }
        finish()
        sdkConfig.onboardingFinished = true
    }

    private fun configureContentView(
        view: View,
        screenData: OnboardingScreenData,
        actionTextListener: (ContentState) -> Unit
    ) {
        when (screenData.content) {
            is OnboardingContent.Image -> {
                val imageView: ImageView = view.findViewById(R.id.onboarding_imageView)
                imageView.setImageDrawable(
                    ContextCompat.getDrawable(
                        this,
                        screenData.content.imageDrawableRes
                    )
                )
            }

            is OnboardingContent.ImageText -> {
                val imageView: ImageView = view.findViewById(R.id.onboarding_imageView)
                val textView: TextView = view.findViewById(R.id.description_textView)
                imageView.setImageDrawable(
                    ContextCompat.getDrawable(
                        this,
                        screenData.content.imageDrawableRes
                    )
                )
                val content = screenData.contentWithAppName(
                    context = this,
                    stringRes = screenData.content.textRes
                )
                textView.text = content
            }

            is OnboardingContent.FirstAccessibility -> {
                val descriptionTextView: TextView = view.findViewById(R.id.description_textView)

                val appIconImageView: ImageView = view.findViewById(R.id.logo_imageView)
                val appNameTextView: TextView = view.findViewById(R.id.appName_textView)

                val content = screenData.contentWithAppName(
                    context = this,
                    stringRes = screenData.content.textRes
                )
                descriptionTextView.text = content

                val drawable = packageManager.getApplicationIcon(packageName)
                appIconImageView.setImageDrawable(drawable)

                appNameTextView.setText(applicationInfo.labelRes)
            }

            is OnboardingContent.SecondAccessibility -> {
                val descriptionTextView: TextView = view.findViewById(R.id.description_textView)

                val appNameTextView: TextView = view.findViewById(R.id.appName_textView)
                val useAppTextView: TextView = view.findViewById(R.id.useApp_textView)
                val appShortcutTextView: TextView = view.findViewById(R.id.appShortcut_textView)

                descriptionTextView.text = screenData.contentWithAppName(
                    context = this,
                    stringRes = screenData.content.titleRes
                )
                useAppTextView.text = screenData.contentWithAppName(
                    context = this,
                    stringRes = screenData.content.useAppRes
                )
                appShortcutTextView.text = screenData.contentWithAppName(
                    context = this,
                    stringRes = screenData.content.appShortcutRes
                )

                appNameTextView.setText(applicationInfo.labelRes)
            }

            is OnboardingContent.TextWithMore -> {
                val descriptionTextView: TextView = view.findViewById(R.id.description_textView)
                val actionTextView: TextView = view.findViewById(R.id.action_textView)
                actionTextView.setOnClickListener {
                    if (actionTextView.text == getString(R.string.more_information)) {
                        actionTextView.text = getString(R.string.less)
                        descriptionTextView.maxLines = Int.MAX_VALUE
                        actionTextListener(ContentState.EXPANDED)
                    } else {
                        actionTextView.text = getString(R.string.more_information)
                        descriptionTextView.maxLines = 7
                        actionTextListener(ContentState.COLLAPSED)
                    }
                }
                descriptionTextView.setText(screenData.content.textRes)
            }
        }
    }

    private fun showProminentDialog() {
        if (!PROMINENT_DIALOG_ENABLED) {
            CouponsSdkFacade.goToToAccessibilitySettings(this)
            return
        }
        val alert = Dialog(this)
        val layout = View.inflate(this, R.layout.dialog_layout, null)
        layout.findViewById<Button>(R.id.negative_button).setOnClickListener {
            alert.dismiss()
            endOnboarding()
        }
        layout.findViewById<Button>(R.id.positive_button).setOnClickListener {
            alert.dismiss()
            CouponsSdkFacade.goToToAccessibilitySettings(this)
        }
        alert.setContentView(layout)
        alert.setCancelable(true)
        alert.window?.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        alert.show()
    }

    enum class ContentState {
        EXPANDED,
        COLLAPSED
    }

    data class OnboardingViews(
        val contentView: View,
        val buttonView: View
    )

    companion object {
        const val TO_ACCESSIBILITY_STEP_ACTION = "action_to_accessibility"
    }
}
