package app.peac.core.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.core.type.filter.AnnotationTypeFilter;

import java.lang.annotation.Annotation;
import java.util.Set;
import java.util.stream.Collectors;

@Slf4j
public abstract class AnnotationProcessor<T extends Annotation> {

    private static final String BASE_PACKAGE = "app.peac";

    protected abstract Class<T> getAnnotationType();

    protected abstract void initializeAnnotation(String className);

    protected void beforeAllAnnotations(Set<String> classes) {
        log.info("Found {} classes with '{}' annotation", classes.size(), getAnnotationType().getSimpleName());
    }

    protected void initialize() {
        var scanner = new ClassPathScanningCandidateComponentProvider(false);
        scanner.addIncludeFilter(new AnnotationTypeFilter(getAnnotationType()));

        var allClassesWithAnnotation = scanner.findCandidateComponents(BASE_PACKAGE).stream()
                .map(BeanDefinition::getBeanClassName)
                .collect(Collectors.toSet());

        beforeAllAnnotations(allClassesWithAnnotation);

        allClassesWithAnnotation.forEach(this::initializeAnnotation);

        afterAllAnnotations(allClassesWithAnnotation);
    }

    protected void afterAllAnnotations(Set<String> classes) {
        log.info("{} classes processed with '{}' annotation", classes.size(), getAnnotationType().getSimpleName());
    }
}
