/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.mongodb.util;

import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import com.mongodb.DBRef;
import com.mongodb.MongoClientSettings;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.StringJoiner;
import java.util.function.Function;
import java.util.stream.StreamSupport;
import org.bson.BSONObject;
import org.bson.BsonBinary;
import org.bson.BsonBoolean;
import org.bson.BsonDouble;
import org.bson.BsonInt32;
import org.bson.BsonInt64;
import org.bson.BsonObjectId;
import org.bson.BsonString;
import org.bson.BsonValue;
import org.bson.Document;
import org.bson.codecs.DocumentCodec;
import org.bson.codecs.configuration.CodecRegistry;
import org.bson.conversions.Bson;
import org.bson.json.JsonParseException;
import org.bson.types.ObjectId;
import org.springframework.core.convert.converter.Converter;
import org.springframework.data.mongodb.CodecRegistryProvider;
import org.springframework.data.mongodb.util.EmptyDocument;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

public class BsonUtils {
    public static final Document EMPTY_DOCUMENT = new EmptyDocument();

    @Nullable
    public static <T> T get(Bson bson, String key) {
        return (T)BsonUtils.asMap(bson).get(key);
    }

    public static Map<String, Object> asMap(Bson bson) {
        return BsonUtils.asMap(bson, MongoClientSettings.getDefaultCodecRegistry());
    }

    public static Map<String, Object> asMap(@Nullable Bson bson, CodecRegistry codecRegistry) {
        if (bson == null) {
            return Collections.emptyMap();
        }
        if (bson instanceof Document) {
            Document document = (Document)bson;
            return document;
        }
        if (bson instanceof BasicDBObject) {
            BasicDBObject dbo = (BasicDBObject)bson;
            return dbo;
        }
        if (bson instanceof DBObject) {
            DBObject dbo = (DBObject)((Object)bson);
            return dbo.toMap();
        }
        return new Document(bson.toBsonDocument(Document.class, codecRegistry));
    }

    public static Document asDocument(Bson bson) {
        return BsonUtils.asDocument(bson, MongoClientSettings.getDefaultCodecRegistry());
    }

    public static Document asDocument(Bson bson, CodecRegistry codecRegistry) {
        Map<String, Object> map = BsonUtils.asMap(bson, codecRegistry);
        if (map instanceof Document) {
            return (Document)map;
        }
        return new Document(map);
    }

    public static Document asMutableDocument(Bson bson) {
        if (bson instanceof EmptyDocument) {
            bson = new Document(BsonUtils.asDocument(bson));
        }
        if (bson instanceof Document) {
            return (Document)bson;
        }
        Map<String, Object> map = BsonUtils.asMap(bson);
        if (map instanceof Document) {
            return (Document)map;
        }
        return new Document(map);
    }

    public static void addToMap(Bson bson, String key, @Nullable Object value) {
        if (bson instanceof Document) {
            ((Document)bson).put(key, value);
            return;
        }
        if (bson instanceof BSONObject) {
            ((BSONObject)((Object)bson)).put(key, value);
            return;
        }
        throw new IllegalArgumentException(String.format("Cannot add key/value pair to %s; as map given Bson must be a Document or BSONObject", bson.getClass()));
    }

    public static void addAllToMap(Bson target, Map<String, ?> source) {
        if (target instanceof Document) {
            ((Document)target).putAll((Map<? extends String, ?>)source);
            return;
        }
        if (target instanceof BSONObject) {
            ((BSONObject)((Object)target)).putAll(source);
            return;
        }
        throw new IllegalArgumentException(String.format("Cannot add all to %s; Given Bson must be a Document or BSONObject.", target.getClass()));
    }

    public static boolean contains(Bson bson, String key, @Nullable Object value) {
        if (bson instanceof Document) {
            Document doc = (Document)bson;
            return doc.containsKey(key) && ObjectUtils.nullSafeEquals(doc.get(key), value);
        }
        if (bson instanceof BSONObject) {
            BSONObject bsonObject = (BSONObject)((Object)bson);
            return bsonObject.containsField(key) && ObjectUtils.nullSafeEquals(bsonObject.get(key), value);
        }
        Map<String, Object> map = BsonUtils.asMap(bson);
        return map.containsKey(key) && ObjectUtils.nullSafeEquals(map.get(key), value);
    }

    public static boolean removeNullId(Bson bson) {
        if (!BsonUtils.contains(bson, "_id", null)) {
            return false;
        }
        BsonUtils.removeFrom(bson, "_id");
        return true;
    }

    static void removeFrom(Bson bson, String key) {
        if (bson instanceof Document) {
            ((Document)bson).remove(key);
            return;
        }
        if (bson instanceof BSONObject) {
            ((BSONObject)((Object)bson)).removeField(key);
            return;
        }
        throw new IllegalArgumentException(String.format("Cannot remove from %s given Bson must be a Document or BSONObject.", bson.getClass()));
    }

    public static Object toJavaType(BsonValue value) {
        switch (value.getBsonType()) {
            case INT32: {
                return value.asInt32().getValue();
            }
            case INT64: {
                return value.asInt64().getValue();
            }
            case STRING: {
                return value.asString().getValue();
            }
            case DECIMAL128: {
                return value.asDecimal128().doubleValue();
            }
            case DOUBLE: {
                return value.asDouble().getValue();
            }
            case BOOLEAN: {
                return value.asBoolean().getValue();
            }
            case OBJECT_ID: {
                return value.asObjectId().getValue();
            }
            case DB_POINTER: {
                return new DBRef(value.asDBPointer().getNamespace(), value.asDBPointer().getId());
            }
            case BINARY: {
                return value.asBinary().getData();
            }
            case DATE_TIME: {
                return new Date(value.asDateTime().getValue());
            }
            case SYMBOL: {
                return value.asSymbol().getSymbol();
            }
            case ARRAY: {
                return value.asArray().toArray();
            }
            case DOCUMENT: {
                return Document.parse(value.asDocument().toJson());
            }
        }
        return value;
    }

    public static BsonValue simpleToBsonValue(Object source) {
        if (source instanceof BsonValue) {
            return (BsonValue)source;
        }
        if (source instanceof ObjectId) {
            return new BsonObjectId((ObjectId)source);
        }
        if (source instanceof String) {
            return new BsonString((String)source);
        }
        if (source instanceof Double) {
            return new BsonDouble((Double)source);
        }
        if (source instanceof Integer) {
            return new BsonInt32((Integer)source);
        }
        if (source instanceof Long) {
            return new BsonInt64((Long)source);
        }
        if (source instanceof byte[]) {
            return new BsonBinary((byte[])source);
        }
        if (source instanceof Boolean) {
            return new BsonBoolean((Boolean)source);
        }
        if (source instanceof Float) {
            return new BsonDouble(((Float)source).floatValue());
        }
        throw new IllegalArgumentException(String.format("Unable to convert %s (%s) to BsonValue.", source, source != null ? source.getClass().getName() : "null"));
    }

    public static Document merge(Document ... documents) {
        if (ObjectUtils.isEmpty(documents)) {
            return new Document();
        }
        if (documents.length == 1) {
            return documents[0];
        }
        Document target = new Document();
        Arrays.asList(documents).forEach(target::putAll);
        return target;
    }

    public static Document toDocumentOrElse(String source, Function<String, Document> orElse) {
        if (source.stripLeading().startsWith("{")) {
            return Document.parse(source);
        }
        return orElse.apply(source);
    }

    @Nullable
    public static String toJson(@Nullable Document source) {
        if (source == null) {
            return null;
        }
        try {
            return source.toJson();
        }
        catch (Exception e) {
            return BsonUtils.toJson((Object)source);
        }
    }

    public static boolean isJsonDocument(@Nullable String value) {
        if (!StringUtils.hasText(value)) {
            return false;
        }
        String potentialJson = value.trim();
        return potentialJson.startsWith("{") && potentialJson.endsWith("}");
    }

    public static boolean isJsonArray(@Nullable String value) {
        return StringUtils.hasText(value) && value.startsWith("[") && value.endsWith("]");
    }

    public static Document parse(String json, @Nullable CodecRegistryProvider codecRegistryProvider) {
        Assert.notNull((Object)json, "Json must not be null");
        if (codecRegistryProvider == null) {
            return Document.parse(json);
        }
        return Document.parse(json, codecRegistryProvider.getCodecFor(Document.class).orElseGet(() -> new DocumentCodec(codecRegistryProvider.getCodecRegistry())));
    }

    @Nullable
    public static Object resolveValue(Bson bson, String key) {
        Map<String, Object> source = BsonUtils.asMap(bson);
        if (source.containsKey(key) || !key.contains(".")) {
            return source.get(key);
        }
        String[] parts = key.split("\\.");
        for (int i = 1; i < parts.length; ++i) {
            Object result = source.get(parts[i - 1]);
            if (!(result instanceof Bson)) {
                return null;
            }
            source = BsonUtils.asMap((Bson)result);
        }
        return source.get(parts[parts.length - 1]);
    }

    public static boolean hasValue(Bson bson, String key) {
        Map<String, Object> source = BsonUtils.asMap(bson);
        if (source.get(key) != null) {
            return true;
        }
        if (!key.contains(".")) {
            return false;
        }
        String[] parts = key.split("\\.");
        for (int i = 1; i < parts.length; ++i) {
            Object result = source.get(parts[i - 1]);
            if ((source = BsonUtils.getAsMap(result)) != null) continue;
            return false;
        }
        return source.containsKey(parts[parts.length - 1]);
    }

    @Nullable
    private static Map<String, Object> getAsMap(Object source) {
        if (source instanceof Document) {
            return (Document)source;
        }
        if (source instanceof BasicDBObject) {
            return (BasicDBObject)source;
        }
        if (source instanceof DBObject) {
            return ((DBObject)source).toMap();
        }
        if (source instanceof Map) {
            return (Map)source;
        }
        return null;
    }

    public static Bson asBson(Object source) {
        if (source instanceof Document) {
            return (Document)source;
        }
        if (source instanceof BasicDBObject) {
            return (BasicDBObject)source;
        }
        if (source instanceof DBObject) {
            return new Document(((DBObject)source).toMap());
        }
        if (source instanceof Map) {
            return new Document((Map)source);
        }
        throw new IllegalArgumentException(String.format("Cannot convert %s to Bson", source));
    }

    public static boolean supportsBson(Object source) {
        return source instanceof DBObject || source instanceof Map;
    }

    public static Collection<?> asCollection(Object source) {
        if (source instanceof Collection) {
            return (Collection)source;
        }
        return source.getClass().isArray() ? CollectionUtils.arrayToList(source) : Collections.singleton(source);
    }

    @Nullable
    private static String toJson(@Nullable Object value) {
        if (value == null) {
            return null;
        }
        try {
            return value instanceof Document ? ((Document)value).toJson(MongoClientSettings.getDefaultCodecRegistry().get(Document.class)) : BsonUtils.serializeValue(value);
        }
        catch (Exception e) {
            if (value instanceof Collection) {
                return BsonUtils.toString((Collection)value);
            }
            if (value instanceof Map) {
                return BsonUtils.toString((Map)value);
            }
            if (ObjectUtils.isArray(value)) {
                return BsonUtils.toString(Arrays.asList(ObjectUtils.toObjectArray(value)));
            }
            throw e instanceof JsonParseException ? (JsonParseException)e : new JsonParseException(e);
        }
    }

    private static String serializeValue(@Nullable Object value) {
        if (value == null) {
            return "null";
        }
        String documentJson = new Document("toBeEncoded", value).toJson();
        return documentJson.substring(documentJson.indexOf(58) + 1, documentJson.length() - 1).trim();
    }

    private static String toString(Map<?, ?> source) {
        return BsonUtils.iterableToDelimitedString(source.entrySet(), "{ ", " }", entry -> String.format("\"%s\" : %s", entry.getKey(), BsonUtils.toJson(entry.getValue())));
    }

    private static String toString(Collection<?> source) {
        return BsonUtils.iterableToDelimitedString(source, "[ ", " ]", BsonUtils::toJson);
    }

    private static <T> String iterableToDelimitedString(Iterable<T> source, String prefix, String suffix, Converter<? super T, String> transformer) {
        StringJoiner joiner = new StringJoiner(", ", prefix, suffix);
        StreamSupport.stream(source.spliterator(), false).map(transformer::convert).forEach(joiner::add);
        return joiner.toString();
    }
}

