package app.pivo.android.micsdk.events

import app.pivo.android.micsdk.model.*

/**
 * Created by murodjon on 2021/01/19
 *
 * [HeadsetEvent] is a list of events, which are sent by Pivo Headset controller through  [PivoEventBus]
 */
class HeadsetEvent {

    /**
     * [ConnectionComplete] event is sent if [IPivoMicSdk.connectTo] completes the connection to Pivo device.
     */
    class ConnectionComplete

    /**
     * ConnectionFailure event is published if Pivo disconnects from an application.
     */
    class ConnectionFailure

    /**
     * [Scanning] is sent by [PivoEventBus] with found Headset devices while headset scanning process.
     */
    class Scanning(private val device: PivoMicDevice) {
        fun getDevice(): PivoMicDevice {
            return device
        }
    }

    /**
     * [NameChanged] is sent by [PivoEventBus] after headset name is changed successfully.
     */
    class NameChanged(private val name: String) {
        fun getName() = name
    }

    /**
     * [BatteryChanged] is sent by sdk to inform the app about remaining battery level.
     */
    class BatteryChanged(private val level: Int) {
        fun getLevel() = level
    }

    /**
     * [MacAddress] is sent by [PivoEventBus] after receiving Mac address from pivo.
     */
    class MacAddress(private val macAddress: String) {
        fun getMacAddress() = macAddress
    }

    /**
     * [SerialNumber] is sent by [PivoEventBus] after receiving serial number from pivo.
     */
    class SerialNumber(private val serialNumber: String) {
        fun getSerialNumber() = serialNumber
    }

    /**
     * [MinusButton] is sent by [PivoEventBus] after pressing minus button on headset
     */
    class MinusButton(private val state: Int) {
        fun getState() = state
        fun isPressed() = state == 0
    }

    /**
     * [PlusButton] is sent by [PivoEventBus] after pressing plus button on headset
     */
    class PlusButton(private val state: Int) {
        fun getState() = state
        fun isPressed() = state == 0
    }

    /**
     * [MainButton] is sent by [PivoEventBus] after pressing main button on headset
     */
    class MainButton(private val state: Int) {
        fun getState() = state
        fun isPressed() = state == 0
    }

    /**
     * [SystemStatus] is sent by [PivoEventBus] if there's device change notification
     */
    class SystemStatus(private val status: Status) {
        fun getStatus() = status
    }

    /**
     * [BypassStatus] is sent by [PivoEventBus] if bypass changes
     */
    class BypassStatus(private val status: Bypass) {
        fun getStatus() = status
    }

    /**
     * [DeviceConnectivity] is sent by [PivoEventBus] if there's plugged earphone
     */
    class DeviceConnectivity(private val status: EarphoneStatus) {
        fun getStatus() = status
    }

    /**
     * [InvalidReply] is sent by [PivoEventBus] if invalid or incomplete bytes received
     */
    class InvalidReply(private val text: String) {
        fun getOriginalReply() = text
    }

    /**
     * [LEDCntrlType] is sent by [PivoEventBus] after changing [LED] control
     */
    class LEDCntrlType(private val led: LED) {
        fun getType() = led
    }

    /**
     * [MicModeCntrlType] is sent by [PivoEventBus] after changing [MicMode] control
     */
    class MicModeCntrlType(private val micMode: MicMode) {
        fun getType() = micMode
    }

    /**
     * [SoundCntrlType] is sent by [PivoEventBus] after changing [Sound] control
     */
    class SoundCntrlType(private val sound: Sound) {
        fun getType() = sound
    }

    /**
     * [NoiseCntrlType] is sent by [PivoEventBus] after changing [Noise] control
     */
    class NoiseCntrlType(private val noise: Noise) {
        fun getType() = noise
    }

    /**
     * [DeviceInfoReceived] is sent by [PivoEventBus] if getting a device info is requested
     */
    class DeviceInfoReceived(private val version: DeviceInfo) {
        fun getVersion() = version
    }
}