package app.pivo.android.micsdk.util

import app.pivo.android.micsdk.model.*

/**
 * Created by murodjon on 2021/01/19
 *
 * This [CommandUtil] class is used to generate command bytes to send to device through bluetooth.
 */
internal object CommandUtil {

    /**
     * get check connection bytes
     */
    fun getDeviceInfoBytes(): ByteArray =
        byteArrayOf(0x05, 0xBF.toByte(), 0x03, 0x00, 0x00, 0x00)

    /**
     * get device status
     */
    fun getDeviceStatusBytes(): ByteArray =
        byteArrayOf(0x05, 0xBF.toByte(), 0x1D, 0x00, 0x00, 0x00)

    /**
     * get battery level bytes
     */
    fun getBatteryLevelBytes() = byteArrayOf(0x05, 0xBF.toByte(), 0x04, 0x00, 0x00, 0x00)

    /**
     * get LED control bytes
     */
    fun getLEDControlBytes(led: LED): ByteArray {
        return when (led) {
            LED.BLINKING_WHITE -> byteArrayOf(0x05, 0xBF.toByte(), 0x10, 0x00, 0x00, 0x01)
            LED.WHITE -> byteArrayOf(0x05, 0xBF.toByte(), 0x10, 0x00, 0x00, 0x02)
            LED.BLINKING_BLUE -> byteArrayOf(0x05, 0xBF.toByte(), 0x10, 0x00, 0x00, 0x03)
            LED.BLUE -> byteArrayOf(0x05, 0xBF.toByte(), 0x10, 0x00, 0x00, 0x04)
            LED.BLINKING_RED -> byteArrayOf(0x05, 0xBF.toByte(), 0x10, 0x00, 0x00, 0x05)
            LED.RED -> byteArrayOf(0x05, 0xBF.toByte(), 0x10, 0x00, 0x00, 0x06)
            else -> byteArrayOf(0x05, 0xBF.toByte(), 0x10, 0x00, 0x00, 0x00)
        }
    }

    /**
     * get Noise control bytes
     */
    fun getNoiseControlBytes(noise: Noise): ByteArray {
        return if (noise == Noise.ENABLED) {
            byteArrayOf(0x05, 0xBF.toByte(), 0x11, 0x00, 0x00, 0x01)
        } else byteArrayOf(0x05, 0xBF.toByte(), 0x11, 0x00, 0x00, 0x00)
    }

    /**
     * get Sound control bytes
     */
    fun getSoundControlBytes(sound: Sound): ByteArray {
        return when (sound) {
            Sound.BASS -> byteArrayOf(0x05, 0xBF.toByte(), 0x12, 0x00, 0x00, 0x02)
            Sound.SMOOTH -> byteArrayOf(0x05, 0xBF.toByte(), 0x12, 0x00, 0x00, 0x03)
            Sound.VIBRANT -> byteArrayOf(0x05, 0xBF.toByte(), 0x12, 0x00, 0x00, 0x04)
            Sound.VOICE -> byteArrayOf(0x05, 0xBF.toByte(), 0x12, 0x00, 0x00, 0x05)
            Sound.TREBLE -> byteArrayOf(0x05, 0xBF.toByte(), 0x12, 0x00, 0x00, 0x06)
            else -> byteArrayOf(0x05, 0xBF.toByte(), 0x12, 0x00, 0x00, 0x01)
        }
    }

    /**
     * select mic mode
     */
    fun selectMicMode(micMode: MicMode): ByteArray {
        return when (micMode) {
            MicMode.EXTERNAL_MIC_ONLY -> {
                byteArrayOf(0x05, 0xBF.toByte(), 0x13, 0x00, 0x00, 0x03)
            }
            MicMode.INTERNAL_MIC_ONLY -> {
                byteArrayOf(0x05, 0xBF.toByte(), 0x13, 0x00, 0x00, 0x02)
            }
            MicMode.ALL_MIC_OFF -> {
                byteArrayOf(0x05, 0xBF.toByte(), 0x13, 0x00, 0x00, 0xFF.toByte())
            }
            else -> {
                byteArrayOf(0x05, 0xBF.toByte(), 0x13, 0x00, 0x00, 0x01)
            }
        }
    }

    /**
     * set bypass mode
     */
    fun setBypassMode(bypass: Bypass): ByteArray {
        return when (bypass) {
            Bypass.ON -> {
                byteArrayOf(0x05, 0xBF.toByte(), 0x14, 0x00, 0x00, 0xA1.toByte())
            }
            Bypass.OFF -> {
                byteArrayOf(0x05, 0xBF.toByte(), 0x14, 0x00, 0x00, 0xA0.toByte())
            }
        }
    }

    /**
     * get change name bytes
     * @param name is new name for headset
     * @return an array of byte values is returned from function
     */
    fun getChangeNameBytes(name: String): ByteArray? {
        val length = name.length
        if (length == 0) return null
        // final byte array size for changing name is 5+(length of name)
        val bytes = ByteArray(length + 5)
        bytes[0] = 0x05.toByte()
        bytes[1] = 0xBF.toByte()
        bytes[2] = 0x7F.toByte()
        bytes[3] = length.toByte()
        bytes[4] = 0x00.toByte()
        name.forEachIndexed { i, v ->
            bytes[5 + i] = name[i].toByte()
        }
        return bytes
    }

    /**
     * get mac address requester bytes
     * @return an array of byte values
     */
    fun getMacAddressBytes(): ByteArray =
        byteArrayOf(0x05, 0xBF.toByte(), 0x1B, 0x00, 0x00, 0x00)

    /**
     * get serial number requester bytes
     */
    fun getSerialNumberBytes(): ByteArray = byteArrayOf(0x05, 0xBF.toByte(), 0x1C, 0x00, 0x00, 0x00)
}