package app.pivo.android.ndisdk.util

import android.media.Image
import android.media.ImageReader
import android.renderscript.*
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.nio.FloatBuffer


/**
 *  Created by Tom Kim.
 *  Copyright (c) 2022 3i Inc. All rights reserved.
 */
object NDIConvertUtil {

    /**
     * Convert from singed 16-bit PCM to 32-bit short PCM.
     * @param array short array from audio read
     * @return byte array that converted float byte array
     */
    fun convert16BitTo32Bit(array: ShortArray): ByteArray? {
        val audioDataF = shortToFloat(array)
        for (i in array.indices) {
            val fValue = array[i]
            val divided = fValue / 32768.0
            audioDataF[i] = divided.toFloat()
        }

        val fb = FloatBuffer.wrap(audioDataF)
        val byteBuffer = ByteBuffer.allocate(fb.capacity() * Float.SIZE_BYTES)
        byteBuffer.order(ByteOrder.LITTLE_ENDIAN)
        byteBuffer.asFloatBuffer().put(fb)
        return byteBuffer.array()
    }

    /**
     * Convert from singed 16-bit PCM to 32-bit float PCM.
     * @param array short array from audio read
     * @return byte array that converted float byte array
     */
    fun convert16BitTo32Bit(array: FloatArray): ByteArray? {
//        val audioDataF = shortToFloat(array)
        for (i in array.indices) {
            val fValue = array[i]
            val divided = fValue / 32768.0
            array[i] = divided.toFloat()
        }

        val fb = FloatBuffer.wrap(array)
        val byteBuffer = ByteBuffer.allocate(fb.capacity() * Float.SIZE_BYTES)
        byteBuffer.order(ByteOrder.LITTLE_ENDIAN)
        byteBuffer.asFloatBuffer().put(fb)
        return byteBuffer.array()
    }

    /**
     * clone buffer
     * @param original original ByteBuffer
     * @return cloned ByteBuffer
     */
    private fun clone(original: ByteBuffer): ByteBuffer {
        val clone = ByteBuffer.allocate(original.capacity())
        original.rewind() //copy from the beginning
        clone.put(original)
        original.rewind()
        clone.flip()
        return clone
    }

    /**
     * Convert from array of short to array of float.
     * @param shortArray short array
     * @return float array
     */
    private fun shortToFloat(shortArray: ShortArray): FloatArray {
        val floatOut = FloatArray(shortArray.size)
        for (i in shortArray.indices) {
            floatOut[i] = shortArray[i].toFloat()
        }
        return floatOut
    }

    /**
     * convert YUV_420_888 image to RGB as bytebuffer.
     * @param rs RenderScript that receives from view
     * @param image current video frame image. usually receives from onImageAvailable
     * @see ImageReader
     */
    @Suppress("DEPRECATION")
    fun convertYUVtoRGB(rs: RenderScript, image: Image): ByteBuffer {
        val W: Int = image.width
        val H: Int = image.height
        val Y: Image.Plane = image.planes[0]
        val U: Image.Plane = image.planes[1]
        val V: Image.Plane = image.planes[2]
        val Yb: Int = Y.buffer.remaining()
        val Ub: Int = U.buffer.remaining()
        val Vb: Int = V.buffer.remaining()
        val data = ByteArray(Yb + Ub + Vb)
        clone(Y.buffer).get(data, 0, Yb)
        clone(V.buffer).get(data, Yb, Vb)
        clone(U.buffer).get(data, Yb + Vb, Ub)

        // All the image is inside data byte array
        // NOTICE: including PixelStrides !

        // initialization:
        val yuvToRgbIntrinsic: ScriptIntrinsicYuvToRGB = ScriptIntrinsicYuvToRGB.create(rs, Element.U8_4(rs))
        val yuvType: Type.Builder = Type.Builder(rs, Element.U8(rs)).setX(data.size)
        val `in` = Allocation.createTyped(rs, yuvType.create(), Allocation.USAGE_SCRIPT)
        val rgbaType: Type.Builder = Type.Builder(rs, Element.RGBA_8888(rs)).setX(W).setY(H)
        val out = Allocation.createTyped(rs, rgbaType.create(), Allocation.USAGE_SCRIPT)
        `in`.copyFrom(data)
        yuvToRgbIntrinsic.setInput(`in`) // Set the input yuv allocation, must be Element#U8.
        yuvToRgbIntrinsic.forEach(out) // Convert the image to RGB.(to 'out' Allocation)

        val result = out.byteBuffer
        `in`.destroy()
        out.destroy()
        yuvToRgbIntrinsic.destroy()

        return result
    }

}