package app.pivo.android.ndisdk

import android.content.Context
import android.graphics.*
import android.media.AudioFormat
import android.media.AudioRecord
import android.media.Image
import android.media.ImageReader
import android.net.nsd.NsdManager
import android.net.nsd.NsdServiceInfo
import android.renderscript.*
import android.util.Log
import android.util.Size
import android.util.SizeF
import androidx.core.content.ContextCompat
import androidx.core.graphics.drawable.toBitmap
import app.pivo.android.ndisdk.api.NDIAudioFrame
import app.pivo.android.ndisdk.api.NDIFrameFourCCType
import app.pivo.android.ndisdk.api.NDISender
import app.pivo.android.ndisdk.api.NDIVideoFrame
import app.pivo.android.ndisdk.model.Watermark
import app.pivo.android.ndisdk.util.NDIAudioRecord
import app.pivo.android.ndisdk.util.NDIConvertUtil.convertYUVtoRGB
import java.nio.ByteBuffer
import java.nio.ByteOrder
import java.nio.FloatBuffer
import java.util.*


/**
 *  Created by Tom Kim.
 *  Copyright (c) 2022 3i Inc. All rights reserved.
 */

class NDIService constructor(
    private var serviceName:String,
    private var clippingAudio:Boolean,
    private var previewSize:SizeF,
    private val context:Context
) {

    private var sender: NDISender? = null
    private var nsdManager: NsdManager? = null
    private var isNsdRegistered = false
    private var isSending = false
    private var audio: NDIAudioRecord? = null
    private lateinit var registrationListener:NsdManager.RegistrationListener
    private val renderScript:RenderScript

    private var ndiInitListener: (() -> Unit)? = null
    private var ndiDestroyedListener: (() -> Unit)? = null

    private var audioThread:Thread? = null

    data class Builder(
        private var serviceName:String = "NDI_ANDROID_${UUID.randomUUID()}",
        private var previewSize: SizeF = SizeF(0f,0f),
        private var clippingAudio:Boolean = false,
    ) {
        fun serviceName(serviceName:String) = apply { this.serviceName = serviceName}
        fun clippingAudio(clippingAudio:Boolean) = apply { this.clippingAudio = true}
        fun previewSize(size:SizeF) = apply {this.previewSize = size}
        fun build(context:Context): NDIService {
            return NDIService(serviceName, clippingAudio, previewSize, context)
        }
    }

    companion object {
        private const val FRAME_RATE_N = 24000
        private const val FRAME_RATE_D = 1001
        private const val NDI_NSD_PORT = 5960
        private const val NDI_SERVICE_TYPE = "_ndi._tcp"
        const val tag = "NDIWrapper"

        @JvmStatic
        var watermark:Watermark? = null

        @JvmStatic
        val videoFrame = NDIVideoFrame().apply {
            fourCCType = NDIFrameFourCCType.RGBA
            setFrameRate(FRAME_RATE_N, FRAME_RATE_D)
        }

        @JvmStatic
        val audioFrame = NDIAudioFrame().apply {
            sampleRate = NDIAudioRecord.NDI_AUDIO_SAMPLE_RATE
            channels =
                if (NDIAudioRecord.NDI_AUDIO_CHANNEL == AudioFormat.CHANNEL_IN_MONO) 1 else 2
        }
    }


    init {


        val serviceInfo = NsdServiceInfo().apply {
            serviceName = this@NDIService.serviceName
            port = NDI_NSD_PORT
            serviceType = NDI_SERVICE_TYPE
        }

        renderScript = RenderScript.create(context)

        registerNsdListener()

        nsdManager = (context.getSystemService(Context.NSD_SERVICE) as NsdManager).apply {
            registerService(serviceInfo, NsdManager.PROTOCOL_DNS_SD, registrationListener)
        }
    }

    /**
     * listener for NSD service.
     */
    private fun registerNsdListener() {
        registrationListener = object : NsdManager.RegistrationListener {
            override fun onServiceRegistered(serviceInfo: NsdServiceInfo) {
                serviceName = serviceInfo.serviceName
                isNsdRegistered = true
                try {
                    audio = NDIAudioRecord(context).also {
                        it.startRecording()
                        audioThread = Thread(AudioThread(it))
                        audioThread?.start()
                    }

                    isSending = true
                }
                catch (e:Exception) {
                    when(e) {
                        is SecurityException -> {
                            throw NSDRegistrationFailedException("permission error. Manifest.permission.RECORD_AUDIO is required.")
                        }
                        else -> throw e
                    }
                }

                ndiInitListener?.invoke()
                Log.i(tag, "onServiceRegistered ${serviceInfo}")
            }

            override fun onRegistrationFailed(serviceInfo: NsdServiceInfo, errorCode: Int) {
                isNsdRegistered = false
                throw NSDRegistrationFailedException("##onRegistrationFailed. error code : $errorCode")
            }

            override fun onServiceUnregistered(serviceInfo: NsdServiceInfo) {
                isNsdRegistered = false
                Log.i(tag, "onServiceUnregistered ${serviceInfo}")
            }

            override fun onUnregistrationFailed(serviceInfo: NsdServiceInfo, errorCode: Int) {
                throw NSDUnregistrationFailedException("onUnregistrationFailed. error code : $errorCode")
            }
        }
    }
    private fun setSender() {
        sender = NDISender(serviceName, null, false, clippingAudio)
    }

    /**
     * Set videoFrame's resolution
     * @param size size of the frame
     */
    fun setResolution(size:Size) {
        videoFrame.setResolution(size.width, size.height)
    }

    /**
     * Set watermark
     * @param drawableId resource id
     * @param point position from preview screen
     * @return true when successfully initialized. otherwize return false.
     */

    fun setFrameWatermark(drawableId:Int, point: PointF):Boolean {
        if(drawableId != 0) {
            ContextCompat.getDrawable(context, drawableId)?.toBitmap()?.let {
                watermark = Watermark(
                    it,
                    point)
            } ?: run {
                return false
            }
        }
        return true
    }

    /**
     * Set watermark
     * @param bitmap bitmap
     * @param point position from preview screen
     */

    fun setFrameWatermark(bitmap:Bitmap, point: PointF):Boolean {
        watermark = Watermark(
            bitmap,
            point)

        return true
    }

    /**
     * send video frame to NDI. use this when application uses camera2 api & YUV_420_888
     * @param image image from video frame (must be YUV_420_888 formatted
     * @param channelCount color channel count. Typically it is 4
     * @param width frame width
     * @param height frame height
     *
     * @see ImageFormat
     * @see ImageReader
     */
    fun sendYUVVideoFrame(image:Image, channelCount: Int = 4, width:Int, height:Int) {
        if(nsdManager == null) {
            throw NSDRegistrationFailedException("NSD is not registered.")
        }
        if(sender == null) {
            setSender()
        }
        if(!isSending) return

        val convertedArray = convertYUVtoRGB(renderScript, image)
        sendVideoFrame(convertedArray, channelCount, width, height)
    }

    /**
     * send video frame to NDI. use this when application uses cameraX api & RGB buffer from FrameAnalyzer
     * @param buffer RGB byteBuffer array
     * @param channelCount color channel count. Typically it is 4
     * @param width frame width
     * @param height frame height
     *
     * @see ImageFormat
     * @see FrameAnalyzer
     */
    fun sendRGBVideoFrame(buffer:ByteBuffer, channelCount: Int, width:Int, height:Int) {
        if(nsdManager == null) {
            throw NSDRegistrationFailedException("NSD is not registered.")
        }
        if(sender == null) {
            setSender()
        }
        if(!isSending) return

        sendVideoFrame(buffer, channelCount, width, height)
    }

    /**
     * send video frame to NDI. supports RGBA only for now, will be updated with multiple imageformats
     * @param byteBuffer RGBA or RGBX byteBuffer array
     * @param channelCount color channel count. Typically it is 4
     * @param width frame width
     * @param height frame height
     *
     * @see ImageFormat
     * @see NDIFrameFourCCType
     */

    private fun sendVideoFrame(byteBuffer:ByteBuffer, channelCount: Int = 4, width:Int, height:Int) {

        if(videoFrame.xResolution != width || videoFrame.yResolution != height) {
            videoFrame.setResolution(width, height)
        }

        val watermarkedBuffer = addWaterMark(byteBuffer, width, height)
        videoFrame.data = watermarkedBuffer
        sender!!.sendVideoFrame(videoFrame)
        watermarkedBuffer.clear()
    }

    /**
     * Add watermark for NDI frame.
     * @param buffer buffer from camera frame
     * @param width frame width
     * @param height frame height
     */

    private fun addWaterMark(buffer:ByteBuffer, width:Int, height:Int):ByteBuffer {

        return watermark?.let { watermark ->
            if(previewSize.width <= 0f || previewSize.height <= 0f) { return buffer }
            val frame = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888).apply {
                copyPixelsFromBuffer(buffer)
            }

            buffer.clear()

            val resBuffer = ByteBuffer.allocateDirect(frame.rowBytes * frame.height)

            val frameRatio = SizeF(width / previewSize.width, height / previewSize.height)
            val ratio = if(frameRatio.width < frameRatio.height) frameRatio.width else frameRatio.height
            val wmSpacePoint = PointF(watermark.point.x * ratio, watermark.point.y * ratio)

            val wmMatrix = Matrix().apply {
                postScale((watermark.bitmap.width * ratio) / watermark.bitmap.width, (watermark.bitmap.height * ratio) / watermark.bitmap.height)
                postTranslate(wmSpacePoint.x, wmSpacePoint.y)
            }

            val result = Bitmap.createBitmap(frame.width, frame.height, frame.config)

            val canvas = Canvas(result)
            canvas.drawBitmap(frame, 0f, 0f, null)
            canvas.drawBitmap(watermark.bitmap, wmMatrix, null)
            result.copyPixelsToBuffer(resBuffer)
            resBuffer
        } ?: run {
            buffer
        }
    }
    /**
     * resume NDI service
     */
    fun resume() {
        isSending = true
        audio?.let {
            it.startRecording()
            audioThread = Thread(AudioThread(it))
            audioThread?.start()
        }
    }
    /**
     * pause NDI service
     */
    fun pause() {
        isSending = false
        audio?.stop()
    }

    /**
     * destroy NDI service
     */
    fun destroy() {
        try {
            nsdManager?.unregisterService(registrationListener)
        }
        catch (e:IllegalArgumentException) {
            Log.i(tag, "nsdManager service is not registered. ")
        }

        watermark?.bitmap?.recycle()
        watermark = null
        audio?.stop()

        videoFrame.close()
        audioFrame.close()

        nsdManager = null

        isSending = false

        sender?.close()
        sender = null

        renderScript.destroy()

        isNsdRegistered = false
        ndiDestroyedListener?.invoke()
    }

    /**
     * check nsdServiceRegistered
     * @return boolean
     */
    fun isNsdRegistered() = isNsdRegistered

    /**
     * listener on NDI init success
     */
    fun setOnNDIInitialized(listener: () -> Unit) {
        ndiInitListener = listener
    }

    /**
     * listener on NDI destroyed
     */
    fun setOnNDIDestroyed(listener: () -> Unit) {
        ndiDestroyedListener = listener
    }

    /**
     * Thread for AudioRecording. send audio frame to NDI.
     * @param audio audioRecord object
     * @see NDIAudioRecord
     */
    inner class AudioThread(private val audio: NDIAudioRecord):Runnable {
        override fun run() {
            if (nsdManager == null) {
                throw NSDRegistrationFailedException("NSD is not registered.")
            }
            if (sender == null) {
                setSender()
            }

            while (audio.recordingState == AudioRecord.RECORDSTATE_RECORDING) {
                val audioBuffer = FloatArray(audio.bufferSizeInFrames)
                val readSize = audio.read(audioBuffer, 0, audioBuffer.size, AudioRecord.READ_BLOCKING)

                if (readSize > -1) {
                    val fb = FloatBuffer.wrap(audioBuffer)
                    val byteBuffer = ByteBuffer.allocateDirect(fb.capacity() * Float.SIZE_BYTES)
                    byteBuffer.order(ByteOrder.LITTLE_ENDIAN)
                    byteBuffer.asFloatBuffer().put(fb)

                    audioFrame.apply {
                        samples = readSize
                        channelStride = readSize
                        data = byteBuffer
                    }

                    sender?.sendAudioFrame(audioFrame)
//                    audioFrame.close()
                }
            }
        }
    }
}

class NoResolutionDefinedException(msg:String?) : Exception(msg)
class NoSenderInitializedException(msg:String?) : Exception(msg)
class NSDRegistrationFailedException(msg:String?) : Exception(msg)
class NSDUnregistrationFailedException(msg:String?) : Exception(msg)