package app.pivo.android.plussdk;

import android.graphics.PointF;

class PivoSpeedController {

    private static PivoSpeedController instance;

    private final PivoRotatorConfig mPivoRotatorConfig = PivoRotatorConfig.getInstance();

    private PivoPlusSdk plusSdk;


    private int mWidth;
    private int mHeight;
    private int mOrientation;
    private double mZoom = 1.f;

    private boolean isFront = true;
    private boolean isOrientationLocked;

    private PositionType positionType = PositionType.MIDDLE;

    private SensitivityMode mode = SensitivityMode.NORMAL;


    private PointF lastPoint = new PointF(0,0);

    public static PivoSpeedController getInstance (){
        if (instance ==null){
            instance = new PivoSpeedController();
            return instance;
        }
        return instance;
    }

    public PivoSpeedController setController(PivoPlusSdk sdk) {
        plusSdk = sdk;
        return this;
    }

    public PivoSpeedController setPositionType(PositionType positionType) {
        this.positionType = positionType;
        return this;
    }

    public PivoSpeedController setPreviewSize(int width, int height) {
        mWidth = width;
        mHeight = height;
        return this;
    }

    public PivoSpeedController setCameraFacing(boolean isFront) {
        this.isFront = isFront;
        return this;
    }

    public PivoSpeedController setOrientation(int orientation) {
        mOrientation = orientation;
        return this;
    }

    public PivoSpeedController setZoomFactor(double zoom) {
        mZoom = zoom;
        return this;
    }

    public PivoSpeedController setOrientationLocked(boolean locked) {
        isOrientationLocked = locked;
        return this;
    }

    public PivoSpeedController setSensitivityMode(SensitivityMode sensitivityMode) {
        mode = sensitivityMode;
        return this;
    }

    public void track(PointF currentPoint) {
        if(mode.equals(SensitivityMode.NONE)){
            plusSdk.stop();
            return;
        }
        mPivoRotatorConfig.initConfig(mZoom , mode, plusSdk.getVersion()); // initialize with zoom and Sensitivity
        PointF diff = new PointF(getDesiredPoint().x - currentPoint.x, getDesiredPoint().y - currentPoint.y);


        if (lastPoint.equals(new PointF(0, 0))) {
            lastPoint = currentPoint;
        }
        if (isOrientationLocked){
            if ((mOrientation % 2 == 1)){
                handleTracking(currentPoint,mWidth,diff.x, true);
            }else {
                handleTracking(currentPoint,mHeight,diff.y,false);
            }
        }else{
            if ((mOrientation % 2 == 1)){
                handleTracking(currentPoint,mWidth,diff.x, true);
            }else {
                handleTracking(currentPoint,mHeight,diff.x,true);
            }
        }
    }

    private PointF getDesiredPoint() {

        if (isOrientationLocked) {
            if (mOrientation % 2 == 1) {
                if (positionType == PositionType.LEFT) {
                    return new PointF(mWidth * 3 / 4.f, 0);
                } else if (positionType == PositionType.RIGHT) {
                    return new PointF(mWidth / 4.f, 0);
                } else {
                    return new PointF(mWidth / 2.f, 0);
                }
            } else {
                if (positionType == PositionType.LEFT) {
                    return new PointF(0, mHeight * 3 / 4.f);
                } else if (positionType == PositionType.RIGHT) {
                    return new PointF(0, mHeight / 4.f);
                } else {
                    return new PointF(0, mHeight / 2.f);
                }
            }
        }else{ // Orientation not locked
            if (mOrientation % 2 == 1) {
                if (positionType == PositionType.LEFT) {
                    return new PointF(mWidth * 3 / 4.f, 0);
                } else if (positionType == PositionType.RIGHT) {
                    return new PointF(mWidth / 4.f, 0);
                } else {
                    return new PointF(mWidth / 2.f, 0);
                }
            } else {
                if (positionType == PositionType.LEFT) {
                    return new PointF(mHeight * 3 / 4.f, 0);
                } else if (positionType == PositionType.RIGHT) {
                    return new PointF(mHeight / 4.f, 0);
                } else {
                    return new PointF(mHeight / 2.f, 0);
                }
            }
        }
    }


    private void handleTracking(PointF currentPoint, int dimension, float coordinate, boolean isPortrait){
        final float scaleFactor = (dimension / 2.f) / mPivoRotatorConfig.scaleFactor;
        final float pdPixel;
        if (isPortrait){
            pdPixel = Math.abs( mPivoRotatorConfig.getKP()* (currentPoint.x - getDesiredPoint().x)
                    + mPivoRotatorConfig.getKD() * (currentPoint.x - lastPoint.x));
        }else{
            pdPixel = Math.abs( mPivoRotatorConfig.getKP()* (currentPoint.y - getDesiredPoint().y)
                    + mPivoRotatorConfig.getKD() * (currentPoint.y - lastPoint.y));
        }

        final float pd_angle = pdPixel / scaleFactor;

        float PD_ANGLE_THRESHOLD = 2.5f;
        if (pd_angle< PD_ANGLE_THRESHOLD || Math.abs(coordinate)<= mPivoRotatorConfig.MARGIN * dimension){
            plusSdk.stop();
            return;
        }

        int s;
        if (plusSdk.getVersion().getVersion()==0){
            s = SpeedConstraint.getInstance().constraint(pd_angle).getValue();
        }else {
            s = 2 * SpeedConstraint.getInstance().constraintAngle(pd_angle);
        }

        lastPoint = currentPoint;
        if (coordinate < 0) {
            if (isFront) {
                plusSdk.turnLeftContinuously(s);
            } else {
                plusSdk.turnRightContinuously(s);
            }
        } else {
            if (isFront) {
                plusSdk.turnRightContinuously(s);
            } else {
                plusSdk.turnLeftContinuously(s);
            }
        }
    }
}
