package app.pivo.android.plussdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.ImageFormat;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.RectF;
import android.media.Image;
import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;
import android.util.Size;

import androidx.annotation.NonNull;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import app.pivo.android.plussdk.tracking.FrameMetadata;
import app.pivo.android.plussdk.tracking.GraphicOverlay;
import app.pivo.android.plussdk.tracking.Tracking;
import app.pivo.android.plussdk.util.ITrackingListener;
import app.pivo.android.targetdetectionsdk.detectionresult.FailureCallback;
import app.pivo.android.targetdetectionsdk.detectionresult.SingleTargetCallback;
import app.pivo.android.targetdetectionsdk.frame.FrameContainer;
import app.pivo.android.targetdetectionsdk.frame.NV21ByteArrayContainer;
import app.pivo.android.targetdetectionsdk.processors.DetectionManager;
import app.pivo.android.targetdetectionsdk.processors.tflite.detector.TFDetectorOld;
import app.pivo.android.targetdetectionsdk.target.DetectionTarget;
import app.pivo.android.targetdetectionsdk.target.TargetType;

class FrameProcessingTask implements IFrameProcessingTask {
    private final String TAG = this.getClass().getSimpleName();

    private FrameProcessingTask(){
    }

    private static FrameProcessingTask instance;

    static class Builder{
        FrameProcessingTask instance = new FrameProcessingTask();

        Builder setContext(Context context){
            instance.mContext = context;
            return this;
        }

        FrameProcessingTask build(){
            return instance;
        }
    }

    public static FrameProcessingTask getInstance(){
        if (instance == null){
            instance = new FrameProcessingTask();
        }
        return instance;
    }

    public FrameProcessingTask setContext(Context context){
        this.mContext = context;
        return this;
    }

    private void setWidth(int width){
        this.width = width;
    }

    private void setHeight(int height){
        this.height = height;
    }

    private void setRotation(int rotation){
        this.rotation = rotation;
    }

    private void setFrontCamera(boolean frontCamera){
        this.frontCamera = frontCamera;
    }

    private FrameProcessingTask setOrientationLocked(boolean locked){
        this.orientationLocked = locked;
        return this;
    }

    public FrameProcessingTask setTrackingLayout(GraphicOverlay trackingLayout){
        this.mTrackingLayout = trackingLayout;
        return this;
    }

    private void updateFrameMetaData(FrameMetadata frameMetadata){
        setWidth(frameMetadata.getWidth());
        setHeight(frameMetadata.getHeight());
        setRotation(frameMetadata.getRotation());
        setFrontCamera(frameMetadata.isFrontCamera());
        setOrientationLocked(frameMetadata.isOrientationLocked());
    }

    public synchronized void startPersonTracking(FrameMetadata frameMetadata, Image image, ITrackingListener listener){
        //stop tracking beforstarting
        stopTracking();

        this.tracking = Tracking.PERSON;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());

        //start person tracking processing thread
        handlerThread = new HandlerThread("inference");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        updateFrameMetaData(frameMetadata);

        //init action tracking processor
        initActionProcessor(listener);
        //init person tracking processor
        initProcessor(TargetType.BODY);
        //initPersonProcessor();

        image.close();
    }

    @Override
    public synchronized void startFaceTracking(FrameMetadata frameMetadata, Image image, ITrackingListener listener){
        //stop tracking before starting
        stopTracking();

        this.tracking = Tracking.FACE;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        //start person tracking processing thread
        handlerThread = new HandlerThread("inference");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        updateFrameMetaData(frameMetadata);

        //init action tracking processor
        initActionProcessor(listener);
        //init person tracking processor
        initProcessor(TargetType.FACE);

        image.close();
    }
    @Override
    public synchronized void startDogTracking(FrameMetadata frameMetadata, Image image, ITrackingListener listener){
        //stop tracking before starting
        stopTracking();

        this.tracking = Tracking.DOG;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        //start person tracking processing thread
        handlerThread = new HandlerThread("inference");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        updateFrameMetaData(frameMetadata);

        //init action tracking processor
        initActionProcessor(listener);
        //init person tracking processor
        initProcessor(TargetType.DOG);

        image.close();
    }
    @Override
    public synchronized void startHorseTracking(FrameMetadata frameMetadata, Image image, ITrackingListener listener){
        //stop tracking before starting
        stopTracking();

        this.tracking = Tracking.HORSE;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        //start person tracking processing thread
        handlerThread = new HandlerThread("inference");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        updateFrameMetaData(frameMetadata);

        //init action tracking processor
        initActionProcessor(listener);
        //init person tracking processor
        initProcessor(TargetType.HORSE);

        image.close();
    }

    @Override
    public void startActionTracking(FrameMetadata frameMetadata,  Rect region, Image image, ITrackingListener listener){
        //first stop tracking before starting
        stopTracking();
        this.tracking = Tracking.ACTION;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        this.objROI = region;

        this.isObjSelected = true;
        this.isInitialized = false;

        //start tracking processing thread
        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        //init action tracking processor
        initActionProcessor(listener);

        image.close();
    }

    @Override
    public void startActionTracking(FrameMetadata frameMetadata, Rect region, byte[] image, ITrackingListener listener) {
        //first stop tracking before starting
        stopTracking();
        this.tracking = Tracking.ACTION;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        this.objROI = region;

        this.isObjSelected = true;
        this.isInitialized = false;

        //start tracking processing thread
        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        //init action tracking processor
        initActionProcessor(listener);
    }

    @Override
    public synchronized void startPersonTracking(FrameMetadata frameMetadata, byte[] image, ITrackingListener listener) {
        //stop tracking before starting
        stopTracking();

        this.tracking = Tracking.PERSON;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        //start person tracking processing thread
        handlerThread = new HandlerThread("inference");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        updateFrameMetaData(frameMetadata);

        //init action tracking processor
        initActionProcessor(listener);
        //init person tracking processor
        initProcessor(TargetType.BODY);
    }


    @Override
    public synchronized void startFaceTracking(FrameMetadata frameMetadata, byte[] image, ITrackingListener listener) {
        //stop tracking before starting
        stopTracking();

        this.tracking = Tracking.FACE;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        //start person tracking processing thread
        handlerThread = new HandlerThread("inference");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        updateFrameMetaData(frameMetadata);

        //init action tracking processor
        initActionProcessor(listener);
        //init person tracking processor
        //initFaceProcessor();
        initProcessor(TargetType.FACE);
    }

    @Override
    public synchronized void startDogTracking(FrameMetadata frameMetadata, byte[] image, ITrackingListener listener) {
        //stop tracking before starting
        stopTracking();

        this.tracking = Tracking.DOG;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        //start person tracking processing thread
        handlerThread = new HandlerThread("inference");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        updateFrameMetaData(frameMetadata);

        //init action tracking processor
        initActionProcessor(listener);
        //init person tracking processor
        initProcessor(TargetType.DOG);
    }

    @Override
    public void startHorseTracking(FrameMetadata frameMetadata, byte[] image, ITrackingListener listener) {
        //stop tracking before starting
        stopTracking();

        this.tracking = Tracking.HORSE;
        this.frameMetadata = frameMetadata;
        this.frameSize = new Size(frameMetadata.getWidth(), frameMetadata.getHeight());
        //start person tracking processing thread
        handlerThread = new HandlerThread("inference");
        handlerThread.start();
        handler = new Handler(handlerThread.getLooper());

        processingRunnable = new FrameProcessingRunnable();
        processingThread = new Thread(processingRunnable);
        processingRunnable.setActive(true);
        processingThread.start();

        updateFrameMetaData(frameMetadata);

        //init action tracking processor
        initActionProcessor(listener);
        //init horse tracking processor
        initProcessor(TargetType.BODY);
    }

    public void update(Image image, FrameMetadata metadata){

        //Log.d("FrameProcessingTask", "update -> " + isProcessingFrame);
        if (tracking == Tracking.PERSON || tracking == Tracking.HORSE || tracking == Tracking.FACE || tracking == Tracking.DOG) {

            updateFrameMetaData(metadata);
            frameMetadata = metadata;
            if (rgbBytes == null){
                rgbBytes = new int[width*height];
            }

            if (isProcessingFrame){
                image.close();
                return;
            }

            if (mPersonTracker!=null){
                mPersonTracker.setLayout(metadata.getWidth(), metadata.getHeight());
            }
            isProcessingFrame = true;
            final Image.Plane[] planes = image.getPlanes();
            fillBytes(planes, yuvBytes);
            final int yRowStride = planes[0].getRowStride();
            final int uvRowStride = planes[1].getRowStride();
            final int uvPixelStride = planes[1].getPixelStride();

            imageConverterRunnable =
                    () -> ImageUtils.convertYUV420ToARGB8888(
                            yuvBytes[0],
                            yuvBytes[1],
                            yuvBytes[2],
                            width,
                            height,
                            yRowStride,
                            uvRowStride,
                            uvPixelStride,
                            rgbBytes);

            n21ConverterRunnable = () -> nv21Data = ImageUtils.convertYUV420888ToNV21(yuvBytes[0],yuvBytes[2]);
            postInferenceRunnable = () -> isProcessingFrame = false;
        }

        if (tracking != Tracking.NONE){
            if (processingRunnable!=null){
                if (tracking == Tracking.PERSON || tracking == Tracking.HORSE || tracking == Tracking.FACE || tracking == Tracking.DOG){
                    processingRunnable.setNextFrame(new byte[0]);
                }else{
                    processingRunnable.setNextFrame(convertYUV420888ToNV21(image));
                }
            }
        }

        image.close();
    }

    @Override
    public void update(byte[] image, FrameMetadata metadata) {
//        if (tracking == Tracking.PERSON || tracking == Tracking.HORSE){
//
//            updateFrameMetaData(metadata);
//            frameMetadata = metadata;
//            if (rgbBytes == null){
//                rgbBytes = new int[width*height];
//            }
//
//            if (isProcessingFrame){
//                return;
//            }
//
//            if (mPersonTracker!=null){
//                mPersonTracker.setLayout(metadata.getWidth(), metadata.getHeight());
//            }
//            isProcessingFrame = true;
//
//            imageConverterRunnable =
//                    () -> ImageUtils.convertYUV420SPToARGB8888(image, width, height, rgbBytes);
//
//            n21ConverterRunnable = () -> nv21Data = image;
//            postInferenceRunnable = () -> isProcessingFrame = false;
//        }

        if (tracking != Tracking.NONE){
            if (processingRunnable!=null){
                processingRunnable.setNextFrame(image);
            }
        }
    }

    /**
     * Stops the camera and releases the resources of the camera and underlying detector.
     */
    public void stopTracking() {
        synchronized (processorLock) {
            //stop thread before release
            stopThread();

            // reset action tracking
            resetActionTracking();
        }
    }

    public Boolean isTrackingOn() {
        return (tracking != Tracking.NONE);
    }

    @Override
    public void setTargetToTrack(Rect rect) {
        if (mPersonTracker!=null){
            mPersonTracker.setTargetToTrack(rect);
        }
    }

    private Context mContext;
    private Thread processingThread;
    private FrameProcessingRunnable processingRunnable;
    private final Object processorLock = new Object();

    //action tracking
    private VisionImageProcessor actionProcessor;
    private ActionTracker actionTracker;
    private final Object actionTrackingLock = new Object();

    // person tracking
    private Handler handler;
    private HandlerThread handlerThread;
    private MultiBoxTracker mPersonTracker;
    private Classifier classifier;
    private Classifier.Recognition target;
    private final Object aiTrackingLock = new Object();

    private final float MINIMUM_CONFIDENCE_TF_OD_API = 0.6f;
    private int cropSize = 0;
    // Configuration values for the prepackaged SSD model.
    private static final int TF_OD_API_INPUT_SIZE = 300;
    private static final boolean TF_OD_API_IS_QUANTIZED = true;
    private static final String TF_OD_API_MODEL_FILE = "detect.tflite";
    private static final String TF_OD_API_LABELS_FILE = "file:///android_asset/labelmap.txt";

    private Runnable postInferenceRunnable;
    private Runnable imageConverterRunnable;
    private Runnable n21ConverterRunnable;

    // if skipOneFrame is set true, it skips
    // one frame and gets the next one
    private boolean isProcessingFrame = false;
    private byte[][] yuvBytes = new byte[3][];

    private byte[] nv21Data = null;
    private int[] rgbBytes = null;

    private boolean computingDetection = false;
    private Bitmap croppedBitmap = null;
    private Matrix cropToFrameTransform;

    private boolean chosen = false;

    // Tracking required fields
    private FrameMetadata frameMetadata;
    private int width;
    private int height;
    private int rotation;
    private boolean frontCamera;
    private Tracking tracking = Tracking.NONE;
    private boolean orientationLocked;
    private Size frameSize = new Size(1280, 720);


    private Rect objROI;
    private boolean isObjSelected = false;
    private boolean isInitialized = false;

    private GraphicOverlay mTrackingLayout;


    private TFDetectorOld.Callback callbackListener = new TFDetectorOld.Callback() {
        @Override
        public void onTFDetection(@NonNull DetectionTarget target) {
            Log.d("Frame", "detected! -> " + target);
        }

        @Override
        public void onTFDetectionFailed() {
            Log.e("Frame", "detection failed!");
        }
    };
    private void resetActionTracking(){
        objROI = null;
        isObjSelected = false;
        isInitialized = false;
        cleanScreen();
    }

    // clean the screen
    private void cleanScreen(){
        if (mTrackingLayout!=null)mTrackingLayout.clear();
    }

    // initialize action tracking
    private void initActionProcessor(ITrackingListener actionTrackerListener){
        if (actionTrackerListener!=null){
            synchronized (actionTrackingLock){
                cleanScreen();
                actionTracker = new ActionTracker(actionTrackerListener);
                if (actionProcessor!=null){
                    actionProcessor.stop();
                }
                isInitialized = false;
                actionProcessor = new ActionTrackingProcessor(actionTracker, !frameMetadata.isOrientationLocked()?1:0);
            }
        }
    }

    // initialize person processor
    private void initPersonProcessor(){
        mPersonTracker = new MultiBoxTracker(mContext, actionTracker);
        mPersonTracker.setFrontCamera(frontCamera);

        try{
            classifier =
                    TFLiteObjectDetectionAPIModel.create(
                            mContext.getAssets(),
                            TF_OD_API_MODEL_FILE,
                            TF_OD_API_LABELS_FILE,
                            TF_OD_API_INPUT_SIZE,
                            TF_OD_API_IS_QUANTIZED);

            cropSize = TF_OD_API_INPUT_SIZE;
        }catch (IOException e){
            e.printStackTrace();
        }
    }

    private DetectionManager detectionManager = null;
    private TargetType targetType = TargetType.NONE;

    FailureCallback failureCallback = () -> {
        Log.i("DetectFrame", "failureCallback");
        computingDetection = false;
    };

    SingleTargetCallback singleTargetCallback = new SingleTargetCallback() {
        @Override
        public void onSingleTargetSelected(@NonNull DetectionTarget target) {
            Log.i("DetectFrame", "SelectedTargetDetection");
            Classifier.Recognition recognition = new Classifier.Recognition(String.valueOf(target.getTargetId()), getLabel(), 1.0f, null);
            RectF resultRect;
            resultRect = target.getTargetBox().mapToScreen(rotation, frontCamera, frameMetadata.isOrientationLocked()).scaleUpTo(frameSize);
            recognition.setLocation(resultRect);
            List<Classifier.Recognition> recognitions = new ArrayList<>();
            recognitions.add(recognition);

            if (recognition != null) {
                mPersonTracker.trackResults(recognitions);
                recognition = mPersonTracker.handleDetection(recognitions, recognition);
            }

            if (recognition != null){
                mPersonTracker.track(nv21Data, width, height, recognition, rotation, frameMetadata.isOrientationLocked(), getLabel());
            }

            getTracker().draw();

            computingDetection = false;
        }
    };

    private void initProcessor(TargetType type){
        targetType = type;

        mPersonTracker = new MultiBoxTracker(mContext, actionTracker);
        mPersonTracker.setFrontCamera(frontCamera);

        if (detectionManager == null) {
            detectionManager = new DetectionManager(mContext);

            detectionManager.initialize();
            detectionManager.unregisterSingleTargetListener(singleTargetCallback);
            detectionManager.registerSingleTargetListener(singleTargetCallback);

            detectionManager.unregisterFailureListener(failureCallback);
            detectionManager.registerFailureListener(failureCallback);
        }
    }

    public MultiBoxTracker getTracker(){
        return mPersonTracker;
    }

    private void readyForNextImage() {
        if (postInferenceRunnable != null) {
            postInferenceRunnable.run();
        }
    }

    private int[] getRgbBytes() {
        imageConverterRunnable.run();
        return rgbBytes;
    }

    private byte[] getNV21(){
        n21ConverterRunnable.run();
        return nv21Data;
    }

    private void fillBytes(final Image.Plane[] planes, final byte[][] yuvBytes) {
        // Because of the variable row stride it's not possible to know in
        // advance the actual necessary dimensions of the yuv planes.
        for (int i = 0; i < planes.length; ++i) {
            final ByteBuffer buffer = planes[i].getBuffer();
            if (yuvBytes[i] == null) {
                yuvBytes[i] = new byte[buffer.capacity()];
            }
            buffer.get(yuvBytes[i]);
        }
    }

    private static byte[] convertYUV420888ToNV21(Image imgYUV420) {
        // Converting YUV_420_888 data to NV21.
        try {
            byte[] data;
            ByteBuffer buffer0 = imgYUV420.getPlanes()[0].getBuffer();
            ByteBuffer buffer2 = imgYUV420.getPlanes()[2].getBuffer();
            int buffer0_size = buffer0.remaining();
            int buffer2_size = buffer2.remaining();
            data = new byte[buffer0_size + buffer2_size];
            buffer0.get(data, 0, buffer0_size);
            buffer2.get(data, buffer0_size, buffer2_size);
            return data;
        }catch (IllegalStateException e){
            e.printStackTrace();
            return null;
        }
    }

    private float getConfidence(){
        if (tracking == Tracking.HORSE){
            return 0.39f;
        }
//        else if (tracking == Tracking.DOG){
//            return 0.39f;
//        }else if (tracking == Tracking.CAT){
//            return 0.39f;
//        }
        else {
            return MINIMUM_CONFIDENCE_TF_OD_API;
        }
    }

    private Classifier.Recognition getHighConfidence(List<Classifier.Recognition> outputs){
        Classifier.Recognition targetHighConfidence = null;
        float confidence = Float.MIN_VALUE;
        for (Classifier.Recognition result: outputs){
            if (result.getConfidence()>confidence){
                targetHighConfidence = result;
                confidence =  targetHighConfidence.getConfidence();
            }
        }
        return targetHighConfidence;
    }

    private synchronized void runInBackground(final Runnable r) {
        if (handler != null) {
            handler.post(r);
        }
    }

    private String getLabel(){
        if (tracking == Tracking.HORSE){
            return "horse";
        } else if (tracking == Tracking.DOG){
            return "dog";
        } else if (tracking == Tracking.PERSON){
            return "person";
        } else if (tracking == Tracking.FACE) {
            return "face";
        }
        else {
            return "person";
        }
    }

    private void processNV21Image() {
        int sensorOrientation;
        if (frameMetadata.isOrientationLocked()){
            sensorOrientation = 90;
        }else{
            sensorOrientation = rotation * 90;
        }
        mPersonTracker.setFrameConfiguration(width, height,  sensorOrientation);
        byte[] data = getNV21();

        readyForNextImage();

        runInBackground(
                () -> {
                    FrameContainer frameContainer;

                    frameContainer = new NV21ByteArrayContainer(
                        data,
                        new app.pivo.android.targetdetectionsdk.frame.FrameMetadata(new Size(frameMetadata.getWidth(), frameMetadata.getHeight()), sensorOrientation)
                    );
                    detectionManager.findAndFollowTargetIn(frameContainer, targetType);
                });
    }

    private void processImage() {
        Bitmap rgbFrameBitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        croppedBitmap = Bitmap.createBitmap(cropSize, cropSize, Bitmap.Config.ARGB_8888);

        Matrix frameToCropTransform = ImageUtils.getTransformationMatrix(
                width, height,
                cropSize, cropSize,
                rotation * 90, false);

        cropToFrameTransform = new Matrix();
        frameToCropTransform.invert(cropToFrameTransform);

        int sensorOrientation;
        if (frameMetadata.isOrientationLocked()){
            sensorOrientation = 90;
        }else{
            sensorOrientation = rotation * 90;
        }
        mPersonTracker.setFrameConfiguration(width, height,  sensorOrientation);
        byte[] data = getNV21();

        // No mutex needed as this method is not reentrant.
        if (computingDetection) {
            readyForNextImage();
            return;
        }

        computingDetection = true;

        rgbFrameBitmap.setPixels(getRgbBytes(), 0, width, 0, 0, width, height);

        readyForNextImage();

        final Canvas canvas = new Canvas(croppedBitmap); // Create Bitmap of cropped size
        canvas.drawBitmap(rgbFrameBitmap, frameToCropTransform, null); // Set pixel of the bitmap with RGBframe gotten using imageConverter runnable

        runInBackground(
                () -> {
                    List<Classifier.Recognition> results;

                    if (!mPersonTracker.isTrackingProcessing()) {
                        results = classifier.recognizeImage(croppedBitmap); // Detection

                        //Log.d(TAG, "detected result : " + results);
                        final List<Classifier.Recognition> mappedRecognitions = new LinkedList<>();

                        for (final Classifier.Recognition result : results) {
                            final RectF location = result.getLocation();

                            if (location != null && result.getConfidence() >= getConfidence() && result.getTitle().compareTo(getLabel()) == 0) {

                                cropToFrameTransform.mapRect(location);
                                Log.i(TAG, "location : " + location + " confidence : " + result.getConfidence());
                                result.setLocation(location);
                                mappedRecognitions.add(result);
                            }
                        }

                        if (!mappedRecognitions.isEmpty()) {
                            if (!chosen) {
                                target = getHighConfidence(mappedRecognitions);
                                chosen = true;
                            }
                        }

                        if (target != null || chosen) {
                            mPersonTracker.trackResults(mappedRecognitions);
                            target = mPersonTracker.handleDetection(mappedRecognitions, target);
                        }
                    }

                    if (target != null){
                        mPersonTracker.track(data, width, height, target, rotation,frameMetadata.isOrientationLocked(), getLabel());
                    }
                    else{
                        chosen = false;
                    }

                    getTracker().draw();

                    computingDetection = false;
                });
    }

    //stop thread after closing the camera
    private void stopThread(){
        this.tracking = Tracking.NONE;
        if (processingRunnable!=null){
            processingRunnable.setActive(false);
            // release processing runnable
            processingRunnable.release();
        }

        try {
            //stop person tracking processing thread
            if (handlerThread!=null){
                // Wait for the thread to complete to ensure that we can't have multiple threads
                // executing at the same time (i.e., which would happen if we called init too
                // quickly after stop).
                handlerThread.quitSafely();
                handlerThread.join();
                handlerThread = null;
                handler = null;
            }
        } catch (InterruptedException e) {
            Log.d(TAG, "Frame processing thread interrupted on release.");
        }
        processingThread = null;
    }

    //************************************ Inner class for processing frames *************************************
    private class FrameProcessingRunnable implements Runnable{

        // This lock guards all of the member variables below.
        private final Object lock = new Object();
        private boolean active = true;

        // These pending variables hold the state associated with the new frame awaiting processing.
        private byte[] pendingFrameData;
        private Image pendingImage = null;

        FrameProcessingRunnable() {}

        /**
         * Releases the underlying receiver. This is only safe to do after the associated thread has
         * completed, which is managed in camera source's release method above.
         */
        @SuppressLint("Assert")
        void release() {
            try{
                processingThread.join(1000);
            }catch (Exception e){
                e.printStackTrace();
            }
            assert (processingThread.getState() == Thread.State.TERMINATED);
        }

        /** Marks the runnable as active/not active. Signals any blocked threads to continue. */
        void setActive(boolean active) {
            synchronized (lock) {
                this.active = active;
                lock.notifyAll();
            }
        }

        /**
         * Sets the frame data received from the camera. This adds the previous unused frame buffer (if
         * present) back to the camera, and keeps a pending reference to the frame data for future use.
         */
        private void setNextFrame(byte[] data) {
            if (data == null)return;
            synchronized (lock) {
                if (pendingFrameData != null) {
                    pendingFrameData = null;
                }

                pendingFrameData =  data;
                // Notify the processor thread if it is waiting on the next frame (see below).
                lock.notifyAll();
            }
        }

        @SuppressLint("InlinedApi")
        @SuppressWarnings("GuardedBy")
        @Override
        public void run() {
            byte[] data;

            while (true) {
                synchronized (lock) {
                    while (active && pendingFrameData == null) {
                        try {
                            // Wait for the next frame to be received from the camera, since we
                            // don't have it yet.
                            lock.wait();
                        } catch (InterruptedException e) {
                            Log.e(TAG, "Frame processing loop terminated.", e);
                            return;
                        }
                    }

                    if (!active) {
                        // Exit the loop once this camera source is stopped or released.  We check
                        // this here, immediately after the wait() above, to handle the case where
                        // setActive(false) had been called, triggering the termination of this
                        // loop.
                        return;
                    }

                    // Hold onto the frame data locally, so that we can use this for detection
                    // below.  We need to clear pendingFrameData to ensure that this buffer isn't
                    // recycled back to the camera before we are done using that data.
                    data = pendingFrameData;
                    pendingFrameData = null;
                    //image = pendingImage;
                    //pendingImage = null;
                }

                // The code below needs to run outside of synchronization, because this will allow
                // the camera to add pending frame(s) while we are running detection on the current
                // frame.
                try {
                    if (tracking == Tracking.ACTION) {
                        Log.d(TAG, "tracking == Tracking.ACTION");
                        synchronized (actionTrackingLock){

                            if (isObjSelected && !isInitialized){

                                if (actionTracker !=null){
                                    actionTracker.initCamera(data,
                                            ImageFormat.YUV_420_888,
                                            frameMetadata.getWidth(),
                                            frameMetadata.getHeight(),
                                            objROI,
                                            frameMetadata.isFrontCamera()?1:0,
                                            frameMetadata.getRotation(),
                                            frameMetadata.isOrientationLocked()?0:1);
                                    isObjSelected = !isObjSelected;
                                    isInitialized = true;
                                }
                            }
                            if (isInitialized){
                                Log.d(TAG, "initialized");
                                actionProcessor.process(data, frameMetadata, mTrackingLayout);
                            }
                        }
                    }
                    else if (  tracking == Tracking.PERSON
                            || tracking == Tracking.FACE
                            || tracking == Tracking.HORSE
                            || tracking == Tracking.DOG
                    ){
                        synchronized (aiTrackingLock){
                                processNV21Image();
                        }
                    }
                } catch (Throwable t) {
                    Log.e(TAG, "Exception thrown from receiver.", t);
                }
            }
        }
    }
}