// Copyright 2018 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
package app.pivo.android.plussdk;

import android.graphics.ImageFormat;

import androidx.annotation.GuardedBy;

import app.pivo.android.plussdk.tracking.FrameMetadata;
import app.pivo.android.plussdk.tracking.GraphicOverlay;


/**
 * @author murodjon
 * Action Tracking Processor.
 */
class ActionTrackingProcessor implements VisionImageProcessor {

    private ActionTracker tracker;
    private int mode;

    public ActionTrackingProcessor(ActionTracker tracker, int mode) {
        this.tracker = tracker;
        this.mode = mode;
    }

    @GuardedBy("this")
    private byte[] latestImage;

    @GuardedBy("this")
    private FrameMetadata latestImageMetaData;

    // To keep the images and metadata in process.
    @GuardedBy("this")
    private byte[] processingImage;

    @GuardedBy("this")
    private FrameMetadata processingMetaData;

    private synchronized void processLatestImageRest() {
        processingImage = latestImage;
        processingMetaData = latestImageMetaData;
        latestImage = null;
        latestImageMetaData = null;
        if (processingImage != null && processingMetaData != null) {
            if (processingMetaData.getWidth()<0 || processingMetaData.getHeight()<0)return;

            FrameMetadata metadata = new FrameMetadata.Builder()
                    .setWidth(processingMetaData.getWidth())
                    .setHeight(processingMetaData.getHeight())
                    .setRotation(processingMetaData.getRotation())
                    .setCameraFacing(processingMetaData.isFrontCamera())
                    .build();

                tracker.updateCameraFrame(processingImage, ImageFormat.NV21, metadata, processingMetaData.isFrontCamera()?1:0, processingMetaData.getRotation(), mode);
        }
    }

    @Override
    public void process(byte[] data, FrameMetadata frameMetadata, GraphicOverlay graphicOverlay){
        latestImage = data;
        latestImageMetaData = frameMetadata;
        processLatestImageRest();
    }

    public void stop() {
    }
}
