package app.pivo.android.plussdk

import android.content.Context
import android.graphics.PointF
import android.util.Log
import android.util.Size
import android.view.Surface
import app.pivo.android.basicsdk.PivoSdk
import app.pivo.android.plussdk.controller.ControlTracker
import app.pivo.android.plussdk.controller.PivoConfigulation
import app.pivo.android.plussdk.controller.model.Movement
import app.pivo.android.plussdk.controller.model.MultiMovement
import app.pivo.android.plussdk.controller.model.MultiMovingDirection
import app.pivo.android.plussdk.controller.model.SensitivityValue
import app.pivo.android.plussdk.controller.model.TrackingType
import app.pivo.android.plussdk.tracking.FrameMetadata
import app.pivo.android.podsdk.model.HorizontalDirection
import app.pivo.android.podsdk.model.MixedDirection
import app.pivo.android.podsdk.model.PivoDeviceCategory
import app.pivo.android.podsdk.model.SpeedUnit
import java.util.concurrent.atomic.AtomicBoolean


typealias SDKSpeed = app.pivo.android.podsdk.model.Speed
class NativeController: ControlTracker, Callback {

    private val TAG = this.javaClass.simpleName

    // isLagecyController == true : use return value
    // isLagecyController == false : use callbaack funcition - callback function will use kalman filter
    private var isLagecyController = false

    private var isCreateController = false
    private var isUpdatingSetting = false

    private var isAllowedMultiMovement = false

    private var plusSdk: PivoPlusSdk? = null
    private var configuration: PivoConfigulation? = null
    private var deviceController: PivoDeviceCategory? = null

    companion object {
        private const val CENTER_H_THRESHOLD = 0.05f
        private const val CENTER_V_THRESHOLD = 0.05f

        const val DEFAULT_HORIZONTAL_ROTATION_ANGLE = 40
        const val DEFAULT_VERTICAL_ROTATION_ANGLE = 30

    }
    private val lock = Object()

    var frontCamera: Boolean = true

    var trackingType: TrackingType = TrackingType.BODY

    var zoomScale: Float = 1f

    var frameSize: Size = Size(1280, 720)

    var sensitivityValue: SensitivityValue = SensitivityValue.NORMAL

    var orientation: Int = Surface.ROTATION_0

    var isOrientationLocked: Boolean = false

    var isPredictiveFollowOn: Boolean = false
        set(value) {
            field = value
        }

    private val pauseTracking = AtomicBoolean(false)

    init {
        try {
            synchronized(lock = lock) {
                System.loadLibrary("native_controller")
                Log.i("NativeController", "native_controller detected")
            }
        } catch (e: UnsatisfiedLinkError) {
            Log.e("NativeController", "native_controller not detected")
        }
    }

    fun initConfig(context: Context, sensitivityValue: SensitivityValue, trackingType: TrackingType) {
        try {
            deviceController = PivoSdk.getInstance().getDeviceInfo()?.getPivoCategory()
            plusSdk = PivoPlusSdk.getInstance()
            configuration = PivoConfigulation(context)

            this.sensitivityValue = sensitivityValue
            this.trackingType = trackingType

            deviceController?.let {controller ->
                if (controller is PivoDeviceCategory.PivoPodX)
                    isAllowedMultiMovement = true
            }

            configuration?.initConfig(sensitivityValue, trackingType)
            val sensitivityConfiguration = configuration?.getSensitivityConfig()

            if (!isCreateController) {
                synchronized(lock = lock) {
                    createController(
                        plusSdk!!.getSupportedSpeeds(),
                        intArrayOf(),
                        getSpeedType(SpeedUnit.SEC_PER_ROUND)
                    )
                    isCreateController = true

                    pauseDispatching()
                }
            }
            sensitivityConfiguration?.let { it ->
                updatePIDParams(it.getHPDValues().kp, 0f, it.getHPDValues().kd)

                val config = configuration?.getSensitivityConfig()
                config?.let {
                    if (isAllowedMultiMovement) {
                        update2DSensitivity(
                            it.getHPDValues().minSpeed.toInt(),
                            it.getHPDValues().maxSpeed.toInt(),
                            it.getVPDValues().minSpeed.toInt(),
                            it.getVPDValues().maxSpeed.toInt()
                        )
                    } else {
                        updateSensitivity(
                            it.getHPDValues().minSpeed.toInt(),
                            it.getHPDValues().maxSpeed.toInt()
                        )
                    }
                }
            }

            updateThreshold(
                CENTER_H_THRESHOLD,
                CENTER_V_THRESHOLD,
                plusSdk!!.getSupportedSpeeds().last(),
                0
            )
        } catch (e: Exception) {
            e.printStackTrace()
            Log.e(TAG, "Failed to initConfig")
        }
    }

    override fun track(movement: Movement) {
        //TODO("Not yet implemented")
    }

    fun track(currentPoint: PointF) {
        if (!isCreateController) return

        if (isUpdatingSetting) return

        //Log.d(TAG, "TTT orientation : " + orientation + " frameSize : " + frameSize + " currentPoint : " + currentPoint)
        var normalizedCenterPos = getNormalizedCenterPoint(currentPoint.x, currentPoint.y, orientation)

        updateAlignment(0.5f, 0.5f)
        //Log.d(TAG, "TTT normalized : " + normalizedCenterPos + " threshold x : " + abs(0.5f - normalizedCenterPos.x) + " threshold y : " + abs(0.5f - normalizedCenterPos.y))
        val movement = followTarget(normalizedCenterPos.x, normalizedCenterPos.y)

        if (isLagecyController) {
            if (isAllowedMultiMovement)
                rotateToMultiDirection(movement)
            else
                rotateHorizontally(movement.hDirection, movement.hSpeed.toInt())
        } else {
            resumeDispatching()
        }

    }

    fun setMetadata(metadata: FrameMetadata) {
        frontCamera = metadata.isFrontCamera()
        orientation = metadata.getRotation()
        isOrientationLocked = metadata.isOrientationLocked

        if (orientation % 2 == 1) {// portrait
            frameSize = Size(
                minOf(metadata.height, metadata.width),
                maxOf(metadata.height, metadata.width)
            )
            updateOrientation(true)
        } else {
            frameSize = Size(
                maxOf(metadata.height, metadata.width),
                minOf(metadata.height, metadata.width)
            )
            updateOrientation(false)
        }
    }

    private fun rotateToMultiDirection(
        movement: MultiMovement
    ) {
        PivoSdk.getInstance().turn(
            direction = movement.mapToMultiMovement(isFrontCamera = frontCamera).toSDKType(),
            hSpeed = movement.hSpeed.toInt(), hDegree = movement.hRotationDegree,
            vSpeed = movement.vSpeed.toInt(), vDegree = movement.vRotationDegree
        )
    }

    fun MultiMovingDirection.toSDKType(): MixedDirection = when (this) {
        MultiMovingDirection.RIGHT_UP -> MixedDirection.RIGHT_UP
        MultiMovingDirection.RIGHT_DOWN -> MixedDirection.RIGHT_DOWN
        MultiMovingDirection.LEFT_UP -> MixedDirection.LEFT_UP
        MultiMovingDirection.LEFT_DOWN -> MixedDirection.LEFT_DOWN
    }

    private fun rotateHorizontally(direction: Int, outputSpeed: Int) {
        when (direction) {
            MultiMovement.NativeMovement_LEFT ->
                PivoSdk.getInstance().turnLeft(DEFAULT_HORIZONTAL_ROTATION_ANGLE, outputSpeed)

            MultiMovement.NativeMovement_RIGHT ->
                PivoSdk.getInstance().turnRight(DEFAULT_HORIZONTAL_ROTATION_ANGLE, outputSpeed)
            else ->
                PivoSdk.getInstance().stop()
        }
    }

    private fun getNormalizedCenterPoint(
        centerx: Float,
        centery: Float,
        deviceOrientation: Int
    ): PointF {
        var normalizeX : Float
        var normalizeY : Float

        return when (deviceOrientation) {
            0, 2 -> { // landscape
                if (isOrientationLocked) {
                    if (deviceOrientation == 2) {
                        normalizeX = centerx / frameSize.height.toFloat()
                        if (frontCamera)
                            normalizeY = centery / frameSize.width.toFloat()
                        else
                            normalizeY = 1 - centery / frameSize.width.toFloat()
                    } else {
                        normalizeX = 1 - centerx / frameSize.height.toFloat()
                        if (frontCamera)
                            normalizeY = 1- centery / frameSize.width.toFloat()
                        else
                            normalizeY = centery / frameSize.width.toFloat()
                    }
                    PointF(normalizeY, normalizeX)
                } else {
                    normalizeX = centerx / frameSize.width.toFloat()
                    normalizeY = centery / frameSize.height.toFloat()
                    PointF(normalizeX, normalizeY)
                }
            }
            3 -> { // portrait
                normalizeX = 1 - centerx / frameSize.width.toFloat()
                normalizeY = centery / frameSize.height.toFloat()
                PointF(normalizeX, normalizeY)
            }
            1 -> {
                normalizeX = centerx / frameSize.width.toFloat()
                normalizeY = centery / frameSize.height.toFloat()
                PointF(normalizeX, normalizeY)
            }
            else -> {
                normalizeX = centerx / frameSize.width.toFloat()
                normalizeY = centery / frameSize.height.toFloat()
                PointF(normalizeX, normalizeY)
            }
        }
    }

    /**
     * This is a callback that is sent from the native code.
     * It sends commands with 20Hz frequency (every 50ms)
     */
    override fun onReady(hSpeed: Float, hDirection: Int, hDegree: Int, vSpeed: Float, vDirection: Int, vDegree: Int) {
        //Log.d(TAG, "TTT --> onReady hDirection : " + hDirection + " hSpeed : " + hSpeed)
        if (PivoSdk.getInstance().getDeviceInfo()?.getPivoCategory() != null) {
         if (PivoSdk.getInstance().getDeviceInfo()!!.getPivoCategory() is PivoDeviceCategory.PivoPodX) {
             PivoSdk.getInstance().turn(
                 getMixedDirection(hDirection, vDirection),
                 hSpeed = hSpeed.toInt(), hDegree = hDegree,
                 vSpeed = vSpeed.toInt(), vDegree = vDegree,
             )
         } else {
             if (getHDirection(hDirection) == HorizontalDirection.RIGHT) {
                 PivoSdk.getInstance().turnRight(hSpeed.toInt(), hDegree)
             } else {
                 PivoSdk.getInstance().turnLeft(hSpeed.toInt(), hDegree)
             }
         }
        } else {
            if (getHDirection(hDirection) == HorizontalDirection.RIGHT) {
                PivoSdk.getInstance().turnRight(hSpeed.toInt(), hDegree)
            } else {
                PivoSdk.getInstance().turnLeft(hSpeed.toInt(), hDegree)
            }
        }
    }

    private fun getMixedDirection(hDirection: Int, vDirection: Int) : MixedDirection {
        return when(hDirection) {
            MultiMovement.NativeMovement_LEFT -> when (vDirection) {
                MultiMovement.NativeMovement_UP -> MixedDirection.LEFT_UP
                MultiMovement.NativeMovement_DOWN -> MixedDirection.LEFT_DOWN
                else -> MixedDirection.LEFT_UP
            }
            MultiMovement.NativeMovement_RIGHT -> when (vDirection) {
                MultiMovement.NativeMovement_UP -> MixedDirection.RIGHT_UP
                MultiMovement.NativeMovement_DOWN -> MixedDirection.RIGHT_DOWN
                else -> MixedDirection.RIGHT_UP
            }
            else -> MixedDirection.LEFT_UP
        }
    }

    private fun getHDirection(hDirection: Int) : HorizontalDirection {
        return when(hDirection) {
            MultiMovement.NativeMovement_RIGHT -> HorizontalDirection.RIGHT
            else -> HorizontalDirection.LEFT
        }
    }

    private fun getSpeedType(speedUnit: SpeedUnit): Int {
        if (speedUnit == SpeedUnit.SEC_PER_ROUND)
            return 0
        return 1
    }

    external fun createController(
        horizontalSpeeds: IntArray,
        verticalSpeeds: IntArray,
        speedUnit: Int,
    )

    external fun pauseDispatching()
    external fun resumeDispatching()

    external fun followTarget(
        x: Float, y: Float
    ): MultiMovement

    external fun updateSensitivity(from:Int, to:Int)
    external fun update2DSensitivity(hFrom:Int, hTo:Int, vFrom:Int, vTo:Int)

    external fun updatePIDParams(kp: Float, ki: Float, kd: Float)

    external fun updateZoom(zoom:Float)

    external fun updateOrientation(isPortrait: Boolean)

    external fun releaseController()

    external fun updateAlignment(horizontal: Float, vertical: Float)

    external fun updateThreshold(hThreshold: Float, vThreshold: Float, hMinSpeed: Int, vMinSpeed: Int)
}

interface Callback {
    fun onReady(hSpeed: Float, hDirection: Int, hDegree: Int, vSpeed: Float, vDirection: Int, vDegree: Int)
}