package app.pivo.android.podsdk.events

import android.util.Log
import android.util.SparseArray
import androidx.annotation.IntDef
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.functions.Consumer
import io.reactivex.subjects.PublishSubject
import java.util.*

/**
 * Created by murodjon on 2021/01/19
 */
/**
 * PivoEventBus allows to listen and send Pivo events [PivoEvent] in different components of the application.
 */
object PivoEventBus {
    private val sSubjectMap = SparseArray<PublishSubject<Any>>()
    private val sSubscriptionsMap: MutableMap<Any, CompositeDisposable> = HashMap()

    /**
     * Invalid reply
     */
    const val INVALID_REPLY_RECEIVED = -1

    /**
     * Connection completed subject.
     */
    const val CONNECTION_COMPLETED = 0

    /**
     * Connection failed or an app is disconnected from Pivo.
     */
    const val CONNECTION_FAILED = 1

    /**
     * Scan device subject.
     */
    const val SCAN_DEVICE = 2

    /**
     * Battery level subject
     */
    const val BATTERY_LEVEL = 3

    /**
     * Name is changed subject.
     */
    const val NAME_CHANGED = 4

    /**
     * DEVICE profile subject
     * Mac address, serial number, device info can be listened
     */
    const val DEVICE_PROFILE = 5

    /**
     * Pivo notification subject
     */
    const val MOVEMENT_NOTIFICATION = 6

    /**
     * REMOTE CONTROLLER subject
     */
    const val REMOTE_CONTROLLER = 7

    /**
     * PAIRING subject
     */
    const val PAIRING = 8

    /**
     * Sound subject
     */
    const val SOUND = 9

    /**
     * LED subject
     */
    const val LED = 11

    /**
     * Get the subject or create it if it's not already in memory.
     */
    private fun getSubject(@Subject subjectCode: Int): PublishSubject<Any> {
        var subject = sSubjectMap[subjectCode]
        if (subject == null) {
            subject = PublishSubject.create()
            subject.subscribeOn(AndroidSchedulers.mainThread())
            sSubjectMap.put(subjectCode, subject)
        }
        return subject
    }

    /**
     * Get the CompositeDisposable or create it if it's not already in memory.
     */
    private fun getCompositeDisposable(`object`: Any): CompositeDisposable {
        var compositeDisposable = sSubscriptionsMap[`object`]
        if (compositeDisposable == null) {
            compositeDisposable = CompositeDisposable()
            sSubscriptionsMap[`object`] = compositeDisposable
        }
        return compositeDisposable
    }

    /**
     * Subscribe to the specified subject and listen for updates on that subject. Pass in an object to associate
     * your registration with, so that you can unsubscribe later.
     * <br></br><br></br>
     * **Note:** Make sure to call [PivoEventBus.unregister] to avoid memory leaks.
     */
    fun subscribe(@Subject subject: Int, lifecycle: Any, action: Consumer<Any>) {
        val disposable = getSubject(subject).subscribe(action,
            { throwable: Throwable? ->
                Log.e(
                    "PivoEventBus",
                    "Exception: ${throwable?.message}"
                )
            }) //.subscribe(action);
        getCompositeDisposable(lifecycle).add(disposable)
    }

    /**
     * Unregisters this object from the bus, removing all subscriptions.
     * This should be called when the object is going to go out of memory.
     */
    fun unregister(lifecycle: Any) {
        //We have to remove the composition from the map, because once you dispose it can't be used anymore
        val compositeDisposable = sSubscriptionsMap.remove(lifecycle)
        compositeDisposable?.dispose()
    }

    /**
     * Publish an object to the specified subject for all subscribers of that subject.
     */
    fun publish(@Subject subject: Int, message: Any) {
        getSubject(subject).onNext(message)
    }

    @kotlin.annotation.Retention(AnnotationRetention.SOURCE)
    @IntDef(
        INVALID_REPLY_RECEIVED,
        CONNECTION_COMPLETED,
        CONNECTION_FAILED,
        SCAN_DEVICE,
        NAME_CHANGED,
        MOVEMENT_NOTIFICATION,
        DEVICE_PROFILE,
        REMOTE_CONTROLLER,
        BATTERY_LEVEL,
        LED,
        SOUND,
        PAIRING
    )
    internal annotation class Subject
}