package app.pivo.android.podsdk.cmd.builder.impl

import app.pivo.android.podsdk.auth.PodAuthenticator
import app.pivo.android.podsdk.cmd.builder.CommandBuilder
import app.pivo.android.podsdk.cmd.builder.ext.getHexValue
import app.pivo.android.podsdk.model.*
import app.pivo.android.podsdk.ota.FirmwareChunkProvider
import java.util.*
import kotlin.math.floor

/**
 * Created by murodjon on 2021/08/17
 */
internal class PodCmdBuilderImpl : CommandBuilder {

    override fun getOTAInitiationBytes(): ByteArray {
        return byteArrayOf(0x05, 0xAF.toByte(), 0x8F.toByte(), 0x00, 0x00, 0x00)
    }

    override fun getFirmwareFileChunkBytes(
        firmwareBytes: ByteArray,
        packetSize: Int
    ): ByteArray {
        return FirmwareChunkProvider.getInitialFirmwareChunkBytes(firmwareBytes, packetSize)
    }

    override fun getFirmwareFileChunkBytes(
        firmwareBytes: ByteArray,
        replyBytes: ByteArray,
        packetSize: Int
    ): ByteArray {
        return FirmwareChunkProvider.getFirmwareFileChunkBytes(
            firmwareBytes, replyBytes,
            packetSize
        )
    }

    override fun getDeviceInfoBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x03.toByte(), 0x00, 0x00, 0x00)

    override fun getBatteryLevelBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x04, 0x00, 0x00, 0x00)

    override fun getChangeNameBytes(name: String): ByteArray? {
        // 5 elements are needed for changing name
        val length = name.length
        // final byte array size for changing name is 5+(length of name)
        val data = ByteArray(length + 5)
        data[0] = 0x05.toByte()
        data[1] = 0xAF.toByte()
        data[2] = 0x7F.toByte()
        data[3] = length.toByte()
        data[4] = 0x00.toByte()

        for (i in name.indices) {
            data[5 + i] = name[i].code.toByte()
        }
        return data
    }

    override fun getMacAddressBytes() = byteArrayOf(0x05, 0xAF.toByte(), 0x1B, 0x00, 0x00, 0x00)

    override fun getSerialNumberBytes(): ByteArray = byteArrayOf(
        0x05, 0xAF.toByte(), 0x1C, 0x00, 0x00, 0x00
    )

    override fun getRemoteSwitcherBytes(remoteEnabled: Boolean) = byteArrayOf(
        0x05,
        0xAF.toByte(),
        0x60.toByte(), if (!remoteEnabled) 0xA0.toByte() else 0xA1.toByte(), 0x00, 0x00
    )

    override fun getTimeoutBytes(time: Int): ByteArray = byteArrayOf(
        0x05,
        0xAF.toByte(), 0xA1.toByte(), 0x00, 0x00, (time / 10).toByte()
    )

    override fun getSoundBytes(sound: SoundType) =
        byteArrayOf(0x05, 0xAF.toByte(), 0x08, sound.getHexValue(), 0x00, 0x00)

    override fun getNotifierSwitcherBytes(notifierEnabled: Boolean): ByteArray = byteArrayOf(
        0x05,
        0xAF.toByte(),
        0x09.toByte(),
        if (notifierEnabled) 0xA1.toByte() else 0xA0.toByte(),
        0x00,
        0x00
    )

    override fun getMoveHorizontallyBytes(
        version: Int,
        direction: HorizontalDirection,
        speed: LegacySpeed,
        degree: Int
    ): ByteArray?{
        return when {
            contains(getRemoteSupportedSpeedsSecondsPerRound(version), speed.value) -> {
                getRotationBytes(version, degree, direction, speed.value)
            }
            else -> null
        }
    }

    override fun getMoveHorizontallyBytes(
        version: Int,
        direction: HorizontalDirection,
        speed: Speed,
        degree: Int
    ): ByteArray? {
        this.mSpeed = speed
        return when {
            version == 0 -> {
                getRotationBytes(version, degree, direction, mSpeed.value)
            }
            version > 0 -> {
                when {
                    contains(getRemoteSupportedSpeedsSecondsPerRound(version), mSpeed.value) -> {
                        getRotationBytes(version, degree, direction, mSpeed.value)
                    }
                    contains(getV0AboveSpeeds(), mSpeed.value) -> {
                        val adaptedSpeed = if (version == 7){
                            mSpeed.value - 2
                        }else mSpeed.value
                        getV1RotationBytes(degree, direction, adaptedSpeed / 2)
                    }
                    else -> {
                        null
                    }
                }
            }
            else -> null
        }
    }

    private fun contains(speeds: IntArray, speed: Int): Boolean {
        for (value in speeds) {
            if (value == speed) return true
        }
        return false
    }

    private var mSpeed: Speed = Speed(10)
    private var min4RotationDegreeEnabled = true// pod can rotate with minimum 4 and 1 degree
    private fun getRotationBytes(
        version: Int,
        angle: Int,
        direction: HorizontalDirection,
        speed: Int
    ): ByteArray {
        val data: ByteArray = byteArrayOf(0x05, 0xAF.toByte(), 0x21, 0x00, 0x00, 0x00)
        if (!min4RotationDegreeEnabled) {//min rotation degree is 1.
            if (angle != -1) {
                data[2] = (if (direction === HorizontalDirection.RIGHT) 0x25 else 0x27).toByte()
                data[3] = getSpeedByte(version, speed)
                data[5] = angle.toByte()
            } else {
                data[2] = (if (direction === HorizontalDirection.RIGHT) 0x25 else 0x27).toByte()
                data[3] = getSpeedByte(version, speed)
                data[5] = 0x00.toByte()
            }
        } else {//min rotation degree is 4.
            if (angle != -1) {
                data[2] = (if (direction === HorizontalDirection.RIGHT) 0x21 else 0x23).toByte()
                data[3] = getSpeedByte(version, speed)
                data[5] = floor((angle / 4).toDouble()).toInt().toByte()
            } else {
                data[2] = (if (direction === HorizontalDirection.RIGHT) 0x21 else 0x23).toByte()
                data[3] = getSpeedByte(version, speed)
                data[5] = 0x00.toByte()
            }
        }
        return data
    }

    /**
     * This function should be called if the pivo version is v1,1
     * @param angle rotation angle
     * @param speed speed of
     * @param direction is pivo direction
     * @return an array of byte values is returned from function
     */
    private fun getV1RotationBytes(
        angle: Int,
        direction: HorizontalDirection,
        speed: Int
    ): ByteArray {
        val data: ByteArray = byteArrayOf(0x05, 0xAF.toByte(), 0x21, 0x00, 0x00, 0x00)
        if (!min4RotationDegreeEnabled) {
            //min rotation degree is 1.
            if (angle != -1) {
                data[2] = (if (direction === HorizontalDirection.RIGHT) 0x26 else 0x28).toByte()
                data[3] = getV1SpeedByte(speed)
                data[5] = angle.toByte()
            } else {
                data[2] = (if (direction === HorizontalDirection.RIGHT) 0x26 else 0x28).toByte()
                data[3] = getV1SpeedByte(speed)
                data[5] = 0x00.toByte()
            }
        } else {
            //min rotation degree is 4.
            if (angle != -1) {
                data[2] = (if (direction === HorizontalDirection.RIGHT) 0x22 else 0x24).toByte()
                data[3] = getV1SpeedByte(speed)
                data[5] = floor((angle / 4).toDouble()).toInt().toByte()
            } else {
                data[2] = (if (direction === HorizontalDirection.RIGHT) 0x22 else 0x24).toByte()
                data[3] = getV1SpeedByte(speed)
                data[5] = 0x00.toByte()
            }
        }
        return data
    }

    /**
     * This function generates speed for V1.
     * @return an array of byte values is returned from function
     */
    private fun getV1SpeedBytes(speed: Int): ByteArray? {
        val data: ByteArray = byteArrayOf(0x05, 0xAF.toByte(), 0x21, 0x00, 0x00, 0x00)
        data[2] = 0x10.toByte()
        data[3] = getV1SpeedByte(speed)
        return data
    }

    /**
     * This function generates speed for V0 and V1.
     * @return an array of byte values is returned from function
     */
    private fun getSpeedBytes(version: Int, speed: Speed): ByteArray? {
        val data: ByteArray = byteArrayOf(0x05, 0xAF.toByte(), 0x21, 0x00, 0x00, 0x00)
        data[2] = 0x06.toByte()
        data[3] = getSpeedByte(version, speed.value)
        return data
    }

    /**
     * This function returns byte for specific speed level, it is supported by V1 and V0
     * @return a byte value is returned from function
     */
    /**
     * This function returns byte for specific speed level, it is supported by V1 and V0
     * @return a byte value is returned from function
     */
    /**
     * This function returns byte for specific speed level, it is supported by V1 and V0
     * @return a byte value is returned from function
     */
    private fun getSpeedByte(version: Int, speed: Int): Byte {
        var data = 0x00.toByte()
        when (speed) {
            10 -> data = 0x11.toByte()
            20 -> data = 0x12.toByte()
            30 -> data = 0x13.toByte()
            50 -> data = 0x14.toByte()
            60 -> data = 0x15.toByte()
            120 -> data = 0x16.toByte()
            300 -> data = 0x17.toByte()
            600 -> data = 0x18.toByte()
            1200 -> data = 0x19.toByte()
            1800 -> data = 0x1A.toByte()
            3600 -> data = 0x1B.toByte()
        }
        if (version > 0) {
            when (speed) {
                7200 -> data = 0x1C.toByte()
                10800 -> data = 0x1D.toByte()
                21600 -> data = 0x1E.toByte()
                43200 -> data = 0x1F.toByte()
                64800 -> data = 0x20.toByte()
                86400 -> data = 0x21.toByte()
            }
        }
        return data
    }

    /**
     * This function returns byte for specific speed level, it is supported only by V1
     * @return a byte value is returned from function
     */
    private fun getV1SpeedByte(speed: Int): Byte {
        if (speed in 2..255) {
            val hexCode = Integer.toHexString(speed)
            return hexCode.toInt(16).toByte()
        }
        return 0x01.toByte()
    }

    override fun getSetHorizontalSpeedBytes(version: Int, speed: Speed): ByteArray? {
        mSpeed = speed
        if (version == 0) {
            return getSpeedBytes(version, speed)
        } else if (version > 0) {
            when {
                contains(getRemoteSupportedSpeedsSecondsPerRound(version), speed.value) -> {
                    return getSpeedBytes(version, speed)
                }
                contains(getV0AboveSpeeds(), speed.value) -> {
                    return getV1SpeedBytes(speed.value / 2)
                }
                else -> {
                    mSpeed = Speed(10, SpeedUnit.SEC_PER_ROUND)
                }
            }
        }
        return ByteArray(0)
    }

    override fun getStopBytes() = byteArrayOf(0x05, 0xAF.toByte(), 0x00, 0x00, 0x00, 0x00)

    override fun getStartPairingBytes() = byteArrayOf(
        0x05,
        0xAF.toByte(), 0x51.toByte(), 0x00.toByte(), 0x00, 0x00
    )

    override fun getCancelPairingBytes() = byteArrayOf(
        0x05,
        0xAF.toByte(), 0x52.toByte(), 0x00.toByte(), 0x00, 0x00
    )

    private var inquiryForAuth = 0
    override fun getAuthenticatePodBytes(
        deviceCategory: PivoDeviceCategory,
        seqNum: Byte,
        answerFromPod: Int,
        inquiryFromPod: Int
    ): ByteArray? {
        val current: ByteArray
        if (seqNum.toInt() == 0) {
            val randInquiry = Random(System.currentTimeMillis())
            val a = randInquiry.nextInt()

//            Log.e("TTT","inquriy: " + a);
            current = byteArrayOf(
                0x05, 0xAF.toByte(),
                0x6F, 0x00/*sequence*/, 0x00/*direction*/,
                0x00,
                (a shr 24).toByte(),
                (a shr 16).toByte(), (a shr 8).toByte(), a.toByte(), 0x00, 0x00, 0x00, 0x00
            )
            inquiryForAuth = a
        } else if (seqNum.toInt() == 2) {
            val authMode = PodAuthenticator(deviceCategory)
            if (!authMode.verifyAuth(inquiryForAuth, answerFromPod)) {
//                Log.e(TAG, "Error: Pivo Authenticatin Fail!");
//                current = byteArrayOf(
//                    0x05, 0xAF.toByte(),
//                    0x6F, 0x02/*sequence*/, 0x00/*direction*/,
//                    0x00/*result*/,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00
//                )
                return null
            } else {
                val code: Int = authMode.getAnswer(inquiryFromPod)

//                Log.e(TAG, "Inform : Pivo Authenticatin Succeed!");
                current = byteArrayOf(
                    0x05, 0xAF.toByte(),
                    0x6F, 0x02/*sequence*/, 0x00/*direction*/,
                    0x01/*result*/,
                    0x00,
                    0x00,
                    0x00,
                    0x00,
                    (code shr 24).toByte(),
                    (code shr 16).toByte(),
                    (code shr 8).toByte(),
                    code.toByte()
                )
            }
        } else {// invalid sequence number
            return null
        }
        return current
    }

    private fun getV0Speeds() =
        intArrayOf(10, 20, 30, 50, 60, 120, 300, 600, 1200, 1800, 3600)

    private fun getV0AboveSpeeds(): IntArray {
        val internalUse = true
        val delta = if (internalUse) 3 else 0
        val supportedSpeed =
            IntArray(255 - 5 + 1 +  /*remote supported speeds(start from 600 spr and 86400 spr)*/+10 + delta)
        var count = 0

        for (i in 5 - delta..255) {
            supportedSpeed[count] = i * 2
            count++
        }
        supportedSpeed[count] = 600
        supportedSpeed[count + 1] = 1200
        supportedSpeed[count + 2] = 1800
        supportedSpeed[count + 3] = 3600
        supportedSpeed[count + 4] = 7200
        supportedSpeed[count + 5] = 10800
        supportedSpeed[count + 6] = 21600
        supportedSpeed[count + 7] = 43200
        supportedSpeed[count + 8] = 64800
        supportedSpeed[count + 9] = 86400
        return supportedSpeed
    }

    override fun getHorizontalSpeedsSecPerRound(version: Int): IntArray {
        return when {
            version == 0 -> {
                getV0Speeds()
            }
            version > 0 -> {
                getV0AboveSpeeds()
            }
            else -> {
                intArrayOf(0)
            }
        }
    }

    private fun getRemoteSupportedSpeedsSecondsPerRound(version: Int): IntArray {
        return when {
            version == 0 -> intArrayOf(
                10,
                20,
                30,
                50,
                60,
                120,
                300,
                600,
                1200,
                1800,
                3600
            )
            version > 0 -> intArrayOf(
                10,
                20,
                30,
                50,
                60,
                120,
                300,
                600,
                1200,
                1800,
                3600,
                7200,
                10800,
                21600,
                43200,
                64800,
                86400
            )
            else -> intArrayOf(0)
        }
    }
}