package app.pivo.android.podsdk.controller.pivo.impl

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.util.Log
import app.pivo.android.podsdk.auth.AuthState
import app.pivo.android.podsdk.controller.ble.BluetoothController
import app.pivo.android.podsdk.cmd.builder.CommandBuilder
import app.pivo.android.podsdk.cmd.builder.factory.impl.CommandBuilderFactoryImpl
import app.pivo.android.podsdk.controller.ble.BluetoothControllerCallback
import app.pivo.android.podsdk.controller.ble.BluetoothControllerImpl
import app.pivo.android.podsdk.controller.pivo.PivoDeviceManager
import app.pivo.android.podsdk.model.PivoDevice
import app.pivo.android.podsdk.model.*
import app.pivo.android.podsdk.ota.FirmwareUpdateState
import app.pivo.android.podsdk.cmd.parser.ReplyCommandParser
import app.pivo.android.podsdk.cmd.parser.factory.impl.ReplyCommandParserFactoryImpl
import app.pivo.android.podsdk.model.callback.*
import app.pivo.android.podsdk.util.HexUtil
import app.pivo.android.podsdk.util.ext.launchPeriodicAsync
import app.pivo.android.podsdk.uuid.PodUUIDs
import kotlinx.coroutines.*
import kotlinx.coroutines.Dispatchers.Default

/**
 * Created by murodjon on 2021/05/20
 */
internal class PivoDeviceManagerImpl(
    context: Context
) : PivoDeviceManager {

    private val listenersMap = mutableMapOf<Any, OnPodListener>()

    private var bluetoothController: BluetoothController? = null

    private var pivoDeviceType = PivoDeviceType.POD
    private var podDevice: PivoDevice? = null

    private var otaBytes: ByteArray? = null
    private var deviceInfo: DeviceInfo? = null

    private var isManualAuthentication = false

    // command builder
    private var commandBuilder: CommandBuilder? = null

    // reply command parser
    private var commandParser: ReplyCommandParser? = null
    private var connectionJob: Job? = null

    init {
        bluetoothController =
            BluetoothControllerImpl(context, object : BluetoothControllerCallback {
                override fun onAddDevice(device: PivoDevice) {
                    listenersMap.values.forEach { it.onDiscovered(device) }
                }

                override fun onConnectionFailed() {
                    onDisconnected()
                }

                override fun onConnectionEstablished(type: PivoDeviceType) {
                    initDeviceController(type)

                    connectionJob = CoroutineScope(Default).launch {
                        delay(3_000L)
                        Log.e(TAG, "Could not get Authentication or Serial Number or DeviceInfo Event")
                        disconnect()
                    }
                }

                override fun onNotificationReceived(bytes: ByteArray) {
                    this@PivoDeviceManagerImpl.onNotificationReceived(bytes)
                }
            }, PodUUIDs)
    }

    private fun initDeviceController(type: PivoDeviceType) {
        pivoDeviceType = type

        commandBuilder = CommandBuilderFactoryImpl(pivoDeviceType)
            .getCommandBuilder()
        podDevice?.let {
            commandParser = ReplyCommandParserFactoryImpl(pivoDeviceType, it)
                .getReplyCommandParser()
        }

        if (type != PivoDeviceType.POD_REMOTE) {
            Handler(Looper.getMainLooper()).post {
                requestDeviceInfo()
            }
        } else {
            onConnected()
        }
    }

    private fun onNotificationReceived(data: ByteArray) {
        val event = commandParser?.getEvent(data)

        Log.e(
            TAG,
            "onNotificationReceived: ${HexUtil.convertHexToString(data)}, event: $event"
        )
        returnOnMainThread {
            when (event) {
                is PivoEvent.FirmwareUpdateEvent -> {
                    onOTAProgress(event.firmwareUpdateState, data)
                }
                is PivoEvent.DeviceInfoEvent -> {
                    onDeviceInfoReceived(event.deviceInfo, data)
                }
                is PivoEvent.SerialNumberEvent -> {
                    onSerialNumberReceived(event.serialNumber)
                }
                is PivoEvent.AuthenticationEvent -> {
                    onAuthBytesReceived(event.state)
                }
                is PivoEvent.BatteryEvent -> {
                    listenersMap.values.forEach { it.onBatteryReceived(event.battery.level) }
                }
                is PivoEvent.HorizontalMovementEvent -> {
                    listenersMap.values.forEach { it.onMovingHorizontally(event.movement) }
                }
                is PivoEvent.HorizontalResetToOrigin -> {
                    listenersMap.values.forEach { it.onResetPositionHorizontally() }
                }
                is PivoEvent.VerticalMovementEvent -> {
                    listenersMap.values.forEach { it.onMovingVertically(event.movement) }
                }
                is PivoEvent.HorizontalPositionEvent -> {
                    listenersMap.values.forEach { it.onGetHorizontalPosition(event.horizontalPosition) }
                }
                is PivoEvent.GoToHorizontalPositionEvent -> {
                    listenersMap.values.forEach { it.onHorizontalPositionChanged(event.goToHorizontalPosition) }
                }
                is PivoEvent.VerticalPositionEvent -> {
                    listenersMap.values.forEach { it.onGetVerticalPosition(event.verticalPosition) }
                }
                is PivoEvent.GoToVerticalPositionEvent -> {
                    listenersMap.values.forEach { it.onVerticalPositionChanged(event.verticalPosition) }
                }
                is PivoEvent.ConnectionTimeOutEvent -> listenersMap.values.forEach {
                    it.onConnectionTimeoutReceived(
                        event.timeOut.timeOut
                    )
                }
                is PivoEvent.CurrentLEDEvent -> listenersMap.values.onEach {
                    it.onGetCurrentLED(
                        event.led
                    )
                }
                is PivoEvent.LEDSetEvent -> listenersMap.values.onEach { it.onSetLED(event.led) }
                is PivoEvent.LEDStopPreset -> listenersMap.values.onEach { it.onLEDStopPreset() }
                is PivoEvent.LEDPresetEvent -> listenersMap.values.onEach {
                    it.onLEDPreset(
                        event.patternType,
                        event.brightnessLevel
                    )
                }
                is PivoEvent.MacAddressEvent -> listenersMap.values.onEach {
                    it.onMacAddressReceived(
                        macAddress = event.macAddress.macAddress
                    )
                }
                is PivoEvent.MixedMovementEvent -> listenersMap.values.forEach {
                    it.onMovingOrthogonally(
                        event.mixedMovement
                    )
                }
                is PivoEvent.NameChangedEvent -> listenersMap.values.forEach { it.onNameChanged(name = "") }
                is PivoEvent.NotifierEvent -> listenersMap.values.forEach { }
                is PivoEvent.RemoteConButtonEvent -> listenersMap.values.forEach {
                    it.onRemoteConEvent(
                        event.remoteConButton
                    )
                }
                is PivoEvent.RemoteConBypassEvent -> listenersMap.values.forEach {
                    it.onRemoteConBypassEnabled(
                        event.bypass
                    )
                }
                is PivoEvent.RemoteConPairingEvent -> listenersMap.values.forEach {
                    it.onPairingStatus(
                        event.pairing.pairingStatus
                    )
                }
                is PivoEvent.SoundEvent -> listenersMap.values.forEach { it.onSound(event.sound.soundType) }
                is PivoEvent.SpeedEvent -> listenersMap.values.forEach { it.onSpeedChange(event.speed) }
                PivoEvent.Stop -> listenersMap.values.forEach { it.onStop() }
                is PivoEvent.InvalidReplyEvent -> {
                }
                null -> {
                }
            }
        }
    }

    private fun onDeviceInfoReceived(device: DeviceInfo, data: ByteArray) {
        deviceInfo = device
        deviceInfo?.setDeviceInfoInHex(HexUtil.convertHexToString(data))
        podDevice?.let {
            deviceInfo?.setMacAddress(it.getMacAddress())
            deviceInfo?.setName(it.getName())
        }

        if (device.getPivoCategory() is PivoDeviceCategory.PivoPod) {
            if ((device.getPivoCategory() as PivoDeviceCategory.PivoPod).version > 1 && !isManualAuthentication) {
                authenticatePod()
            } else {
                onConnected()
            }
        } else if (
            device.getPivoCategory() is PivoDeviceCategory.PivoPodMax ||
            device.getPivoCategory() is PivoDeviceCategory.PivoPodX
        ) {
            if (isManualAuthentication) {
                onConnected()
            } else {
                authenticatePod()
            }
        } else {
            requestSerialNumber()
        }
    }

    private fun onSerialNumberReceived(serialNumber: SerialNumber) {
        listenersMap.values.forEach { it.onSerialNumberReceived(serialNumber = serialNumber.serialNumber) }
        deviceInfo?.setSerialNumber(serialNumber.serialNumber)

        onConnected()
    }

    private fun authenticatePod() {
        commandBuilder?.let { commandBuilder ->
            deviceInfo?.let {
                commandBuilder
                    .getAuthenticatePodBytes(it.getPivoCategory(), 0x00, 0, 0)
                    ?.let { it1 -> bluetoothController?.write(it1) }
            }
        }
    }

    private fun onAuthBytesReceived(state: AuthState) {
        when (state) {
            is AuthState.AuthFailed -> {
                // Status 1: Pivo HW rejected this app
                if (isManualAuthentication) {
                    listenersMap.values.forEach { it.onAuthenticationFailed() }
                } else {
                    disconnect()
                }
            }
            is AuthState.AuthSucceed -> {
                // Status 2: Mutual Authentication Success
                onAuthenticationSucceed()
            }
            is AuthState.AuthProgress -> {
                val sendBytes: ByteArray? = deviceInfo?.let {
                    commandBuilder?.getAuthenticatePodBytes(
                        it.getPivoCategory(),
                        0x02.toByte(),
                        state.answer,
                        if (authWithWrongCode) state.inquiry + 1 else state.inquiry
                    )
                }
                if (sendBytes == null) {
                    if (isConnected()) {
                        disconnect()// disconnect the connection
                    }
                } else {
                    bluetoothController?.write(sendBytes)
                }
            }
        }
    }

    private fun onAuthenticationSucceed() {
        deviceInfo?.let {
            if (isManualAuthentication) {
                listenersMap.values.forEach { it.onAuthenticated() }
            } else {
                if (it.getPivoCategory().version >= 5) {
                    requestSerialNumber()
                } else {
                    onConnected()
                }
            }
        }
    }

    private fun onConnected() {
        connectionJob?.let {
            if (it.isActive) {
                it.cancel()
            }
        }

        returnOnMainThread {
            listenersMap.values.forEach { it.onConnected() }
        }
    }

    private fun returnOnMainThread(f: () -> Unit) {
        val handler = Handler(Looper.getMainLooper())
        handler.post {
            f.invoke()
        }
    }

    private var otaStarted = true
    private fun onOTAProgress(state: FirmwareUpdateState, bytes: ByteArray) {
        when (state) {
            FirmwareUpdateState.INITIATED -> {
                listenersMap.values.forEach { it.onInitiated() }
                startSendingFirmwareChunks()
            }
            FirmwareUpdateState.INITIATION_FAILED -> {
                releaseFirmwareBytes()
                // OTA initiation failed
                listenersMap.values.forEach { it.onInitiationFailed() }
            }
            FirmwareUpdateState.OTA_FAILED -> {
                cancelReSendFileChunks()
                releaseFirmwareBytes()
                listenersMap.values.forEach { it.onOTAFailed() }
            }
            FirmwareUpdateState.FILE_CHUNKS_RECEIVED -> {
                cancelReSendFileChunks()
                val offsetBytes = bytes.copyOfRange(7, bytes.size)

                continueSendingOTABytes(offsetBytes)

                val offset = HexUtil.convertByteArrayToInt(offsetBytes)
                otaBytes?.let { otaBytes ->
                    listenersMap.values.forEach { it.onOTAProgress(offset.toDouble() / otaBytes.size.toDouble() * 100.0) }
                }
            }
            FirmwareUpdateState.ALL_FILE_CHUNKS_RECEIVED -> {
                cancelReSendFileChunks()
                releaseFirmwareBytes()
                // ota update completed
                listenersMap.values.forEach { it.onOTACompleted() }
            }
            FirmwareUpdateState.FILE_CHUNKS_NOT_RECEIVED -> {
                cancelReSendFileChunks()

                continueSendingOTABytes(bytes)
            }
        }
    }

    private var authWithWrongCode = false
    override fun authPod(wrongAuth: Boolean) {
        isManualAuthentication = true
        this.authWithWrongCode = wrongAuth

        authenticatePod()
    }

    private fun startSendingFirmwareChunks() {
        otaStarted = true
        commandBuilder?.let { commandBuilder ->
            otaBytes?.let { firmwareBytes ->
                val bytes = commandBuilder.getFirmwareFileChunkBytes(
                    firmwareBytes,
                    packetSize = otaPacketSize
                )
                bluetoothController?.write(bytes)
            }
        }
    }

    var otaContinuousJob: Deferred<Unit>? = null

    private lateinit var otaResponseData: ByteArray
    private fun continueSendingOTABytes(data: ByteArray) {
        otaResponseData = data
        commandBuilder?.let { commandBuilder ->
            otaBytes?.let { firmwareBytes ->
                val bytes = commandBuilder.getFirmwareFileChunkBytes(
                    firmwareBytes = firmwareBytes,
                    replyBytes = data,
                    packetSize = otaPacketSize
                )
                bluetoothController?.write(bytes)

                resendIfNoReply()
            }
        }
    }

    private fun resendIfNoReply() {
        if (!otaStarted) {
            otaContinuousJob?.cancel()
            otaContinuousJob = null
            return
        }
        otaContinuousJob = CoroutineScope(Dispatchers.IO)
            .launchPeriodicAsync(1000) {
                continueSendingOTABytes(otaResponseData)
            }
    }

    private fun cancelReSendFileChunks() {
        otaContinuousJob?.cancel()

        otaContinuousJob = null
    }

    private fun releaseFirmwareBytes() {
        otaBytes = null
        otaStarted = false
    }

    private fun onDisconnected() {
        commandBuilder = null
        commandParser = null
        podDevice = null
        connectionJob = null

        listenersMap.values.forEach { it.onConnectionFailed() }
    }

    override fun scan() {
        bluetoothController?.scan()
    }

    override fun stopScan() {
        bluetoothController?.cancelScan()
    }

    override fun connectTo(device: PivoDevice) {
        this.podDevice = device
        bluetoothController?.connectTo(device)
    }

    override fun disconnect() {
        bluetoothController?.disconnect()
    }

    override fun isConnected(): Boolean {
        return bluetoothController?.isConnected() ?: false
    }

    override fun getDeviceInfo() = deviceInfo

    override fun requestDeviceInfo() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getDeviceInfoBytes())
    }

    override fun changeName(name: String) {
        commandBuilder?.getChangeNameBytes(name)?.let { bluetoothController?.write(it) }
    }

    override fun requestBatteryLevel() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getBatteryLevelBytes())
    }

    override fun requestMacAddress() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getMacAddressBytes())
    }

    override fun requestSerialNumber() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getSerialNumberBytes())
    }

    override fun stop() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        deviceInfo?.let {
            if (it.getPivoCategory().version == 7) {
                holdTurningHorizontally()
                return@stop
            }
            else bluetoothController?.write(commandBuilder!!.getStopBytes())
        } ?: run {
            bluetoothController?.write(commandBuilder!!.getStopBytes())
        }
    }

    override fun setConnectionTimeout(time: Int) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getTimeoutBytes(time))
    }

    override fun moveHorizontally(direction: HorizontalDirection, speed: Speed, degree: Int) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }

        deviceInfo?.let {
            commandBuilder!!.getMoveHorizontallyBytes(
                it.getPivoCategory().version,
                direction,
                speed,
                degree
            )
                ?.let {
                    bluetoothController?.write(it)
                }
        }
    }

    override fun moveHorizontally(direction: HorizontalDirection, speed: LegacySpeed, degree: Int) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }

        deviceInfo?.let {
            commandBuilder!!.getMoveHorizontallyBytes(
                it.getPivoCategory().version,
                direction,
                speed,
                degree
            )?.let { bytes ->
                bluetoothController?.write(bytes)
            }
        }
    }

    override fun setHorizontalMotorSpeed(speed: Speed) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        deviceInfo?.let {
            commandBuilder!!.getSetHorizontalSpeedBytes(it.getPivoCategory().version, speed)?.let {
                bluetoothController?.write(it)
            }
        }
    }

    override fun getHorizontalSupportedSpeedsInSecondsPerRound(): IntArray {
        if (commandBuilder == null) {
            onDisconnected()
            return intArrayOf(0)
        }
        val speeds =
            deviceInfo?.let { commandBuilder!!.getHorizontalSpeedsSecPerRound(it.getPivoCategory().version) }
        return speeds ?: intArrayOf(0)
    }

    override fun enableRemoteController(on: Boolean) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getRemoteSwitcherBytes(on))
    }

    override fun startPairing() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getStartPairingBytes())
    }

    override fun cancelPairing() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getCancelPairingBytes())
    }

    override fun enableMovementNotifier(enabled: Boolean) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getNotifierSwitcherBytes(enabled))
    }

    override fun makeBeamSound(sound: SoundType) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        bluetoothController?.write(commandBuilder!!.getSoundBytes(sound))
    }

    private var otaPacketSize: Int = 192
    override fun initiateOTA(data: ByteArray, otaPacketSize: Int) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        if (data.isNotEmpty()) {
            this.otaPacketSize = otaPacketSize
            otaBytes = data
            bluetoothController?.write(commandBuilder!!.getOTAInitiationBytes())
        }
    }


    override fun holdTurningHorizontally() {
        commandBuilder?.let { cmdBuilder ->
            deviceInfo?.let { deviceInfo ->
                if (deviceInfo.getPivoCategory() is PivoDeviceCategory.PivoPod) {
                    if (deviceInfo.getPivoCategory().version == 0) {
                        cmdBuilder.getHoldTurningHorizontallyBytes(
                            deviceInfo.getPivoCategory().version,
                            HorizontalDirection.LEFT,
                            Speed(3600),
                        )?.let { bluetoothController?.write(it) }
                    } else {
                        cmdBuilder.getHoldTurningHorizontallyBytes(
                            deviceInfo.getPivoCategory().version,
                            HorizontalDirection.LEFT,
                            Speed(510),
                        )?.let { bluetoothController?.write(it) }
                    }
                } else {
                    cmdBuilder.getHoldTurningHorizontallyBytes(
                        deviceInfo.getPivoCategory().version,
                        HorizontalDirection.LEFT,
                        Speed(64800),
                    )?.let { bluetoothController?.write(it) }
                }
            }
        } ?: run {
            onDisconnected()
        }
    }

    override fun getHorizontalPosition() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getHorizontalPositionBytes()?.let { bluetoothController?.write(it) }
    }

    override fun moveToSpecificHorizontalPosition(speed: Speed, degree: Int) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getGoToHorizontalPositionBytes(speed, degree)?.let {
            bluetoothController?.write(it)
        }
    }

    override fun move(
        direction: MixedDirection,
        hSpeed: Speed,
        hDegree: Int,
        vSpeed: Speed,
        vDegree: Int
    ) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getMovingCombinationBytes(hSpeed, hDegree, vSpeed, vDegree, direction)
            ?.let {
                bluetoothController?.write(it)
            }
    }

    override fun resetPositionHorizontally() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getResetPositionBytes()?.let { bluetoothController?.write(it) }
    }

    override fun moveVertically(direction: VerticalDirection, speed: Speed, degree: Int) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getMoveVerticallyBytes(direction, speed, degree)?.let {
            bluetoothController?.write(it)
        }
    }

    override fun getVerticalPosition() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getVerticalPositionBytes()?.let { bluetoothController?.write(it) }
    }

    override fun moveToSpecificVerticalPosition(speed: Speed, degree: Int) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getGoToVerticalPositionBytes(speed, degree)?.let {
            if (commandBuilder == null) {
                onDisconnected()
                return
            }
            bluetoothController?.write(it)
        }
    }

    override fun setVerticalMotorSpeed(speed: Speed) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getSetVerticalSpeedBytes(speed)?.let {
            bluetoothController?.write(it)
        }
    }

    override fun setLED(led: CurrentLED) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getSetLEDColorBytes(led.colors, led.brightnessLevel)?.let {
            bluetoothController?.write(it)
        }
    }

    override fun getLEDColor() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        commandBuilder!!.getRetrieveLEDColorBytes()?.let { bluetoothController?.write(it) }
    }

    override fun runLEDPreset(patternType: LightPatternType, level: Int) {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }
        if (level !in 0..11) {
            return
        }
        commandBuilder?.getRunLightPresetBytes(patternType, level)?.let {
            bluetoothController?.write(it)
        }
    }

    override fun stopLEDPreset() {
        if (commandBuilder == null) {
            onDisconnected()
            return
        }

        commandBuilder?.getStopLightPresetBytes()?.let {
            bluetoothController?.write(it)
        }
    }

    override fun registerConnectivityListener(listener: OnConnectionControllerListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerOTAUpdateListener(listener: OnOTAStatusListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerMovementListener(listener: OnMovementControlListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerRemoteControllerListener(listener: OnRemoteControllerListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerDeviceInfoListener(listener: OnDeviceInfoListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerBatteryListener(listener: OnBatteryListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerOnPodListener(listener: OnPodListener) {
        listenersMap[listener] = listener
    }

    override fun registerBeamSoundListener(listener: OnBeamSoundListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerOnLEDListener(listener: OnLEDListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerDeviceStatusListener(listener: OnDeviceStatusListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun registerAuthenticationListener(listener: OnAuthenticationListener) {
        listenersMap[listener] = RegisteredCallbackAdapter(listener)
    }

    override fun unregisterAuthenticationListener(listener: OnAuthenticationListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterConnectivityListener(listener: OnConnectionControllerListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterRemoteControllerListener(listener: OnRemoteControllerListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterMovementListener(listener: OnMovementControlListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterOTAUpdateListener(listener: OnOTAStatusListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterDeviceInfoListener(listener: OnDeviceInfoListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterOnPodListener(listener: OnPodListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterBeamSoundListener(listener: OnBeamSoundListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterOnLEDListener(listener: OnLEDListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterDeviceStatusListener(listener: OnDeviceStatusListener) {
        listenersMap.remove(listener)
    }

    override fun unregisterBatteryListener(listener: OnBatteryListener) {
        listenersMap.remove(listener)
    }

    companion object {
        const val TAG = "PivoDeviceManager"
    }
}